#pragma once

/***
    Based on avahi example of core-publish-service.c
***/

#pragma once

#include "glagol_cluster.h"
#include "glagol_ws_server.h"
#include "mdns_holder.h"
#include "net_devices_monitor.h"
#include "responder_status.h"
#include "backend_device_state_updater.h"

#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/interfaces/multiroom/i_multiroom_provider.h>
#include <yandex_io/interfaces/stereo_pair/i_stereo_pair_provider.h>
#include <yandex_io/libs/base/quasar_service.h>
#include <yandex_io/libs/configuration/features_config.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/glagol_sdk/connector.h>
#include <yandex_io/libs/glagol_sdk/defs.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/mdns/nsd_messager.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/libs/websocket/websocket_server.h>

#include <yandex_io/protos/quasar_proto.pb.h>

#include <yandex_io/sdk/interfaces/i_capability.h>
#include <yandex_io/sdk/interfaces/i_endpoint.h>
#include <yandex_io/sdk/private/device_context.h>
#include <yandex_io/sdk/sdk_interface.h>

#include <json/json.h>

#include <array>
#include <map>
#include <memory>
#include <mutex>
#include <string>
#include <thread>
#include <vector>

namespace quasar {

    class INsdMessager;

    class Glagol
        : public QuasarService,
          public YandexIO::IEndpoint::IListener,
          public YandexIO::ICapability::IListener,
          public std::enable_shared_from_this<Glagol> {
    public:
        static const std::string GUEST_FORBIDDEN_ERROR;
        struct Settings {
            bool heartbeatTelemetry{true};
            bool nsdInsteadOfAvahi{false};
            bool skipConnectionResolves{false};
            bool extraLogging{false};
            std::chrono::seconds networkInfoSendInterval{0};
            glagol::AvahiSettings avahi;
            struct GlagolCluster::Settings cluster;
        };

        static const std::string SERVICE_NAME;
        Glagol(std::shared_ptr<YandexIO::IDevice> device,
               std::shared_ptr<ipc::IIpcFactory> ipcFactory,
               std::shared_ptr<IAuthProvider> authProvider,
               std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
               std::shared_ptr<IMultiroomProvider> multiroomProvider,
               std::shared_ptr<IStereoPairProvider> stereoPairProvider,
               MDNSHolderFactory mdnsFactory,
               const Json::Value& config,
               const Settings& settings,
               std::shared_ptr<YandexIO::SDKInterface> sdk);
        ~Glagol();

        std::string getServiceName() const override;

        void start() override;

        static bool buildControlRequest(const Json::Value& payload, quasar::proto::QuasarMessage& messageToBuild, const std::string& vinsRequestId);
        static Settings fromSystemConfig(const Json::Value& /*systemConfig*/);
        static bool allowedGuestCommand(const std::string& command);

        /**
         * @brief Wait until WS Server starts
         */
        void waitWsServerStart() const;

        std::shared_ptr<ResponderStatus> getResponderStatusProvider();

        void onDiscovery(const glagol::IDiscovery::Result& /*newResult*/);
        proto::QuasarMessage makeGlagolDiscoveryResultMessage() const;

        /* methods for tests */
        void forceHeartbeatTelemetry();
        void setOnReconfigure(std::function<void()> /*callback*/);

    private:
        class AliceRequestEventsHandler: public YandexIO::IAliceRequestEvents {
        public:
            AliceRequestEventsHandler();
            quasar::ipc::SharedMessage getResponse();

        private:
            void onAliceRequestStarted(std::shared_ptr<YandexIO::VinsRequest> request) override;
            void onAliceRequestCompleted(std::shared_ptr<YandexIO::VinsRequest> request, const Json::Value& response) override;
            void onAliceRequestError(std::shared_ptr<YandexIO::VinsRequest> request, const std::string& errorCode, const std::string& errorText) override;

        private:
            std::promise<quasar::ipc::SharedMessage> promise_;
            std::future<quasar::ipc::SharedMessage> future_;
        };

    private:
        using Status = std::string;
        using MsgConnInfo = GlagolWsServer::MsgConnInfo;

        static const Status UNSUPPORTED;
        static const Status FAILURE;
        static const Status SUCCESS;

        static const std::string IDLE;
        static const std::string LISTENING;
        static const std::string SPEAKING;
        static const std::string BUSY;
        static const std::string SHAZAM;
        static const std::string NONE;

        // forbidden copy-constructor and =-operator
        Glagol(const Glagol&) = delete;
        Glagol& operator=(const Glagol&) = delete;

        Lifetime lifetime_;
        mutable std::mutex stateMutex_;
        mutable std::mutex quasarStateMutex_; // mutex for quasarish protobuf fields. Cant copy AND write them around: see ALICE-2421
        mutable std::mutex glagoldStateMutex_;
        std::shared_ptr<YandexIO::IDevice> device_;

        FeaturesConfig featuresConfig_;

        std::shared_ptr<ipc::IServer> quasarServer_;
        std::shared_ptr<ipc::IConnector> aliceConnector_;
        std::shared_ptr<ipc::IConnector> syncdConnector_;
        std::shared_ptr<ipc::IConnector> videodConnector_;
        std::shared_ptr<ipc::IConnector> networkdConnector_;

        std::shared_ptr<IMultiroomProvider> multiroomProvider_;
        std::shared_ptr<IStereoPairProvider> stereoPairProvider_;

        YandexIO::DeviceContext deviceContext_;
        proto::AliceState_State lastState_ = proto::AliceState_State_IDLE;

        std::function<void()> onReconfigure_;
        bool nsdInsteadOfAvahi_ = false;
        glagol::AvahiSettings avahiSettings_;
        std::atomic_bool extraLogging_ = false;
        std::atomic_bool heartbeatTelemetry_ = false;
        std::atomic_bool skipConnectionResolves_ = false;
        Json::Value getState() const;
        Json::Value getResponseMessageCore(proto::GlagoldState::ConnectionType /*connectionType*/) const;
        bool stateVolumeChanged(int volume, int maxVolume);
        void notifyHeartbeat();
        Json::Value state_;

        quasar::proto::AppState appState_;
        quasar::proto::WatchedVideoState watchedVideoState_;
        quasar::proto::EnvironmentMessage environmentState_;
        quasar::proto::GlagoldState glagoldState_;
        glagol::IDiscovery::Result discoveryResult_;
        std::optional<std::string> wifiSsid_;
        std::optional<std::string> wifiMac_;
        std::optional<std::string> localAddress_;
        std::vector<std::string> IPsReportedToBackend_;
        std::unique_ptr<glagol::BackendDeviceStateUpdater> backendDeviceStateUpdater_;

        GlagolWsServer wsServer_;

        struct InterfaceState {
            enum class State {
                DOWN,
                UP_NO_ADDR,
                UP_WITH_ADDR
            } state;
            std::chrono::steady_clock::time_point updated;
        };
        std::unordered_map<std::string, InterfaceState> netDevStates_;
        glagol::NetDevicesMonitorPtr netDevicesMonitor_;

        struct MultiRoomDetails {
            std::mutex mutex;
            quasar::MultiroomState::Mode mode{quasar::MultiroomState::Mode::UNDEFINED};
            quasar::MultiroomState::SyncLevel syncLevel{quasar::MultiroomState::SyncLevel::UNDEFINED};
            std::optional<std::string> masterId;
        } multiRoom_;
        std::shared_ptr<const StereoPairState> stereoPair_;
        bool guestMode_{false};

        // last moment of aliceState change
        std::chrono::steady_clock::time_point lastAliceActivity_;

        GlagolCluster glagolCluster_;
        std::mutex mdnsHolderMutex_;
        MDNSHolderFactory mdnsFactory_;
        MDNSHolderPtr mdnsHolder_;

        const std::chrono::steady_clock::time_point startTime_;
        const std::shared_ptr<YandexIO::SDKInterface> sdk_;

        std::string groupSpotterMicrophoneState_ = glagol::NOT_LISTENS;
        std::unique_ptr<PeriodicExecutor> heartbeatNotifier_;

        void reinitMdns(const Settings& /*settings*/);
        void reinitMdns();
        void changeGuestMode(bool guestMode);
        void changeStereopairRole(StereoPairState::Role role);
        void changeTandemRole(glagol::GroupRole /*role*/);
        void applySystemConfig(const Json::Value& /*systemConfig*/);
        void sendHeartbeatTelemetry();
        void onClusterDevicelistChanged(const GlagolCluster::DeviceList& /*deviceList*/);

        using SetraceArgs = std::unordered_map<std::string, std::string>;
        void logSetraceEvent(const std::string& nameSuffix, const std::string& requestId, SetraceArgs args);

        // websocket callbacks
        void onGlagolWSConnectionsChanged(proto::GlagoldState state);
        void onGlagolWSMessage(WebsocketServer::ConnectionHdl hdl, MsgConnInfo connInfo, const Json::Value& request);

        void onAlicedMessage(const quasar::ipc::SharedMessage& message);

        static std::string statusToString(const quasar::proto::VinsCallResponse& vinsCallResponse);
        static std::string mapState(quasar::proto::AliceState_State state);

        void setConnectionDetails(WebsocketServer::ConnectionHdl /*hdl*/, quasar::proto::GlagoldState::ConnectionType /*connType*/, const std::string& deviceId);
        void reRequestAccountDevices();

        /**
         * @brief Send metrica EnvironmentVariables and metrica Events using metricaConnector
         *      "glagold_connections_count" - environment variable where value is current connections
         *       count (got from newState)
         *       Send *_CONNECTED, *_DISCONNECTED metrica events for each GlagoldState::ConnectionType
         * @param[in] oldState - container with old connections
         * @param[in] newState - container with new connections
         */
        void sendConnectionsMetrica(const quasar::proto::GlagoldState& oldState,
                                    const quasar::proto::GlagoldState& newState) const;

        using ConnectionsTypeCounters = std::array<unsigned, quasar::proto::GlagoldState::ConnectionType_ARRAYSIZE>;
        /**
         * @brief Create ConnectionsTypeMultiSet with ConnectionType values from input state
         * @param[in] state - list of current connections to build multiset from
         * @return - built multiset
         */
        static ConnectionsTypeCounters buildConnectionsCounters(const quasar::proto::GlagoldState& state);

        /**
         * @brief Report metrica Event if amount of connections of "type" is different
         *        between oldConnectionsSet and newConnectionsSet
         *        NOTE: Report event for EACH connected/disconnected device,
         *              so if (oldConnectionsSet.count(type) == 5 and newConnectionsSet.count(type) == 7)
         *              TWO *_CONNECTED events will be sent
         * @param[in] type - Connection Type to report metrica off
         * @param[in] oldConnectionsSet - multiset with old connections types
         * @param[in] newConnectionsSet - multiset with old connections types
         * @param[in] metricaEventNamePrefix - Prefix, which used for Metrica Event.
         *            i.e.: type = GLAGOL_APP, prefix = "GLAGOL_APP",
         *                  oldConnectionsSet.count(type) > newConnectionsSet.count(type)
         *                  std::string(prefix + "_DISCONNECTED") event will be sent
         *                  oldConnectionsSet.count(type) < newConnectionsSet.count(type)
         *                  std::string(prefix + "_CONNECTED") event will be sent
         */
        void reportConnectionsChange(quasar::proto::GlagoldState::ConnectionType type,
                                     const ConnectionsTypeCounters& oldConnectionsSet, const ConnectionsTypeCounters& newConnectionsSet,
                                     const std::string& metricaEventNamePrefix) const;

        static quasar::proto::GroupSpotterMicrophoneState toGroupSpotterMicrophoneState(const std::string& string) {
            if (string == glagol::NOT_LISTENS) {
                return quasar::proto::GroupSpotterMicrophoneState::NOT_LISTENS;
            } else if (string == glagol::LISTENS) {
                return quasar::proto::GroupSpotterMicrophoneState::LISTENS;
            } else {
                throw std::runtime_error(std::string("No GroupSpotterMicrophoneState constant for string: ") + string);
            }
        }

        static const std::map<quasar::proto::AliceState_State, std::string> stateToName;
        static const std::map<std::string, quasar::proto::AliceState_State> nameToState;
        static quasar::proto::AliceState_State getStateByName(const std::string& stateName);

        static bool buildNavigationScrollAmount(const Json::Value& payload,
                                                quasar::proto::NavigationScrollAmount& messageToBuild);

        void updateLocalAddress(std::string bestAddr);
        glagol::IBackendApi::NetworkInfo makeNetworkInfo(std::vector<std::string> macs) const;
        void netDeviceUpdated(const glagol::DeviceId& /*myDeviceId*/, const glagol::NetDevicesMonitor::NetDevice& /*netDev*/);

        // IEndpoint::IListener
        void onCapabilityAdded(const std::shared_ptr<YandexIO::IEndpoint>& enpdoint, const std::shared_ptr<YandexIO::ICapability>& capability) override;
        void onCapabilityRemoved(const std::shared_ptr<YandexIO::IEndpoint>& enpdoint, const std::shared_ptr<YandexIO::ICapability>& capability) override;
        void onEndpointStateChanged(const std::shared_ptr<YandexIO::IEndpoint>& endpoint) override;

        // ICapability::IListener
        void onCapabilityStateChanged(const std::shared_ptr<YandexIO::ICapability>& capability, const NAlice::TCapabilityHolder& state) override;
        void onCapabilityEvents(const std::shared_ptr<YandexIO::ICapability>& capability, const std::vector<NAlice::TCapabilityEvent>& events) override;
    };
} // namespace quasar
