#include "glagol_cluster.h"
#include "bypass_directive_handler.h"

#include "google/protobuf/util/message_differencer.h"
#include "proto_utils.h"

#include <yandex_io/capabilities/device_state/converters/converters.h>
#include <yandex_io/capabilities/device_state/device_state_capability.h>
#include <yandex_io/interfaces/auth/connector/auth_provider.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/configuration/configuration.h>
#include <yandex_io/libs/glagol_sdk/backend_api.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <yandex_io/sdk/interfaces/i_capability.h>
#include <yandex_io/sdk/private/weak_listeners_storage.h>

#include <optional>
#include <set>
#include <string_view>
#include <utility>

YIO_DEFINE_LOG_MODULE("glagol");

using namespace quasar;

GlagolCluster::Settings::Settings()
    : peerCleanupPeriod(15)
    , peerKpaTimeout{60}
    , minimumPeersExchangeInterval{60}
    , resolvesOutdatingInterval{15 * 60}
    , peersExchange{true}
{
}

namespace {
    // constexpr std::chrono::seconds PEER_RESCAN_PERIOD{ 60 };
    constexpr std::chrono::seconds CONNECTOR_AWAITING_PERIOD{2};

    const std::string COMMAND_CLUSTER_PING = "clusterPing";
    const std::string COMMAND_CLUSTER_HELO = "clusterHelo";
    const std::string COMMAND_CLUSTER_MESSAGE = "clusterMessage";
    const std::string COMMAND_CLUSTER_PEERS_SHARE = "clusterPeersShare";
    const std::set<std::string_view> CLUSTER_COMMANDS = {
        COMMAND_CLUSTER_PING,
        COMMAND_CLUSTER_HELO,
        COMMAND_CLUSTER_MESSAGE,
        COMMAND_CLUSTER_PEERS_SHARE,
    };

    Json::Value makeCommandClusterPing()
    {
        Json::Value json;
        json["command"] = COMMAND_CLUSTER_PING;
        return json;
    }

    Json::Value makeClusterCommand(std::string myDeviceId, const std::string& command) {
        Json::Value json;
        json["command"] = command;
        json["from_device_id"] = myDeviceId;
        return json;
    }

    Json::Value makeCommandClusterHelo(std::string myDeviceId)
    {
        return makeClusterCommand(std::move(myDeviceId), COMMAND_CLUSTER_HELO);
    }

    Json::Value makeCommandClusterMessage(std::string myDeviceId)
    {
        return makeClusterCommand(std::move(myDeviceId), COMMAND_CLUSTER_MESSAGE);
    }

    Json::Value makeCommandClusterPeersShare(std::string myDeviceId)
    {
        return makeClusterCommand(std::move(myDeviceId), COMMAND_CLUSTER_PEERS_SHARE);
    }

    void fillDeviceStateFromMessage(const std::shared_ptr<YandexIO::IDeviceStateCapability>& capability, const glagol::model::IncomingMessage& msg) {
        const auto& extra = msg.extra;
        if (const auto& appStateData = extra["appState"]; appStateData.isString()) {
            quasar::proto::AppState appState;
            appState.ParseFromStringOrThrow(base64Decode(appStateData.asString()));
            YandexIO::fillFromAppState(capability, appState);
        }
        const auto prevState = capability->getState().GetDeviceStateCapability().GetState();
        if (const auto& watchedVideoData = extra["watchedVideoState"]; watchedVideoData.isString()) {
            quasar::proto::WatchedVideoState state;
            state.ParseFromStringOrThrow(base64Decode(watchedVideoData.asString()));
            if (auto converted = YandexIO::convertLastWatched(state); !google::protobuf::util::MessageDifferencer::Equals(converted, prevState.GetDeviceState().GetLastWatched())) {
                capability->setLastWatched(YandexIO::convertLastWatched(state));
            }
        }
        if (const auto& environmentState = extra["environmentState"]; environmentState.isString()) {
            quasar::proto::EnvironmentMessage environmentMessage;
            environmentMessage.ParseFromStringOrThrow(base64Decode(environmentState.asString()));
            if (!google::protobuf::util::MessageDifferencer::Equals(environmentMessage.device_info(), prevState.GetEnvironmentDeviceInfo())) {
                capability->setEnvironmentDeviceInfo(environmentMessage.device_info());
            }
        }
        if (const auto features = YandexIO::convertFeatures(msg.supportedFeatures); features != std::unordered_set<TString>{prevState.GetSupportedFeatures().begin(), prevState.GetSupportedFeatures().end()}) {
            capability->setSupportedFeatures(features);
        }

        if (const auto exps = YandexIO::convertExperiments(Json::nullValue, msg.experiments); !google::protobuf::util::MessageDifferencer::Equals(exps, prevState.GetExperiments())) {
            capability->setExperiments(exps);
        }
    }
    /******************************************************************/

    class PersistentStorage: public glagol::IDiscoveredItems::Storage {
        const std::string filename_;

    public:
        PersistentStorage(const std::string& name)
            : filename_(name)
                  {};

        void save(const std::string& data) override {
            try {
                quasar::TransactionFile file(filename_);
                file.write(data);
                file.commit();
            } catch (const std::exception& ex) {
                YIO_LOG_ERROR_EVENT("GlagolCluster.save", ex.what());
            }
        }

        std::string getContent() override {
            return getFileContent(filename_);
        }
    };

    glagol::IDiscoveredItems::StoragePtr makeItemsStorage(const std::string& filename) {
        if (filename.empty()) {
            return nullptr;
        }
        return std::make_shared<PersistentStorage>(filename);
    }

    /******************************************************************/

    class GlagolBypassCapability: public YandexIO::WeakListenersStorage<YandexIO::ICapability, YandexIO::ICapability::IListener> {
        const std::string id_{"pilotBypass"};
        mutable std::mutex mutex_;
        NAlice::TCapabilityHolder state_;
        std::shared_ptr<YandexIO::IDirectiveHandler> directiveHandler_;

    public:
        GlagolBypassCapability(std::shared_ptr<YandexIO::IDirectiveHandler> directiveHandler)
            : directiveHandler_(std::move(directiveHandler))
        {
        }

        const std::string& getId() const override {
            return id_;
        }

        NAlice::TCapabilityHolder getState() const override {
            std::scoped_lock<std::mutex> lock(mutex_);
            return state_;
        }

        YandexIO::IDirectiveHandlerPtr getDirectiveHandler() override {
            return directiveHandler_;
        }
    };

} // namespace

ResolveHandler* GlagolCluster::getResolveHandler() {
    return discoveredItems_.get();
}

void GlagolCluster::addResolve(glagol::DeviceId deviceId, glagol::ResolveInfo resolve, glagol::ResolveSource source) {
    lifecycle_->add([this, deviceId = std::move(deviceId), resolve = std::move(resolve), source]() {
        discoveredItems_->addResolve(deviceId, resolve, source);
    });
};

void GlagolCluster::setOnDiscoveryCallback(OnDiscoveryCallback newCallback) {
    onDiscovery_ = std::move(newCallback);
}

GlagolCluster::GlagolCluster(std::shared_ptr<YandexIO::IDevice> device,
                             std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                             std::shared_ptr<IAuthProvider> authProvider,
                             OnDiscoveryCallback onDiscovery,
                             std::function<void()> deviceListRequest,
                             std::shared_ptr<YandexIO::IEndpointStorage> endpointStorage,
                             const Settings& settings)
    : settings_(settings)
    , device_(std::move(device))
    , ipcFactory_(std::move(ipcFactory))
    , myDeviceId_(device_->deviceId())
    , backendUrl_(device_->configuration()->getServiceConfig("common")["backendUrl"].asString())
    , accountDevices_(std::make_shared<glagol::IndirectAccountDevices>())
    , endpointStorage_(std::move(endpointStorage))
    , onDiscovery_(std::move(onDiscovery))
    , neighborsUpdated_(std::chrono::steady_clock::now())
    , authProvider_(std::move(authProvider))
    , backendSettings_{.url = backendUrl_}
    , backendApi_(glagol::createCachedBackendApi(
          device_,
          backendSettings_,
          authProvider_,
          std::move(deviceListRequest),
          accountDevices_->devices()))
    , discoveredItems_(createDiscoveredItems(
          glagol::DeviceId{.id = device_->deviceId(), .platform = device_->configuration()->getDeviceType()},
          device_->telemetry(),
          backendApi_,
          makeItemsStorage(settings_.saveFile),
          [this]() {
              return std::make_shared<glagol::ext::Connector>(backendApi_, device_->telemetry(), "cluster");
          }))
    , deviceList_(std::make_shared<std::vector<std::string>>())
    , lifecycle_(std::make_shared<NamedCallbackQueue>("GlagolCluster"))
{
    YIO_LOG_INFO("Create GlagolCluster");

    authProvider_->ownerAuthInfo().connect([this](std::shared_ptr<const AuthInfo2> authInfo) {
        handleAuthInfo(std::move(authInfo));
    }, lifetime_, lifecycle_);

    lifecycle_->addDelayed([this] { cleanUpPeers(); }, settings_.peerCleanupPeriod, lifetime_);
    discoveredItems_->setOnNeighborsUpdated(
        [this]() {
            neighborsUpdated();
        });

    discoveredItems_->setOnDiscoveryCallback(
        [this](const glagol::DeviceId& id, ConnectorPtr connector) {
            YIO_LOG_INFO("Queueing new connector " << connector.get() << " to " << id.id);
            lifecycle_->add([this, id = id, connector = std::move(connector)]() mutable {
                processDiscoveryResult(std::move(id), std::move(connector));
            });
        });
}

GlagolCluster::~GlagolCluster()
{
    lifetime_.die();
}

void GlagolCluster::neighborsUpdated() {
    lifecycle_->add([this]() {
        YIO_LOG_DEBUG("discovered neighbors updated");
        neighborsUpdated_ = std::chrono::steady_clock::now();
        if (onDiscovery_) {
            onDiscovery_(discoveredItems_->getDiscoveries());
        }
    });
}

std::optional<glagol::DeviceId> GlagolCluster::isKnownAccountDevice(const std::string& id) {
    return (backendApi_->hasDeviceInDeviceList(id));
}

void GlagolCluster::lifecycleExecute(std::function<void()> fn) {
    lifecycle_->add(std::move(fn));
}

void GlagolCluster::updateSettings(const Settings& newSettings) {
    lifecycle_->add([this, newSettings]() {
        settings_ = newSettings;
        glagol::IDiscoveredItems::Settings discoveredItemsSettings;
        discoveredItemsSettings.disableBackendDiscovery = newSettings.disableBackendDiscovery;
        discoveredItems_->setSettings(discoveredItemsSettings);
    });
}

void GlagolCluster::updateAccountDevices(glagol::IBackendApi::DevicesMap newDevices) {
    lifecycle_->add([this, newDevices = std::move(newDevices)]() {
        backendApi_->setDevicesList(newDevices);
        discoveredItems_->updateAccountDevices(newDevices);
        accountDevices_->update(newDevices);
    });
}

void GlagolCluster::completeDevicesTelemetry(Json::Value partialResult) {
    lifecycle_->add(
        [this, result = std::move(partialResult)]() mutable {
            const auto discovered = discoveredItems_->getDevicesAround();
            for (auto& [deviceId, item] : discovered) {
                if (!item.isAccountDevice || deviceId.id == myDeviceId_) {
                    continue;
                }
                result[deviceId.id]["visible_host"] = item.address;
            }
            auto now = std::chrono::steady_clock::now();
            for (auto& [deviceId, peer] : peers_) {
                auto& dst = result[deviceId.id];
                dst["cluster_state"] = std::to_string(peer.state);
                dst["cluster_timeout_ms"] = (int64_t)(std::chrono::duration_cast<std::chrono::milliseconds>(now - peer.incomingMessageTime).count());
            };
            YIO_LOG_DEBUG(jsonToString(result, true));
            device_->telemetry()->reportEvent("glagold_heartbeat", jsonToString(result));
        }, lifetime_);
}

void GlagolCluster::processClusterMessage(proto::GlagolClusterMessage glagolClusterMessage)
{
    lifecycle_->add(
        [this, glagolClusterMessage{std::move(glagolClusterMessage)}]() mutable {
            const auto& deviceIds = glagolClusterMessage.target_device_ids();
            if (glagolClusterMessage.has_target_device_all() ||
                (std::find(deviceIds.begin(), deviceIds.end(), myDeviceId_) != deviceIds.end())) {
                invokeClusterMessage("<internal>", glagolClusterMessage.service_name(), glagolClusterMessage.quasar_message_base64(), myDeviceId_);
            }
            std::set<std::string_view> processedDeviceIds;
            std::optional<Json::Value> optPayload;
            for (auto& [targetDeviceId, peer] : peers_) {
                if (glagolClusterMessage.has_target_device_all() ||
                    glagolClusterMessage.has_target_device_remote() ||
                    (std::find(deviceIds.begin(), deviceIds.end(), targetDeviceId.id) != deviceIds.end())) {
                    if (!optPayload) {
                        optPayload = makeCommandClusterMessage(myDeviceId_);
                        auto& payload = *optPayload;
                        payload["service_name"] = glagolClusterMessage.service_name();
                        payload["quasar_message_base64"] = glagolClusterMessage.quasar_message_base64();
                    }
                    processedDeviceIds.insert(targetDeviceId.id);
                    sendMessage(targetDeviceId, peer, *optPayload);
                }
            }
            for (const auto& deviceId : deviceIds) {
                if (!processedDeviceIds.count(deviceId)) {
                    YIO_LOG_WARN("GlagolCluster: Fail send message to " << deviceId << ", peer not found");
                }
            }
        }, lifetime_);
}

void GlagolCluster::processShares(std::string peerId, std::string msg) {
    lifecycle_->add(
        [this, quasarMessageBase64{std::move(msg)}, peerId{std::move(peerId)}]() mutable {
            quasar::proto::NsdInfoList neighbors;
            Y_PROTOBUF_SUPPRESS_NODISCARD neighbors.ParseFromString(TString(base64Decode(quasarMessageBase64)));
            quasarMessageBase64.clear();

            YIO_LOG_INFO("Process " << neighbors.items().size() << " shared discoveries reported by " << peerId);

            for (auto& nsdItem : neighbors.items()) {
                auto [item, info] = glagol::convertResolveItemFull(nsdItem);
                info.source = glagol::ResolveItem::SourceFlag::PEERS_EXCHANGE;
                discoveredItems_->newResolve(item, info);
            };
        }, lifetime_);
}

std::optional<bool> GlagolCluster::processExternalCommand(Json::Value& /*response*/, const Json::Value& request)
{
    const Json::Value& payload = request["payload"];
    std::string command = payload["command"].asString();
    try {
        if (command == COMMAND_CLUSTER_PING) {
            // Do nothing
        } else if (command == COMMAND_CLUSTER_HELO) {
            YIO_LOG_INFO("Receive HELO from deviceId=" << getString(payload, "from_device_id"));
        } else if (command == COMMAND_CLUSTER_MESSAGE) {
            auto id = getString(request, "id");
            auto serviceName = getString(payload, "service_name");
            auto quasarMessageBase64 = getString(payload, "quasar_message_base64");
            auto fromDeviceId = getString(payload, "from_device_id");
            lifecycle_->add(
                [this, id{std::move(id)}, serviceName{std::move(serviceName)}, quasarMessageBase64{std::move(quasarMessageBase64)}, fromDeviceId{std::move(fromDeviceId)}]() mutable {
                    invokeClusterMessage(id, serviceName, quasarMessageBase64, std::move(fromDeviceId));
                }, lifetime_);
        } else if (command == COMMAND_CLUSTER_PEERS_SHARE && settings_.peersExchange) {
            auto id = getString(payload, "from_device_id");
            processShares(id, getString(payload, "quasar_message_base64"));
        } else {
            YIO_LOG_DEBUG("Unknown command: " << jsonToString(request));
            return {};
        }
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("GlagolCluster.FailedProcessExternalCommand", "Fail to process glagol cluster command: " << ex.what());
        return false;
    } catch (...) {
        YIO_LOG_ERROR_EVENT("GlagolCluster.FailedProcessExternalCommand", "Fail to process glagol cluster command: unexpected exception");
        return false;
    }
    return true;
}

Json::Value GlagolCluster::makeClusterPeersShare() {
    Json::Value result = makeCommandClusterPeersShare(myDeviceId_);
    quasar::proto::NsdInfoList msg;
    discoveredItems_->eachResolvedItem([&msg](const glagol::ResolveItem& item, const glagol::ResolveItem::Info& details) {
        auto nsdItem = msg.add_items();
        nsdItem->set_type(TString(item.type));
        nsdItem->set_name(TString(item.name));
        nsdItem->set_address(TString(details.address));
        nsdItem->set_port(details.port);
        for (const auto& [name, value] : details.txt) {
            auto attr = nsdItem->add_attrs();
            attr->set_name(TString(name));
            attr->set_value(TString(value));
        }
    });
    TString tmpStr;
    Y_PROTOBUF_SUPPRESS_NODISCARD msg.SerializeToString(&tmpStr);
    result["quasar_message_base64"] = base64Encode(tmpStr.data(), tmpStr.size());
    return result;
}

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
bool GlagolCluster::isExternalCommand(const std::string& command)
{
    return CLUSTER_COMMANDS.count(command) > 0;
}

ILiveData<GlagolCluster::DeviceList>& GlagolCluster::deviceList()
{
    return deviceList_;
}

void GlagolCluster::processDiscoveryResult(glagol::DeviceId deviceId, GlagolCluster::ConnectorPtr connector)
{
    Y_ENSURE_THREAD(lifecycle_);
    YIO_LOG_INFO("New discovered connection to " << deviceId.id);

    auto tracker = lifetime_.tracker();

    auto [peerIter, inserted] = peers_.emplace(deviceId, lastUniquePeerId);
    if (inserted) {
        ++lastUniquePeerId;
        Peer& peer = peerIter->second;
        peer.incomingMessageTime = std::chrono::steady_clock::now();
        peer.outcomingMessageTime = std::chrono::steady_clock::now();
        peer.connector = connector;
        peer.connector->setOnPongCallback(makeSafeCallback(
            [this, uniquePeerId = peer.uniquePeerId, deviceId = deviceId]() {
                onPong(uniquePeerId, deviceId);
            }, lifetime_, lifecycle_));
        peer.connector->setOnMessageCallback(makeSafeCallback(
            [this, uniquePeerId = peer.uniquePeerId, deviceId = deviceId](const auto& message) {
                onIncomingMessage(uniquePeerId, deviceId, std::move(message));
            }, lifetime_, lifecycle_));
        peer.connector->setOnStateChangedCallback(makeSafeCallback(
            [this, uniquePeerId = peer.uniquePeerId, deviceId = deviceId](auto state) {
                onConnectorStateChanged(uniquePeerId, deviceId, state);
            }, lifetime_, lifecycle_));
        if (deviceId.platform == "yandexmodule_2") {
            NAlice::TEndpoint::TDeviceInfo deviceInfo;
            deviceInfo.SetModel(TString(deviceId.platform));
            auto directiveHandler = std::make_shared<glagol::BypassDirectiveHandler>(deviceId.id, myDeviceId_, connector, lifecycle_);
            peer.endpoint = endpointStorage_->createEndpoint(deviceId.id, NAlice::TEndpoint::DongleEndpointType, std::move(deviceInfo), std::move(directiveHandler));
            auto deviceStateCapability = std::make_shared<YandexIO::DeviceStateCapability>(lifecycle_);
            peer.endpoint->addCapability(deviceStateCapability);
            endpointStorage_->addEndpoint(peer.endpoint);
            Json::Value json;
            json["command"] = "ping";
            peer.connector->send(json);
        }
        YIO_LOG_INFO("Add deviceId=" << deviceId.id << " to glagol cluster");
        neighborsUpdated();
    } else {
        YIO_LOG_WARN("Already in peers!");
    }
}

// NOLINTNEXTLINE(readability-convert-member-functions-to-static)
void GlagolCluster::clientPeerDisconnected(const std::string& deviceId) {
    YIO_LOG_INFO("Client peer " << deviceId << " disconnected");
}

void GlagolCluster::cleanUpPeers()
{
    Y_ENSURE_THREAD(lifecycle_);

    discoveredItems_->periodicCheck();

    const auto now = std::chrono::steady_clock::now();

    std::lock_guard adLock(deviceList_); // Report about deviceList_ changes only at exit
    bool someDropped = false;
    for (auto it = peers_.begin(); it != peers_.end();) {
        auto& [deviceId, peer] = *it;
        if (now - peer.incomingMessageTime > settings_.peerKpaTimeout) {
            YIO_LOG_INFO("Remove deviceId=" << deviceId.id << " from glagol cluster");
            removeDeviceFromList(deviceId.id);
            if (it->second.endpoint) {
                endpointStorage_->removeEndpoint(it->second.endpoint);
            }
            it = peers_.erase(it);
            someDropped = true;
        } else {
            if (peer.state == glagol::ext::Connector::State::CONNECTED) {
                if (now - peer.outcomingMessageTime >= settings_.peerKpaTimeout / 4 &&
                    now - peer.incomingMessageTime >= settings_.peerKpaTimeout / 4) {
                    sendMessage(deviceId, peer, makeCommandClusterPing());
                }
                if (neighborsUpdated_ > peer.lastSharesExchangeTime &&
                    (neighborsUpdated_ - peer.lastSharesExchangeTime) > settings_.minimumPeersExchangeInterval) {
                    sendPeersExchage(deviceId, peer, now);
                }
            }
            ++it;
        }
    }

    if (someDropped) {
        neighborsUpdated();
    }

    lifecycle_->addDelayed([this] { cleanUpPeers(); }, settings_.peerCleanupPeriod, lifetime_);
}

void GlagolCluster::handleAuthInfo(std::shared_ptr<const AuthInfo2> authInfo)
{
    Y_ENSURE_THREAD(lifecycle_);
    auto oldSettings = std::exchange(backendSettings_, glagol::BackendApi::Settings{.url = backendUrl_, .token = authInfo->authToken});
    if (oldSettings != backendSettings_) {
        YIO_LOG_INFO("Backend settings was changed. Recreate cached backend.");
        backendApi_->setSettings(backendSettings_);
    };
}

void GlagolCluster::onPong(uint32_t uniquePeerId, const glagol::DeviceId& deviceId)
{
    Y_ENSURE_THREAD(lifecycle_);
    auto it = peers_.find(deviceId);
    if (it == peers_.end() || it->second.uniquePeerId != uniquePeerId) {
        return;
    }

    Peer& peer = it->second;
    peer.incomingMessageTime = std::chrono::steady_clock::now();
}

void GlagolCluster::onIncomingMessage(uint32_t uniquePeerId, const glagol::DeviceId& deviceId, const glagol::model::IncomingMessage& message)
{
    Y_ENSURE_THREAD(lifecycle_);
    auto it = peers_.find(deviceId);
    if (it == peers_.end() || it->second.uniquePeerId != uniquePeerId) {
        return;
    }

    Peer& peer = it->second;
    peer.incomingMessageTime = std::chrono::steady_clock::now();

    if (!message.extra.empty() && peer.endpoint) {
        try {
            if (const auto capability = peer.endpoint->findCapability<YandexIO::IDeviceStateCapability>()) {
                fillDeviceStateFromMessage(capability, message);
            }
        } catch (...) {
            YIO_LOG_WARN("Invalid incoming endpoint state");
        }
    }
}

void GlagolCluster::onConnectorStateChanged(uint32_t uniquePeerId, const glagol::DeviceId& deviceId, glagol::ext::Connector::State state)
{
    Y_ENSURE_THREAD(lifecycle_);
    auto it = peers_.find(deviceId);
    if (it == peers_.end() || it->second.uniquePeerId != uniquePeerId) {
        return;
    }
    it->second.state = state;
    if (state == glagol::ext::Connector::State::CONNECTED) {
        sendMessage(it->first, it->second, makeCommandClusterHelo(myDeviceId_));
        if (!discoveredItems_->noDevicesAround()) {
            sendPeersExchage(it->first, it->second, std::chrono::steady_clock::now());
        }
        addDeviceToList(deviceId.id);
    } else {
        removeDeviceFromList(deviceId.id);
    }
}

void GlagolCluster::sendPeersExchage(const glagol::DeviceId& deviceId, Peer& peer, std::chrono::steady_clock::time_point now) {
    if (settings_.peersExchange) {
        YIO_LOG_DEBUG("Send peers exchange to " << deviceId.id);
        sendMessage(deviceId, peer, makeClusterPeersShare());
        peer.lastSharesExchangeTime = now;
    }
};

void GlagolCluster::sendMessage(const glagol::DeviceId& deviceId, Peer& peer, Json::Value payload)
{
    if (deviceId.platform == "yandexmodule_2") {
        // Dirty hack for backward compatibility between modules and speakers with pilot via remoteEndpoints. Fake it's not cluster connection.
        return;
    }
    Y_ENSURE_THREAD(lifecycle_);
    peer.outcomingMessageTime = std::chrono::steady_clock::now();
    auto id = peer.connector->send(payload);
    if (YIO_LOG_DEBUG_ENABLED()) {
        payload.removeMember("quasar_message_base64");
        YIO_LOG_DEBUG("Send cluster message to deviceId=" << deviceId.id << ", id=" << id << ", payload=" << jsonToString(payload));
    }
}

void GlagolCluster::invokeClusterMessage(const std::string& id, const std::string& serviceName, const std::string& quasarMessageBase64, std::string fromDeviceId)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (!device_->configuration()->hasServiceConfig(serviceName)) {
        YIO_LOG_ERROR_EVENT("GlagolCluster.UnknownService", "Fail to invoke \"cluster message\" command: service \"" << serviceName << "\" is unknown");
        return;
    }

    try {
        auto binaryQuasarMessage = base64Decode(quasarMessageBase64);
        proto::QuasarMessage quasarMessage;
        Y_PROTOBUF_SUPPRESS_NODISCARD quasarMessage.ParseFromString(TString(binaryQuasarMessage));
        auto connector = connectors_[serviceName];
        if (!connector) {
            connector = ipcFactory_->createIpcConnector(serviceName);
            connector->connectToService();
            connector->waitUntilConnected(CONNECTOR_AWAITING_PERIOD);
            connectors_[serviceName] = connector;
        }

        quasarMessage.mutable_glagol_cluster_sign()->set_from_device_id(std::move(fromDeviceId));
        YIO_LOG_DEBUG("Invoke cluster message id=" << id << ", send QuasarMessage to \"" << serviceName << "\": " << convertMessageToJsonString(quasarMessage));
        connector->sendMessage(std::move(quasarMessage));
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("GlagolCluster.FailedInvokeClusterMessage", "Fail to invoke \"cluster message\" command: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("GlagolCluster.FailedInvokeClusterMessage", "Fail to invoke \"cluster message\" command: unexpected exception");
    }
}

void GlagolCluster::addDeviceToList(const std::string& deviceId)
{
    Y_ENSURE_THREAD(lifecycle_);

    auto deviceList = deviceList_.value();
    auto it = std::find(deviceList->begin(), deviceList->end(), deviceId);
    if (it == deviceList->end()) {
        auto newDeviceList = std::make_shared<std::vector<std::string>>();
        newDeviceList->reserve(deviceList->size() + 1);
        newDeviceList->assign(deviceList->begin(), deviceList->end());
        newDeviceList->emplace_back(deviceId);
        deviceList_ = std::move(newDeviceList);
    }
}

void GlagolCluster::removeDeviceFromList(const std::string& deviceId)
{
    Y_ENSURE_THREAD(lifecycle_);

    auto deviceList = deviceList_.value();
    auto it = std::find(deviceList->begin(), deviceList->end(), deviceId);
    if (it != deviceList->end()) {
        auto newDeviceList = std::make_shared<std::vector<std::string>>();
        newDeviceList->reserve(deviceList->size() - 1);
        newDeviceList->assign(deviceList->begin(), it);
        ++it;
        newDeviceList->assign(it, deviceList->end());
        deviceList_ = std::move(newDeviceList);
    }
}
