#pragma once

#include "discovered_items.h"
#include "resolve_handler.h"

#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/libs/glagol_sdk/backend_settings.h>
#include <yandex_io/libs/glagol_sdk/cached_backend_api.h>
#include <yandex_io/libs/glagol_sdk/connector2.h>
#include <yandex_io/libs/glagol_sdk/i_backend_api.h>
#include <yandex_io/libs/glagol_sdk/i_discovery.h>
#include <yandex_io/libs/glagol_sdk/indirect_account_devices.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/signals/live_data.h>
#include <yandex_io/libs/threading/unique_callback.h>
#include <yandex_io/protos/model_objects.pb.h>

#include <yandex_io/sdk/interfaces/i_endpoint_storage.h>

#include <json/json.h>

#include <chrono>
#include <map>

namespace quasar {

    class GlagolCluster {
        struct Peer;

    public:
        struct Settings {
            std::chrono::seconds peerCleanupPeriod;
            std::chrono::seconds peerKpaTimeout;
            std::chrono::seconds minimumPeersExchangeInterval;
            std::chrono::seconds resolvesOutdatingInterval;
            bool peersExchange = true;
            bool disableBackendDiscovery = false;
            std::string saveFile = "clusterSaves";
            Settings();
        };

        using ConnectorPtr = glagol::IDiscoveredItems::ConnectorPtr;
        using OnDiscoveryCallback = std::function<void(const glagol::IDiscovery::Result&)>;

        GlagolCluster(std::shared_ptr<YandexIO::IDevice> device,
                      std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                      std::shared_ptr<IAuthProvider> authProvider,
                      OnDiscoveryCallback onDiscovery,
                      std::function<void()> deviceListRequest,
                      std::shared_ptr<YandexIO::IEndpointStorage> endpointStorage,
                      const Settings& settings = Settings());
        ~GlagolCluster();

        void processClusterMessage(proto::GlagolClusterMessage glagolClusterMessage);

        /*! returns nullopt in case of non cluster command */
        std::optional<bool> processExternalCommand(Json::Value& response, const Json::Value& request);
        bool isExternalCommand(const std::string& command);
        void updateAccountDevices(glagol::IBackendApi::DevicesMap /*newDevices*/);
        void updateSettings(const Settings& /*newSettings*/);

        using DeviceList = std::shared_ptr<const std::vector<std::string>>;
        ILiveData<DeviceList>& deviceList();

        void setOnDiscoveryCallback(OnDiscoveryCallback /*newCallback*/);
        void completeDevicesTelemetry(Json::Value /*partialResult*/);
        ResolveHandler* getResolveHandler();

        std::optional<glagol::DeviceId> isKnownAccountDevice(const std::string& /*id*/);

        void clientPeerDisconnected(const std::string& deviceId); // come here from wsserver disconnects

        // test methods
        void lifecycleExecute(std::function<void()> /*fn*/);

        void addResolve(glagol::DeviceId deviceId, glagol::ResolveInfo /*resolve*/, glagol::ResolveSource /*source*/);

    private:
        void processDiscoveryResult(glagol::DeviceId /*deviceId*/, ConnectorPtr /*connector*/);
        void cleanUpPeers();
        void handleAuthInfo(std::shared_ptr<const AuthInfo2> authInfo);
        void onPong(uint32_t uniquePeerId, const glagol::DeviceId& deviceId);
        void onIncomingMessage(uint32_t uniquePeerId, const glagol::DeviceId& deviceId, const glagol::model::IncomingMessage& message);
        void onConnectorStateChanged(uint32_t uniquePeerId, const glagol::DeviceId& deviceId, glagol::ext::Connector::State state);

        void invokeClusterMessage(const std::string& id, const std::string& serviceName, const std::string& quasarMessageBase64, std::string fromDeviceId);
        void sendMessage(const glagol::DeviceId& /*deviceId*/, Peer& peer, Json::Value payload);
        void addDeviceToList(const std::string& deviceId);
        void removeDeviceFromList(const std::string& deviceId);

        Json::Value makeClusterPeersShare();
        void sendPeersExchage(const glagol::DeviceId& deviceId, Peer& peer, std::chrono::steady_clock::time_point now);

        glagol::DiscoveredItemPtr makeDiscoveredItem(glagol::DeviceId, const glagol::IDiscovery::Result::Item&);
        void processShares(std::string peerId, std::string msg);
        void neighborsUpdated();

    private:
        Lifetime lifetime_;
        Settings settings_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
        const std::string myDeviceId_;
        const std::string backendUrl_;
        const std::shared_ptr<glagol::IndirectAccountDevices> accountDevices_;
        const std::shared_ptr<YandexIO::IEndpointStorage> endpointStorage_;
        OnDiscoveryCallback onDiscovery_;
        std::chrono::steady_clock::time_point neighborsUpdated_;

        std::shared_ptr<IAuthProvider> authProvider_;

        glagol::BackendSettings backendSettings_;

        std::shared_ptr<glagol::BackendApi2> backendApi_;

        class DiscoveredItems;
        std::shared_ptr<glagol::IDiscoveredItems> discoveredItems_;

        uint32_t lastUniquePeerId{1};
        struct Peer {
            uint32_t uniquePeerId{0};
            std::chrono::steady_clock::time_point incomingMessageTime;
            std::chrono::steady_clock::time_point outcomingMessageTime;
            std::chrono::steady_clock::time_point lastSharesExchangeTime;
            std::shared_ptr<YandexIO::IEndpoint> endpoint;

            glagol::ext::Connector::State state{glagol::ext::Connector::State::STOPPED};
            ConnectorPtr connector;
            Peer(uint32_t id)
                : uniquePeerId(id)
                      {};
        };
        using PeersMap = std::map<glagol::DeviceId, Peer>;
        PeersMap peers_;
        std::map<std::string, std::shared_ptr<ipc::IConnector>> connectors_;
        LiveData<DeviceList> deviceList_;
        std::function<void()> deviceListRequest_;
        const std::shared_ptr<ICallbackQueue> lifecycle_;
    };

} // namespace quasar
