#pragma once

#include "resolve_info.h"

#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/interfaces/device_state/i_device_state_provider.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/retry_delay_counter.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/glagol_sdk/backend_api.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/jwt/jwt.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/libs/websocket/websocket_server.h>
#include <yandex_io/protos/model_objects.pb.h>

#include <functional>
#include <map>
#include <memory>
#include <optional>

namespace quasar {

    /**
     * A WS server that:
     *  -- verifies incoming connections via JWT tokens
     *  -- keeps track of connections
     *  -- notifies about current connections
     *  -- closes connections on timeouts  TODO: implement!
     */
    class GlagolWsServer {
    public:
        struct ConnectionDetails {
            std::string host;
            std::optional<std::string> deviceId;
            proto::GlagoldState::ConnectionType type = proto::GlagoldState::GLAGOL_APP;

            ConnectionDetails(std::string h);

            bool changeType(proto::GlagoldState::ConnectionType newType);
        };

        GlagolWsServer(std::shared_ptr<YandexIO::IDevice> device,
                       std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                       std::shared_ptr<IAuthProvider> authProvider,
                       std::shared_ptr<IDeviceStateProvider> deviceStateProvider);
        ~GlagolWsServer();

        struct MsgConnInfo {
            proto::GlagoldState::ConnectionType type;
            bool guestMode;
        };

        void setGuestMode(bool /*newValue*/);

        using OnMessageCallback = std::function<void(WebsocketServer::ConnectionHdl, MsgConnInfo, Json::Value)>;
        void setOnMessage(OnMessageCallback onMessage);

        /**
         * Sets a callback that receives state of connections whenever it changes
         * NOTE: This setter should be called once with nonnull value
         */
        void setOnConnectionsChanged(std::function<void(proto::GlagoldState)> onConnectionsChanged);

        using OnCloseCallback = std::function<void(const ConnectionDetails&)>;
        void setOnClose(OnCloseCallback /*cb*/);

        void send(WebsocketServer::ConnectionHdl hdl, const std::string& msg);

        /**
         * Sends `msg` to all verified connections.
         */
        using SendFilterPredicate = std::function<bool(WebsocketServer::ConnectionHdl, proto::GlagoldState::ConnectionType)>;
        void sendAll(const std::string& msg, const SendFilterPredicate& pred = {});

        void start();

        /**
         * Marks this connection as being of said semantic type -- e.g. a leader or a remote app.
         *
         * Probably type is based on a message from that connection handle, see `Glagol.cc`.
         */
        proto::GlagoldState::ConnectionType connectionType(WebsocketServer::ConnectionHdl hdl) const;
        void setConnectionType(WebsocketServer::ConnectionHdl hdl, const proto::GlagoldState::ConnectionType& connectionType);
        std::optional<glagol::ResolveInfo> setConnectionDetailsExt(WebsocketServer::ConnectionHdl hdl,
                                                                   const proto::GlagoldState::ConnectionType& connectionType,
                                                                   std::string deviceId);

        std::optional<int> getConfiguredPort() const;
        /*
         * Test purposes only
         */
        int getPort() const;

        void waitServerStart() const;

        /**
         * call onConnectionsChanged_ with actual status
         */
        void notifyGlagoldStatus();

        void updateAccountDevices(glagol::BackendApi::DevicesMap /*newDevices*/);

        Json::Value connectedDevicesTelemetry(Json::Value /*result*/) const;

        template <typename Func_>
        void applyConnectionDetails(WebsocketServer::ConnectionHdl hdl, Func_ func) {
            std::lock_guard<std::mutex> lock(mutex_);

            auto iter = verifiedConnections_.find(hdl);
            if (iter != verifiedConnections_.end()) {
                func(iter->second);
            }
        }

    private:
        class JwtToken {
        public:
            /**
             * @brief Parses jwt from string.
             * @details The device id, platform, expiration ts are parsed from grants: "sub", "plt", "exp"
             *
             * @param token in jwt format
             */
            explicit JwtToken(const std::string& token);
            bool isForDevice(const glagol::DeviceId& deviceId) const;
            bool isExpired() const;
            bool isGuest() const;
            std::chrono::system_clock::time_point getExpiration() const;

            static const std::string DEVICE_ID_GRANT;
            static const std::string DEVICE_PLATFORM_GRANT;
            static const std::string TOKEN_EXPIRATION_GRANT;
            static const std::string GUEST_GRANT;

        private:
            JwtToken(JwtPtr /*jwt*/);
            const glagol::DeviceId deviceId_;
            const bool guest_;
            const std::chrono::system_clock::time_point expirationTimePoint_;
        };

        void notifyGlagoldStatusNoLock();
        void sendNoLock(WebsocketServer::ConnectionHdl hdl, const std::string& msg);
        bool myCertsChanged(const glagol::BackendApi::DevicesMap& oldDevices) const;
        WebsocketServer::Settings makeWsSettings();
        void handleAuthInfo(std::shared_ptr<const AuthInfo2> authInfo);
        void handleDeviceState(std::shared_ptr<const DeviceState> deviceState);

        void onMessage(WebsocketServer::ConnectionHdl /*hdl*/, const std::string& msgStr);
        void onClose(WebsocketServer::ConnectionHdl hdl, Websocket::ConnectionInfo /*connectionInfo*/);

        WebsocketServer::Settings waitForWsSettings();
        bool wsSettingsChanged(const WebsocketServer::Settings& /*oldSettings*/);
        void serverThreadFunc();
        bool configured() const;
        void closeConnection(WebsocketServer::ConnectionHdl hdl, const std::string& error_string, Websocket::StatusCode error_code);
        bool quit() const;

        /**
         * @brief removes all expired tokens. Must be called under mutex_;
         */
        void removeExpiredTokens();

        /**
         * @brief checks if token is: for this device, not expired, signed by backend
         *
         * @param token
         * @return true if token is ok and false otherwise
         */
        std::optional<JwtToken> processToken(const std::string& token);

        Lifetime lifetime_;
        std::shared_ptr<YandexIO::IDevice> device_;
        std::shared_ptr<ipc::IIpcFactory> ipcFactory_;
        std::shared_ptr<IAuthProvider> authProvider_;
        std::shared_ptr<IDeviceStateProvider> deviceStateProvider_;
        mutable std::mutex mutex_;
        std::thread serverThread_;
        std::unique_ptr<WebsocketServer> server_;
        OnMessageCallback onMessage_;
        OnCloseCallback onClose_;
        std::function<void(proto::GlagoldState)> onConnectionsChanged_;

        std::string backendUrl_;
        std::optional<int> configuredPort_;
        glagol::BackendApi backendApi_;
        RetryDelayCounter backendRetryDelay_;

        glagol::DeviceId deviceId_;

        AuthInfo2 authInfo_;
        DeviceState::Configuration configurationState_ = DeviceState::Configuration::UNDEFINED;
        bool hasCriticalUpdate_ = false;
        std::atomic_bool guestMode_ = {false};

        bool quit_ = false;
        mutable SteadyConditionVariable fromServerThreadCV_;
        mutable SteadyConditionVariable toServerThreadCV_;

        std::map<WebsocketServer::ConnectionHdl, ConnectionDetails, std::owner_less<WebsocketServer::ConnectionHdl>> verifiedConnections_;
        using TokensMap = std::map<std::string, JwtToken>;
        TokensMap verifiedJwtTokens_;

        struct TokensMapIteratorExpireCmp {
            bool operator()(TokensMap::iterator a, TokensMap::iterator b) const {
                return a->second.getExpiration() < b->second.getExpiration();
            }
        };

        std::set<TokensMap::iterator, TokensMapIteratorExpireCmp> jwtTokensExpire_;

        std::mutex accountDevicesMutex_;
        glagol::BackendApi::DevicesMap accountDevices_;

        std::optional<int> port_;

        /* Keep callback queue last, so it will be destroyed first */
        quasar::NamedCallbackQueue userCallbackQueue_{"GlagolWsServer"};
    };

} /* namespace quasar */
