#include "mdns_holder.h"

#include "mdns_responder.h"
#include "proto_utils.h"
#include "resolve_handler.h"
#include "responder_status.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/mdns/nsd_receiver.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <tuple>

using namespace quasar;

namespace {

    class MDNSResponderHolder: public MDNSHolder {
        Lifetime lifetime_;
        MDNSResponder responder_;

    public:
        using Settings = MDNSResponder::Settings;

        static Settings makeSettings(const MdnsSettings& settings, const glagol::AvahiSettings& avahiSettings) {
            Settings result;
            result.ignoreIfNames = {"lo", "usb0"};
            result.externalPort = settings.port;
            result.hostnamePrefix = settings.prefix;
            result.avahi = avahiSettings;
            return result;
        }

        MDNSResponderHolder(std::shared_ptr<YandexIO::IDevice> device,
                            const MdnsSettings& settings,
                            const glagol::AvahiSettings& avahiSettings,
                            ResolveHandler* resolveHandler)
            : responder_(device, makeSettings(settings, avahiSettings), resolveHandler)
        {
        }

        void setNetworkAvailability(bool status) override {
            responder_.setNetworkAvailability(status);
        }

        std::shared_ptr<ResponderStatus> getResponderStatusProvider() override {
            return responder_.getResponderStatusProvider();
        }

        void reconfigure() override{}; // do nothing
    };

    class NsdReceiver: public INsdReceiver {
    public:
        NsdReceiver(ResolveHandler* resolveHandler)
            : resolveHandler_(resolveHandler)
        {
        }

        void onNsdDiscovered(const proto::NsdInfo& nsdInfo) override {
            auto [item, info] = glagol::convertResolveItemFull(nsdInfo);
            YIO_LOG_INFO("NSD item discovered: " << item.name);
            resolveHandler_->newResolve(item, info);
        }

        void onNsdListDiscovered(const proto::NsdInfoList& nsdInfoList) override {
            YIO_LOG_INFO("NSD " << nsdInfoList.items().size() << " discovered items");
            for (const auto& nsdItem : nsdInfoList.items()) {
                auto [item, info] = glagol::convertResolveItemFull(nsdItem);
                YIO_LOG_DEBUG("NSD item: " << item.name);
                resolveHandler_->newResolve(item, info);
            }
        }

        void onNsdLost(const proto::NsdInfo& nsdInfo) override {
            auto item = glagol::convertResolveItem(nsdInfo);
            resolveHandler_->removeResolve(item);
        }

    private:
        ResolveHandler* resolveHandler_;
    };

    class NSDHandlersHolder: public MDNSHolder {
        INsdMessagerPtr nsdMessager_;

    public:
        struct DummyStatusProvider: public ResponderStatus {
            void pollCompleted(std::string /*lastStatus*/) override{};
            std::string getResponderStatusJson() override {
                return "{\"nsdMode\":true}";
            };
            bool isNetworkAvailable() const override {
                return false;
            };
        };

        NSDHandlersHolder(const MdnsSettings& settings,
                          INsdMessagerFactory& nsdMessagerFactory,
                          ResolveHandler* resolveHandler)
            : nsdMessager_{nsdMessagerFactory.createMessager(std::make_unique<NsdReceiver>(resolveHandler))}
        {
            nsdMessager_->enableNsd(settings.flags.guestMode, settings.port, settings.flags.stereopair, settings.flags.tandem);
            YIO_LOG_DEBUG("NSD NSDHandlersHolder sendEnableNsd");
        }

        std::shared_ptr<ResponderStatus> getResponderStatusProvider() override {
            return std::make_shared<DummyStatusProvider>();
        }

        void setNetworkAvailability(bool /*isAvailable*/) override{};

        void reconfigure() override {
            nsdMessager_->disableNsd();
        }
    };

    struct TransportSelector {
        std::shared_ptr<YandexIO::IDevice> device_;
        const MdnsSettings& settings_;
        ResolveHandler* resolveHandler_;
        INsdMessagerFactory& nsdMessagerFactory_;

        TransportSelector(std::shared_ptr<YandexIO::IDevice> device,
                          const MdnsSettings& settings,
                          ResolveHandler* handler,
                          INsdMessagerFactory& nsdMessagerFactory)
            : device_(device)
            , settings_(settings)
            , resolveHandler_(handler)
            , nsdMessagerFactory_(nsdMessagerFactory)
        {
        }

        MDNSHolderPtr operator()(const glagol::AvahiSettings& avahiSettings) {
            return std::make_shared<MDNSResponderHolder>(device_, settings_, avahiSettings, resolveHandler_);
        }

        MDNSHolderPtr operator()(const NsdSettings& /*settings*/) {
            return std::make_shared<NSDHandlersHolder>(settings_, nsdMessagerFactory_, resolveHandler_);
        }
    };

} // anonymous namespace

MDNSHolderFactory quasar::createDefaultMndsFactory(INsdMessagerFactory& nsdMessagerFactory) {
    return [&nsdMessagerFactory](std::shared_ptr<YandexIO::IDevice> device, const MdnsSettings& settings, ResolveHandler* handler, MDNSTransportOpt transport) -> MDNSHolderPtr {
        return std::visit(TransportSelector(device, settings, handler, nsdMessagerFactory), transport);
    };
}
