#include "mdns_responder.h"

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/telemetry/telemetry.h>

#include <avahi-core/log.h>

YIO_DEFINE_LOG_MODULE("glagol");

namespace {
    constexpr std::chrono::duration TIME_TILL_REFRESH_AFTER_BROWSER_CREATE = std::chrono::seconds(10);
    constexpr std::chrono::duration TOO_FAST_COLLISION_PERIOD = std::chrono::seconds(2);
    constexpr const char* YANDEX_IO_SERVICE_TYPE = "_yandexio._tcp";

    std::string serverErrorString(AvahiServer* server) {
        return avahi_strerror(avahi_server_errno(server));
    }

    glagol::ResolveItem::Info makeResolveItemInfo(const char* hostname, int port, const char* address, AvahiStringList* txtList) {
        glagol::ResolveItem::Info::StringMap txt;
        std::string entry;
        for (auto n = txtList; n; n = n->next) {
            entry.assign((char*)n->text, n->size);
            auto pos = entry.find('=');
            if (pos != std::string::npos) {
                txt.emplace(entry.substr(0, pos), entry.substr(pos + 1));
            }
        }

        return glagol::ResolveItem::Info{.hostname = hostname, .port = port, .address = address, .txt = txt};
    }

    std::string leaderOrFollower(bool val) {
        return val ? "leader" : "follower";
    }

    std::string roleToString(const std::optional<bool>& opt) {
        if (opt.has_value()) {
            return leaderOrFollower(opt.value());
        }
        return "unset";
    }

    class AvahiStringListWrapper {
        AvahiStringList* lst;

    public:
        template <typename... Param_>
        AvahiStringListWrapper(Param_... params)
            : lst(avahi_string_list_new(params..., nullptr))
        {
            if (lst == nullptr) {
                throw std::runtime_error("Failed to allocate AvahiStringList");
            }
        }

        ~AvahiStringListWrapper() {
            avahi_string_list_free(lst);
        }

        void add(const std::string& str) {
            add(str.c_str());
        }

        void add(const char* str) {
            auto newBegining = avahi_string_list_add(lst, str);
            if (newBegining == nullptr) {
                throw std::runtime_error("Failed to add to AvahiStringList");
            }
            lst = newBegining;
        }

        AvahiStringList* get() {
            return lst;
        }
    };
} // namespace

namespace glagol {

    std::string toString(const glagol::AvahiSettings::Flags& flags) {
        // clang-format off
        return("guestMode=" + std::to_string(flags.guestMode)
               + ", stereopair=" + roleToString(flags.stereopair)
               + ", tandem=" + roleToString(flags.tandem));
        // clang-format on
    }
} // namespace glagol

using namespace quasar;

class MDNSResponder::StatusProxy: public ResponderStatus {
public:
    StatusProxy(MDNSResponder* responder)
        : responder_(responder)
              {};

    void pollCompleted(std::string lastStatus) override {
        std::lock_guard<std::mutex> lock(mutex_);
        responder_ = nullptr;
        lastStatus_ = std::move(lastStatus);
    }

private:
    mutable std::mutex mutex_;
    MDNSResponder* responder_;
    std::string lastStatus_;

    std::string getResponderStatusJson() override {
        std::lock_guard<std::mutex> lock(mutex_);
        if (responder_) {
            return responder_->getStateJson();
        }
        return lastStatus_;
    }

    bool isNetworkAvailable() const override {
        std::lock_guard<std::mutex> lock(mutex_);
        return responder_->networkAvailable_;
    }
};

MDNSResponder::StateHolder::StateHolder()
    : cur_({State::UNKNOWN, std::chrono::steady_clock::now()})
    , prev_(cur_)
{
}

MDNSResponder::StateHistory MDNSResponder::StateHolder::get() {
    std::lock_guard<std::mutex> lock(mutex_);
    return {cur_.state, prev_.state, std::chrono::steady_clock::now() - cur_.whenSet};
}

MDNSResponder::StateHolder& MDNSResponder::StateHolder::operator=(MDNSResponder::State newValue) {
    std::lock_guard<std::mutex> lock(mutex_);
    if (prev_.state != cur_.state) {
        prev_ = std::exchange(cur_, {newValue, std::chrono::steady_clock::now()});
    };
    return *this;
}

namespace {
    void avahiLoggingFunction(AvahiLogLevel level, const char* txt) {
        const char* AVAHI_TAG = "{glagold-avahi}: ";

        switch (level) {
            case AVAHI_LOG_ERROR:
                YIO_LOG_ERROR_EVENT("Avahi.Error", AVAHI_TAG << txt);
                break;
            case AVAHI_LOG_WARN:
                YIO_LOG_WARN(AVAHI_TAG << txt);
                break;
            case AVAHI_LOG_NOTICE:
            case AVAHI_LOG_INFO:
                YIO_LOG_INFO(AVAHI_TAG << txt);
                break;
            case AVAHI_LOG_DEBUG:
            case AVAHI_LOG_LEVEL_MAX:
                YIO_LOG_DEBUG(AVAHI_TAG << txt);
                break;
        }
    }

    struct AvahiServerConfigHolder {
        AvahiServerConfig config;
        AvahiServerConfigHolder() {
            avahi_server_config_init(&config);
        }
        AvahiServerConfig* operator->() {
            return &config;
        };
        AvahiServerConfig* operator*() {
            return &config;
        };
        ~AvahiServerConfigHolder() {
            avahi_server_config_free(&config);
        }
    };

    struct AvahiSServiceResolverDeleter {
        void operator()(AvahiSServiceResolver* ptr) {
            avahi_s_service_resolver_free(ptr);
        }
    };
} // namespace

std::string MDNSResponder::myServerErrorString() {
    return serverErrorString(server_.get());
}

MDNSResponder::MDNSResponder(std::shared_ptr<YandexIO::IDevice> device,
                             const Settings& settings,
                             ResolveHandler* resolveHandler)
    : device_(std::move(device))
    , externalPort_(settings.externalPort)
    , recreateBrowserInterval_(settings.avahi.recreateBrowserInterval)
    , queuingResolves_(settings.avahi.queuingResolves)
    , txtFlags_(settings.avahi.flags)
    , resolveHandler_(resolveHandler)
    , previousServerCollision_(std::chrono::steady_clock::now())
{
    responderState_ = State::CONSTRUCTING;
    avahi_set_log_function(&avahiLoggingFunction);
    int error{0};

    /* Initialize the pseudo-RNG */
    {
        timeval tv;
        gettimeofday(&tv, nullptr);
        srand(tv.tv_sec + tv.tv_usec);
    };

    simplePoll_.reset(avahi_simple_poll_new());
    /* Allocate main loop object */
    if (!simplePoll_) {
        throw std::runtime_error("Failed to create simple poll object");
    }

    avahiServiceName_.reset(avahi_strdup(("YandexIOReceiver-" + device_->deviceId()).c_str()));

    /* Let's set the host name for this server. */
    AvahiServerConfigHolder config;

    config->host_name = avahi_strdup((settings.hostnamePrefix + device_->deviceId()).c_str()); // avahi needs char*
    config->publish_workstation = 0;
    config->ratelimit_interval = settings.avahi.ratelimitIntervalUsec;
    config->ratelimit_burst = settings.avahi.ratelimitBurst;
    if (!settings.ignoreIfNames.empty()) {
        auto iter = settings.ignoreIfNames.begin();
        config->deny_interfaces = avahi_string_list_new(iter->c_str(), nullptr);
        std::for_each(++iter, settings.ignoreIfNames.end(),
                      [&config](const std::string& iface) {
                          config->deny_interfaces = avahi_string_list_add(config->deny_interfaces, iface.c_str());
                      });
    }
    //    config->n_cache_entries_max = 0; // FIXME: workaround for https://github.com/lathiat/avahi/issues/117 / https://st.yandex-team.ru/QUASAR-1760#5c17f2f7852aee001c5b21b4 ! Remove that once upstream is fixed!

    if (settings.avahi.restrictToIPv4) {
        YIO_LOG_INFO("NB: Restricting avahi advertisement");

        config->publish_domain = 0;
        config->use_ipv6 = 0;
        config->publish_aaaa_on_ipv4 = 0;
        config->publish_a_on_ipv6 = 0;

        device_->telemetry()->reportEvent("GLAGOLD_AVAHI_START_RESTRICTED_V4");
    } else {
        device_->telemetry()->reportEvent("GLAGOLD_AVAHI_START_UNRESTRICTED");
    }

    YIO_LOG_INFO("Creating avahi server with hostname <" << config->host_name << ">");

    /* Allocate a new server */
    server_.reset(
        avahi_server_new(
            avahi_simple_poll_get(simplePoll_.get()),
            *config,
            &MDNSResponder::serverCallbackHelper,
            this,
            &error));

    /* Check whether creating the server object succeeded */
    if (!server_) {
        throw std::runtime_error(std::string("Failed to create server: ") + avahi_strerror(error));
    }

    if (resolveHandler) {
        createBrowser();

        if (!browser_) {
            throw std::runtime_error("Failed to create service browser: " + myServerErrorString());
        }
        browserCreatedTime_ = std::chrono::steady_clock::now();
    } else {
        YIO_LOG_INFO("Resolve handler isn't set. Run in only anouncing mode.");
    }

    struct timeval tv;
    // first time give couple seconds to establish connection
    auto timer = avahi_simple_poll_get(simplePoll_.get())->timeout_new(avahi_simple_poll_get(simplePoll_.get()), avahi_elapse_time(&tv, 1000 * 3, 0), stopWatcherCallbackHelper, this);

    if (!timer) {
        throw std::runtime_error("Failed to create timeout to watch for stop");
    }

    YIO_LOG_INFO("Create MDNS responder: port="
                 << settings.externalPort
                 << ", " << glagol::toString(txtFlags_));
    responderState_ = State::STARTING;
    avahiThread_ = std::thread([this]() { startDiscovery(); });
}

void MDNSResponder::createBrowser() {
    browser_.reset(
        avahi_s_service_browser_new(
            server_.get(),
            AVAHI_IF_UNSPEC,
            AVAHI_PROTO_UNSPEC,
            YANDEX_IO_SERVICE_TYPE,
            nullptr,
            (AvahiLookupFlags)0,
            browseCallbackHelper,
            this));
}

std::string MDNSResponder::getStateJson() {
    auto state = responderState_.get();
    Json::Value jsonVal;
    jsonVal["curState"] = std::to_string(int(state.curState));
    jsonVal["prevState"] = std::to_string(int(state.prevState));
    jsonVal["howLongInCurState"] = std::to_string(state.howLongInCurState.count());
    return jsonToString(jsonVal);
}

std::shared_ptr<ResponderStatus> MDNSResponder::getResponderStatusProvider() {
    if (!statusProvider_) {
        statusProvider_ = std::make_shared<StatusProxy>(this);
    }
    return statusProvider_;
}

void MDNSResponder::setNetworkAvailability(bool isAvailable) {
    networkAvailable_.store(isAvailable);
}

// stops all the threads and waits for them
// TODO: maybe refactor to reduce copypasta within and with other services?
MDNSResponder::~MDNSResponder() {
    YIO_LOG_INFO("Stopping ...");
    stopped_ = true;
    avahiThread_.join();
    YIO_LOG_INFO("... Stopped");
    if (statusProvider_) {
        statusProvider_->pollCompleted(getStateJson());
    }
}

void MDNSResponder::avahiPollQuit() {
    YIO_LOG_DEBUG("avahi_simple_poll_quit");
    avahi_simple_poll_quit(simplePoll_.get());
}

void MDNSResponder::resolveCallbackHelper(AvahiSServiceResolver* resolver, AvahiIfIndex /*idx*/, AvahiProtocol /*protocol*/,
                                          AvahiResolverEvent event, const char* name, const char* type,
                                          const char* domain, const char* host_name, const AvahiAddress* address,
                                          uint16_t port, AvahiStringList* txt, AvahiLookupResultFlags /*flags*/,
                                          void* userdata) noexcept {
    std::unique_ptr<AvahiSServiceResolver, AvahiSServiceResolverDeleter> resolverHolder(resolver);
    auto responder = reinterpret_cast<MDNSResponder*>(userdata);

    responder->resolveCallback(event, name, type, domain, host_name, address, port, txt);
}

void MDNSResponder::resolveCallback(AvahiResolverEvent event, const char* name, const char* type,
                                    const char* domain, const char* hostName, const AvahiAddress* address,
                                    uint16_t port, AvahiStringList* txt) {
    /* Called whenever a service has been resolved successfully or timed out */

    YIO_LOG_INFO("resolverCallback: event = " << event); //<< " name = '" << name << "' type = '" << type <<"' domain = '" << domain << "'");
    switch (event) {
        case AVAHI_RESOLVER_FAILURE: {
            YIO_LOG_WARN("Failed to resolve: " << myServerErrorString());
            break;
        }

        case AVAHI_RESOLVER_FOUND: {
            char a[AVAHI_ADDRESS_STR_MAX];
            avahi_address_snprint(a, sizeof(a), address);
            YIO_LOG_INFO("Resolved: " << name << ' ' << type << ' ' << domain << " @ " << hostName << ':' << port << ' ' << a);
            for (auto n = txt; n; n = n->next) {
                YIO_LOG_INFO('\t' << std::string((char*)n->text, n->size));
            }

            resolveHandler_.newResolve(
                {.name = name, .type = type, .domain = domain},
                makeResolveItemInfo(hostName, port, a, txt));
            break;
        }
    }
    --resolvesInProgress_;
    if (!pendingResolves_.empty()) {
        auto [ifIndex, protocol, name, type, domain] = pendingResolves_.front();
        startResolve(ifIndex, protocol, name.data(), type.data(), domain.data());
        pendingResolves_.pop_front();
    }
}

void MDNSResponder::browseCallbackHelper(AvahiSServiceBrowser* browser [[maybe_unused]], AvahiIfIndex ifIndex, AvahiProtocol protocol,
                                         AvahiBrowserEvent event, const char* name, const char* type, const char* domain,
                                         AvahiLookupResultFlags flags, void* userdata) noexcept {
    auto responder = reinterpret_cast<MDNSResponder*>(userdata);
    if (browser == responder->browser_.get()) {
        responder->browseCallback(event, name, type, domain, ifIndex, protocol, flags);
    } else {
        YIO_LOG_ERROR_EVENT("MDNSResponder.BrowseCallbackFailure", "Called with wrong browser " << browser << " instead of " << responder->browser_.get());
    }
}

void MDNSResponder::startResolve(AvahiIfIndex ifIndex, AvahiProtocol protocol, const char* name, const char* type, const char* domain) {
    YIO_LOG_DEBUG("Start resolve");
    if (!(avahi_s_service_resolver_new(server_.get(), ifIndex, protocol, name, type, domain,
                                       AVAHI_PROTO_UNSPEC, (AvahiLookupFlags)0, resolveCallbackHelper, this))) {
        YIO_LOG_ERROR_EVENT("MDNSResponder.AvahiBrowserNewFailure", "Failed to create new resolver: " << myServerErrorString());
        YIO_LOG_INFO(name << ' ' << type << ' ' << domain);
    } else {
        ++resolvesInProgress_;
    }
}

void MDNSResponder::browseCallback(AvahiBrowserEvent event, const char* name, const char* type, const char* domain, AvahiIfIndex ifIndex, AvahiProtocol protocol, AvahiLookupResultFlags flags) {
    /* Called whenever a new services becomes available on the LAN or is removed from the LAN */
    YIO_LOG_INFO("browserCallback: event = " << event); // << " name = '" << name << "' type = '" << type <<"' domain = '" << domain << "'");
    switch (event) {
        case AVAHI_BROWSER_FAILURE: {
            YIO_LOG_ERROR_EVENT("MDNSResponder.AvahiBrowserFailure", "Browser failure: " << myServerErrorString());
            avahiPollQuit();
            break;
        }
        case AVAHI_BROWSER_NEW: {
            YIO_LOG_INFO("New browser " << name << ' ' << type << ' ' << domain);
            YIO_LOG_DEBUG("LOCAL: " << !!(flags & AVAHI_LOOKUP_RESULT_LOCAL)
                                    << ", OUR_OWN: " << !!(flags & AVAHI_LOOKUP_RESULT_OUR_OWN)
                                    << ", WIDE_AREA: " << !!(flags & AVAHI_LOOKUP_RESULT_WIDE_AREA)
                                    << ", MULTICAST: " << !!(flags & AVAHI_LOOKUP_RESULT_MULTICAST)
                                    << ", CACHED: " << !!(flags & AVAHI_LOOKUP_RESULT_CACHED)
                                    << ", queuingResolves: " << queuingResolves_);
            /* We ignore the returned resolver object. In the callback
           function we free it. If the server is terminated before
           the callback function is called the server will free
           the resolver for us. */
            if (queuingResolves_ && resolvesInProgress_) {
                YIO_LOG_DEBUG("Added to pending resolves");
                pendingResolves_.emplace_back(ifIndex, protocol, name, type, domain);
            } else {
                startResolve(ifIndex, protocol, name, type, domain);
            }
            break;
        }

        case AVAHI_BROWSER_REMOVE: {
            YIO_LOG_INFO("Remove browser " << name << ' ' << type << ' ' << domain);
            resolveHandler_.removeResolve({.name = name, .type = type, .domain = domain});
            break;
        }
        case AVAHI_BROWSER_ALL_FOR_NOW:
            break;
        case AVAHI_BROWSER_CACHE_EXHAUSTED:
            break;
    }
}

/**
 * Userdata actually gets a MDNSResponder*
 */
void MDNSResponder::entryGroupCallbackHelper(AvahiServer* s, AvahiSEntryGroup* g, AvahiEntryGroupState state, AVAHI_GCC_UNUSED void* userdata) noexcept {
    try {
        YIO_LOG_INFO("{avahi} entry group callback: state {" << state << "}");

        MDNSResponder* responder = reinterpret_cast<MDNSResponder*>(userdata);
        if (responder && s) {
            responder->entryGroupCallback(s, g, state);
        } else {
            YIO_LOG_ERROR_EVENT("MDNSResponder.GroupCallbackFailure", "Null " << (responder ? "" : "responder") << " " << (s ? "" : "server"));
        }
    } catch (...) { // to avoid std::terminate
    }
}

void MDNSResponder::entryGroupCallback(AvahiServer* s, AvahiSEntryGroup* g [[maybe_unused]], AvahiEntryGroupState state) {
    if (g != group_) {
        YIO_LOG_ERROR_EVENT("MDNSResponder.GroupCallbackFailure", "Wrong group " << g << " instead of " << group_);
        return;
    }

    /* Called whenever the entry group state changes */

    switch (state) {
        case AVAHI_ENTRY_GROUP_ESTABLISHED:
            groupCollisionsAttempts_ = 0;
            /* The entry group has been established successfully */
            YIO_LOG_INFO("Service '" << avahiServiceName_.get() << "' successfully established.");
            responderState_ = State::RUNNING;
            if (avahiStartedLatencyPoint_) {
                device_->telemetry()->reportLatency(std::move(avahiStartedLatencyPoint_), "GlagolAvahiServerStarting");
            }
            break;

        case AVAHI_ENTRY_GROUP_COLLISION: {
            if (++groupCollisionsAttempts_ > 2) {
                YIO_LOG_ERROR_EVENT("MDNSResponder.TooManyGroupCollisions", "Too many group collisions in row");
                avahiPollQuit();
                break;
            };
            /* A service name collision happened. Let's pick a new name.
               NOTICE: logically it's impossible event and statistics says that it never happened on users */
            avahiServiceName_.reset(avahi_alternative_service_name(avahiServiceName_.get()));

            YIO_LOG_WARN("Service name collision, renaming service to '" << avahiServiceName_.get() << "'");

            /* And recreate the services */
            createServices(s);
            break;
        }

        case AVAHI_ENTRY_GROUP_FAILURE:

            YIO_LOG_ERROR_EVENT("MDNSResponder.EntryGroupFailure", "Entry group failure: " << serverErrorString(s));

            /* Some kind of failure happened while we were registering our services */
            /* NOTICE: never happened on users */
            avahiPollQuit();
            break;

        case AVAHI_ENTRY_GROUP_UNCOMMITED:
        case AVAHI_ENTRY_GROUP_REGISTERING:;
    }
}

void MDNSResponder::createServices(AvahiServer* s) {
    responderState_ = State::CREATING_SERVICES;
    try {
        /* store deviceId as txt record */
        std::string deviceIdTextRecord = "deviceId=" + device_->deviceId();
        std::string platformTextRecord = "platform=" + device_->configuration()->getDeviceType();
        const char* clusterTextRecord = "cluster=yes";
        const char* guestTextRecord = "guest=yes";

        /* If this is the first time we're called, let's create a new entry group */
        if (!group_) {
            if (!(group_ = avahi_s_entry_group_new(s, entryGroupCallbackHelper, this))) {
                throw std::runtime_error("avahi_entry_group_new() failed: " + serverErrorString(s));
            }
        }

        YIO_LOG_INFO("Adding service '" << avahiServiceName_.get() << ' ' << glagol::toString(txtFlags_));

        auto txtRecords = AvahiStringListWrapper(deviceIdTextRecord.c_str(),
                                                 platformTextRecord.c_str(),
                                                 clusterTextRecord);

        if (txtFlags_.guestMode) {
            txtRecords.add(guestTextRecord);
        }
        if (txtFlags_.stereopair) {
            txtRecords.add("sp=" + leaderOrFollower(txtFlags_.stereopair.value()));
        }
        if (txtFlags_.tandem) {
            txtRecords.add("td=" + leaderOrFollower(txtFlags_.tandem.value()));
        }

        /* Add the service for IPP */
        int ret = avahi_server_add_service_strlst(s, group_, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC, AVAHI_PUBLISH_UPDATE, avahiServiceName_.get(), YANDEX_IO_SERVICE_TYPE, nullptr, nullptr, externalPort_,
                                                  txtRecords.get());

        if (ret < 0) {
            throw std::runtime_error(std::string("Failed to add ") + YANDEX_IO_SERVICE_TYPE + " service: " + avahi_strerror(ret) + "/" + std::to_string(ret));
        }

        ret = avahi_s_entry_group_commit(group_);

        /* Tell the server to register the service */
        if (ret < 0) {
            throw std::runtime_error(std::string("Failed to commit entry_group: ") + avahi_strerror(ret));
        }
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("MDNSResponder.FailedCreateServices", "Cannot create services: " << e.what());
        avahiPollQuit();
        throw;
    }
}

/**
 * userdata actually gets MDNSResponder* -- see `serverCallback` usage
 */
void MDNSResponder::serverCallbackHelper(AvahiServer* s, AvahiServerState state, void* userdata) noexcept {
    try {
        YIO_LOG_INFO("{avahi} server callback: state {" << state << "}");

        MDNSResponder* responder = reinterpret_cast<MDNSResponder*>(userdata);
        if (responder && s) {
            responder->serverCallback(s, state);
        } else {
            YIO_LOG_ERROR_EVENT("MDNSResponder.ServerCallbackFailure", "Null " << (responder ? "" : "responder") << " " << (s ? "" : "server"));
        }
    } catch (...) { // to avoid std::terminate
    }
}

int MDNSResponder::serverCollisions() {
    const auto now = std::chrono::steady_clock::now();
    if (now - previousServerCollision_ > TOO_FAST_COLLISION_PERIOD) {
        ++serverCollisionsAttempts_;
    } else {
        serverCollisionsAttempts_ = 0;
    }
    previousServerCollision_ = now;
    return serverCollisionsAttempts_;
}

void MDNSResponder::serverCallback(AvahiServer* s, AvahiServerState state) {
    /* Called whenever the server state changes */

    switch (state) {
        case AVAHI_SERVER_RUNNING:
            /* The serve has startup successfully and registered its host
             * name on the network, so it's time to create our services */

            if (!group_) {
                groupCollisionsAttempts_ = 0;
                createServices(s);
            }

            break;

        case AVAHI_SERVER_COLLISION: {
            responderState_ = State::COLLISION;
            /* This error happens only in rare cases when only restart will help.
             * ~10 attempts are enough to be stopped by watchdog
             */
            if (serverCollisions() > 10) {
                YIO_LOG_ERROR_EVENT("MDNSResponder.TooManyServerCollisions", "Too many server collisions in row");
                avahiPollQuit();
                break;
            };
            /* A host name collision happened. Let's pick a new name for the server */
            std::unique_ptr<char, AvahiFreeDeleter> n(avahi_alternative_host_name(avahi_server_get_host_name(s)));
            YIO_LOG_WARN("Host name collision, retrying with '" << n.get() << "'");
            int r = avahi_server_set_host_name(s, n.get());

            if (r < 0) {
                YIO_LOG_ERROR_EVENT("MDNSResponder.FailedSetHostName", "Failed to set new host name: " << avahi_strerror(r));
                avahiPollQuit();
                return;
            }
        }

            /* Fall through */

        case AVAHI_SERVER_REGISTERING:
            /* Let's drop our registered services. When the server is back
             * in AVAHI_SERVER_RUNNING state we will register them
             * again with the new host name. */
            if (group_) {
                avahi_s_entry_group_reset(group_);
            }

            break;

        case AVAHI_SERVER_FAILURE:

            /* Terminate on failure */

            YIO_LOG_ERROR_EVENT("MDNSResponder.ServerFailure", "Server failure: " << serverErrorString(s));
            avahiPollQuit();
            break;

        case AVAHI_SERVER_INVALID:;
    }
}

void MDNSResponder::stopWatcherCallbackHelper(AVAHI_GCC_UNUSED AvahiTimeout* e, void* userdata) noexcept {
    try {
        MDNSResponder* responder = reinterpret_cast<MDNSResponder*>(userdata);
        if (responder) {
            responder->stopWatcherCallback(e);
        } else {
            YIO_LOG_ERROR_EVENT("MDNSResponder.StopWatcherCallbackFailure", "Null server");
        }
    } catch (...) { // to avoid std::terminate
    }
}

void MDNSResponder::stopWatcherCallback(AvahiTimeout* e) {
    if (stopped_) {
        YIO_LOG_INFO("GlagolAvahiServer stopped gracefully");
        device_->telemetry()->reportEvent("GlagolAvahiGracefullyStop", getStateJson());
        avahiPollQuit();
    } else {
        if (browser_ && recreateBrowserInterval_ != std::chrono::seconds(0) && resolvesInProgress_ == 0) {
            auto now = std::chrono::steady_clock::now();
            auto timePassedSinceBrowseCreation = now - browserCreatedTime_;
            if (timePassedSinceBrowseCreation > recreateBrowserInterval_) {
                YIO_LOG_INFO("Recreate browser");
                browserCreatedTime_ = now;
                needToRefresh_ = true;
                resolveHandler_.refreshStart();
                createBrowser();
            } else if (needToRefresh_ && timePassedSinceBrowseCreation > TIME_TILL_REFRESH_AFTER_BROWSER_CREATE) {
                needToRefresh_ = false;
                resolveHandler_.refresh();
            }
        }
        struct timeval tv;
        avahi_simple_poll_get(simplePoll_.get())->timeout_update(e, avahi_elapse_time(&tv, 1000, 0));
    }
}

void MDNSResponder::startDiscovery() {
    avahiStartedLatencyPoint_ = device_->telemetry()->createLatencyPoint();
    /* Run the main loop */
    if (!stopped_) {
        YIO_LOG_DEBUG("avahi_simple_poll_loop");
        responderState_ = State::POLL_STARTING;
        avahi_simple_poll_loop(simplePoll_.get());
    }

    /* Avahi avahi_simple_poll_quit was called. Thread work is done, so clean up server */

    if (!stopped_) {
        device_->telemetry()->reportError("GlagolAvahiServerStopped");
        std::this_thread::sleep_for(std::chrono::seconds(5));
        if (networkAvailable_.load()) {
            abort(); // FIXME do not abort when glagold moves to yiod
        } else {
            exit(0);
        }
    }
}
