#pragma once

#include "avahi_settings.h"
#include "outdate_wrapper.h"
#include "resolve_handler.h"
#include "responder_status.h"

#include <yandex_io/libs/device/device.h>

#include <avahi-common/alternative.h>
#include <avahi-common/error.h>
#include <avahi-common/malloc.h>
#include <avahi-common/simple-watch.h>

#include <avahi-core/core.h>
#include <avahi-core/lookup.h>
#include <avahi-core/publish.h>

#include <chrono>
#include <list>
#include <mutex>

namespace quasar {
    struct AvahiServerDeleter {
        void operator()(AvahiServer* ptr) {
            avahi_server_free(ptr);
        }
    };
    struct AvahiSimplePollDeleter {
        void operator()(AvahiSimplePoll* ptr) {
            avahi_simple_poll_free(ptr);
        }
    };
    struct AvahiFreeDeleter {
        void operator()(void* ptr) {
            avahi_free(ptr);
        }
    };
    struct AvahiSServiceBrowserDeleter {
        void operator()(AvahiSServiceBrowser* ptr) {
            avahi_s_service_browser_free(ptr);
        }
    };

    /* more features in future, so Responder */
    class MDNSResponder {
    public:
        struct Settings {
            int externalPort;
            std::string hostnamePrefix;
            std::vector<std::string> ignoreIfNames;
            glagol::AvahiSettings avahi;
        };

        using time_point = std::chrono::steady_clock::time_point;
        enum class State {
            UNKNOWN,
            CONSTRUCTING,
            STARTING,
            POLL_STARTING,
            CREATING_SERVICES,
            RUNNING,
            COLLISION
        };

        struct StateHistory {
            State curState,
                prevState;
            time_point::duration howLongInCurState;
        };

        using ResolveItem = glagol::ResolveItem;

        MDNSResponder(std::shared_ptr<YandexIO::IDevice> /*device*/,
                      const Settings& /*settings*/,
                      ResolveHandler* /*resolveHandler*/ = nullptr);
        ~MDNSResponder();

        std::string getStateJson();
        std::shared_ptr<ResponderStatus> getResponderStatusProvider();
        void setNetworkAvailability(bool /*isAvailable*/);

    private:
        struct StateHolder {
            StateHolder();
            StateHistory get();
            StateHolder& operator=(State /*newValue*/);

        private:
            struct Value {
                State state;
                time_point whenSet;
            };
            std::mutex mutex_;
            Value cur_, prev_;
        };

        class StatusProxy;

        std::atomic_bool stopped_{false};
        std::atomic_bool networkAvailable_{false};
        StateHolder responderState_;
        std::shared_ptr<ResponderStatus> statusProvider_;

        std::shared_ptr<YandexIO::IDevice> device_;
        const int externalPort_;
        std::chrono::seconds recreateBrowserInterval_{0};
        bool queuingResolves_{false};
        glagol::AvahiSettings::Flags txtFlags_;
        int resolvesInProgress_{0};
        bool needToRefresh_{false};
        std::chrono::steady_clock::time_point browserCreatedTime_;

        glagol::OutdateWrapper resolveHandler_;
        std::list<std::tuple<AvahiIfIndex, AvahiProtocol, std::string, std::string, std::string>> pendingResolves_;

        std::thread avahiThread_;

        AvahiSEntryGroup* group_ = nullptr;
        std::unique_ptr<AvahiSimplePoll, AvahiSimplePollDeleter> simplePoll_;
        std::unique_ptr<AvahiServer, AvahiServerDeleter> server_;
        std::unique_ptr<AvahiSServiceBrowser, AvahiSServiceBrowserDeleter> browser_;
        int groupCollisionsAttempts_ = 0;
        int serverCollisionsAttempts_ = 0;
        time_point previousServerCollision_;
        std::unique_ptr<char, AvahiFreeDeleter> avahiServiceName_;
        std::shared_ptr<const YandexIO::LatencyData> avahiStartedLatencyPoint_;

        int serverCollisions();

        void startDiscovery();
        void createServices(AvahiServer* /*s*/);
        void avahiPollQuit();

        void createBrowser();

        // avahi callbacks
        static void entryGroupCallbackHelper(AvahiServer* /*s*/, AvahiSEntryGroup* /*g*/, AvahiEntryGroupState /*state*/, AVAHI_GCC_UNUSED void* /* userdata */) noexcept;
        static void serverCallbackHelper(AvahiServer* /*s*/, AvahiServerState /*state*/, AVAHI_GCC_UNUSED void* /* userdata */) noexcept;
        static void stopWatcherCallbackHelper(AVAHI_GCC_UNUSED AvahiTimeout*, void* /* userdata */) noexcept;
        static void browseCallbackHelper(AvahiSServiceBrowser* /*browser*/, AvahiIfIndex /*ifIndex*/, AvahiProtocol /*protocol*/,
                                         AvahiBrowserEvent event, const char* name, const char* type, const char* domain,
                                         AvahiLookupResultFlags /*flags*/, void* userdata) noexcept;
        static void resolveCallbackHelper(AvahiSServiceResolver* resolver,
                                          AVAHI_GCC_UNUSED AvahiIfIndex interface,
                                          AVAHI_GCC_UNUSED AvahiProtocol protocol,
                                          AvahiResolverEvent event,
                                          const char* name,
                                          const char* type,
                                          const char* domain,
                                          const char* host_name,
                                          const AvahiAddress* address,
                                          uint16_t port,
                                          AvahiStringList* txt,
                                          AvahiLookupResultFlags flags,
                                          AVAHI_GCC_UNUSED void* userdata) noexcept;
        // in-class callbacks
        void entryGroupCallback(AvahiServer* /*s*/, AvahiSEntryGroup* /*g*/, AvahiEntryGroupState /*state*/);
        void serverCallback(AvahiServer* /*s*/, AvahiServerState /*state*/);
        void stopWatcherCallback(AvahiTimeout* /*e*/);
        void browseCallback(AvahiBrowserEvent event, const char* name, const char* type, const char* domain,
                            AvahiIfIndex /*ifIndex*/, AvahiProtocol /*protocol*/, AvahiLookupResultFlags /*flags*/);
        void resolveCallback(AvahiResolverEvent event,
                             const char* name, const char* type, const char* domain, const char* hostName,
                             const AvahiAddress* address, uint16_t port,
                             AvahiStringList* txt);

        void startResolve(AvahiIfIndex ifIndex, AvahiProtocol protocol, const char* name, const char* type, const char* domain);

        std::string myServerErrorString();
    };
} // namespace quasar
