#include "mdnsd_nsd_messager.h"

#include "proto_utils.h"

#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/mdns/nsd_receiver.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <yandex_io/services/glagold/proto_utils.h>

#include <chrono>

using namespace quasar;

namespace {
    proto::GlagolDiscoveryItem_GroupRole roleFromOptBool(std::optional<bool> src) {
        if (src) {
            return src.value() ? proto::GlagolDiscoveryItem::LEADER : proto::GlagolDiscoveryItem::FOLLOWER;
        }
        return proto::GlagolDiscoveryItem::STAND_ALONE;
    }
} // namespace

MdnsdNsdMessager::MdnsdNsdMessager(INsdReceiverPtr receiver, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : mdnsdConnector_{ipcFactory->createIpcConnector("mdnsd")}
    , receiver_{std::move(receiver)}
{
    mdnsdConnector_->setMessageHandler(makeSafeCallback(
        [this](const auto& message) {
            if (message->has_nsd_discovered_items()) {
                receiver_->onNsdListDiscovered(message->nsd_discovered_items());
            }
            if (message->has_nsd_discovered_item()) {
                receiver_->onNsdDiscovered(message->nsd_discovered_item());
            }
            if (message->has_nsd_lost_item()) {
                receiver_->onNsdLost(message->nsd_lost_item());
            }
            if (message->has_nsd_disable_completed()) {
                std::lock_guard<std::mutex> lock(disable_completed_mutex_);
                nsdDisabled_ = true;
                cond_.notify_one();
            }
        }, lifetime_));
    mdnsdConnector_->setConnectHandler([this]() { // to avoid misses discovery_updated
        proto::QuasarMessage msg;
        {
            std::scoped_lock lock(data_mutex_);
            if (nsdDisabling_) {
                msg.mutable_nsd_disable();
            } else {
                msg.mutable_nsd_discovered_request();
                auto nsdEnable = msg.mutable_nsd_enable();
                nsdEnable->set_server_port(port_);
                nsdEnable->set_stereopair(stereopair_);
                nsdEnable->set_tandem(tandem_);
                if (guestMode_) {
                    msg.mutable_nsd_guest_mode();
                };
            }
        }
        YIO_LOG_INFO("Send mdnsd configuration to client: " << convertMessageToJsonString(msg));
        mdnsdConnector_->sendMessage(std::move(msg));
    });
    mdnsdConnector_->connectToService();
}

MdnsdNsdMessager::~MdnsdNsdMessager() {
    lifetime_.die();
}

void MdnsdNsdMessager::enableNsd(bool guestMode, uint32_t port, OptBool stereopair, OptBool tandem) {
    nsdDisabling_ = false;
    std::scoped_lock lock(data_mutex_);
    guestMode_ = guestMode;
    port_ = port;
    stereopair_ = roleFromOptBool(stereopair);
    tandem_ = roleFromOptBool(tandem);
    YIO_LOG_INFO("Enable NSD: port="
                 << port
                 << ", guestMode=" << guestMode_
                 << ", stereopair = " << proto::GlagolDiscoveryItem_GroupRole_Name(stereopair_)
                 << ", tandem = " << proto::GlagolDiscoveryItem_GroupRole_Name(tandem_));
}

void MdnsdNsdMessager::disableNsd() {
    nsdDisabling_ = true;
    const auto MAXIMUM_TIME_TO_WAIT_FOR_NSD_DISABLE = std::chrono::seconds(10);
    proto::QuasarMessage msg;
    msg.mutable_nsd_disable();
    mdnsdConnector_->sendMessage(std::move(msg));
    std::unique_lock<std::mutex> lock(disable_completed_mutex_);
    cond_.wait_for(lock, MAXIMUM_TIME_TO_WAIT_FOR_NSD_DISABLE,
                   [this]() {
                       return nsdDisabled_;
                   });
}

MdnsdMessagerFactory::MdnsdMessagerFactory(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : ipcFactory_(std::move(ipcFactory))
{
}

INsdMessagerPtr MdnsdMessagerFactory::createMessager(INsdReceiverPtr receiver) {
    return std::make_shared<MdnsdNsdMessager>(std::move(receiver), ipcFactory_);
}
