#include "resolves_tracker.h"

#include <yandex_io/libs/logging/logging.h>

using namespace glagol;

bool ResolvesTracker::removeOutdates(std::chrono::seconds outdatingPeriod) {
    const auto now = std::chrono::steady_clock::now();
    auto isOutdated = [&now, &outdatingPeriod](const ResolveExt& res) -> bool {
        return (now - res.updated) > outdatingPeriod && (now - res.lastConnected) > outdatingPeriod;
    };

    const bool wasEmpty = resolves_.empty();

    while (!resolves_.empty() && isOutdated(resolves_.back())) {
        if (resolves_.size() == 1 && firstLocked_) {
            YIO_LOG_DEBUG("Do not outdate locked resolve");
            break;
        }
        auto uriIter = resolves_.back().uriIter;
        auto resolvesIter = uriIter->second;
        if (firstLocked_ && resolvesIter == resolves_.begin()) {
            throw std::runtime_error("ResolvesTracker::removeOutdates structural error! Non first item uri same as first item");
        }
        YIO_LOG_DEBUG("Removing outdated " << uriIter->first << " " << resolvesIter->info.uri());
        if (resolves_.back().info.cluster) {
            --clusterCount_;
            hasCluster_ = clusterCount_ > 0;
        }
        uriToResolve_.erase(uriIter);
        resolves_.erase(resolvesIter);
    }

    return !wasEmpty && resolves_.empty();
}

bool ResolvesTracker::hasResolve() const {
    return !resolves_.empty();
}

bool ResolvesTracker::hasClusterResolve() const {
    return hasCluster_ && hasResolve();
}

ResolveInfo ResolvesTracker::getLockedResolve() {
    if (resolves_.empty()) {
        throw std::runtime_error("Trying to get resolve from empty resolves");
    }
    firstLocked_ = true;
    return resolves_.front().info;
}

ResolveInfo ResolvesTracker::getLastResolve() const {
    if (resolves_.empty()) {
        throw std::runtime_error("Trying to get resolve from empty resolves");
    }
    return resolves_.front().info;
}

void ResolvesTracker::unlock(bool connected) {
    if (!resolves_.empty()) {
        if (connected) {
            resolves_.front().lastConnected = std::chrono::steady_clock::now();
        } else if (resolves_.size() > 1) { // move locked resolve to the end to rotate through resolves
            resolves_.splice(resolves_.end(), resolves_, resolves_.begin());
        }
        firstLocked_ = false;
    }
}

/* return true if this resolve *better* than existing one */
bool ResolvesTracker::addResolve(ResolveInfo resolve, ResolveSource src) {
    auto [iter, inserted] = uriToResolve_.emplace(resolve.uri(), resolves_.end());
    if (src == ResolveSource::MDNS) {
        stereopairRole_ = resolve.stereopairRole;
        tandemRole_ = resolve.tandemRole;
        YIO_LOG_DEBUG("Roles updated to: stereopair = "
                      << quasar::proto::GlagolDiscoveryItem_GroupRole_Name(stereopairRole_)
                      << ", tandem = "
                      << quasar::proto::GlagolDiscoveryItem_GroupRole_Name(tandemRole_));
    }

    if (inserted) {
        if (resolve.cluster) {
            ++clusterCount_;
            hasCluster_ = true;
        };
        auto placeToInsert = resolves_.begin();
        if (firstLocked_) {
            ++placeToInsert;
        }
        iter->second = resolves_.insert(
            placeToInsert,
            ResolveExt{
                .info = std::move(resolve),
                .sourceFlags = 1u << int(src),
                .uriIter = iter,
                .updated = std::chrono::steady_clock::now(),
            });
        return hasCluster_; // FIXME: maybe not better resolve
    } else {
        auto& ext = *(iter->second);
        ext.sourceFlags |= 1u << int(src);
        ext.updated = std::chrono::steady_clock::now();
        if (resolve.cluster && !ext.info.cluster) {
            ext.info.cluster = resolve.cluster;
            ++clusterCount_;
            hasCluster_ = true;
        }
    }

    return false;
}

bool ResolvesTracker::isCluster() const {
    return hasCluster_;
}

glagol::GroupRole ResolvesTracker::tandemRole() const {
    return tandemRole_;
}

glagol::GroupRole ResolvesTracker::stereopairRole() const {
    return stereopairRole_;
}

Json::Value ResolvesTracker::serialize() const {
    Json::Value result(Json::arrayValue);
    for (const auto& res : resolves_) {
        Json::Value item(Json::objectValue);
        item["addr"] = res.info.address;
        item["port"] = res.info.port;
        item["cluster"] = res.info.cluster;
        item["src"] = res.sourceFlags;
        result.append(std::move(item));
    }
    return result;
}

void ResolvesTracker::deserialize(const Json::Value& src) {
    if (!src.isArray()) {
        return;
    }

    std::vector<std::tuple<ResolveInfo, unsigned>> tmp;
    tmp.reserve(src.size());

    unsigned size = src.size();
    for (unsigned i = 0; i < size; ++i) {
        const auto& item = src[i];
        if (item.isMember("addr") && item.isMember("port") && item.isMember("cluster") && item.isMember("src")) {
            ResolveInfo res;
            res.address = item["addr"].asString();
            res.port = item["port"].asInt();
            res.protocol = ResolveInfo::protoByAddress(res.address);
            res.cluster = item["cluster"].asBool();
            const unsigned sourceFlags = item["src"].asUInt();

            if (!res.address.empty() && res.port != 0) {
                tmp.emplace_back(std::move(res), sourceFlags);
            }
        }
    }
    std::reverse(tmp.begin(), tmp.end());

    for (auto [res, sourceFlags] : tmp) {
        addResolve(res, ResolveSource::MDNS);
        resolves_.front().sourceFlags = sourceFlags;
    }
}
