#include "iot_endpoint.h"

#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>

#include <map>
#include <memory>

YIO_DEFINE_LOG_MODULE("iot");

using namespace quasar;

namespace {
    std::string getStateName(const proto::IotState::State& state) {
        switch (state) {
            case proto::IotState::IDLE:
                return "IDLE";
            case proto::IotState::STARTING_DISCOVERY:
                return "STARTING_DISCOVERY";
            case proto::IotState::DISCOVERY_IN_PROGRESS:
                return "DISCOVERY_IN_PROGRESS";
            default:
                return "Unknown";
        }
    }
} // namespace

const std::string IotEndpoint::SERVICE_NAME = "iot";

IotEndpoint::IotEndpoint(std::shared_ptr<YandexIO::IDevice> device,
                         std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                         std::shared_ptr<YandexIO::IIotDiscoveryProvider> discoveryProvider,
                         std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability)
    : device_{std::move(device)}
    , discoveryProvider_{std::move(discoveryProvider)}
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , filePlayerCapability_(std::move(filePlayerCapability))
{
    server_->setMessageHandler([this](const auto& message, auto& /*connection*/) { onQuasarMessage(*message); });
    server_->setClientConnectedHandler(std::bind(&IotEndpoint::onClientConnected, this, std::placeholders::_1));

    const auto firstRunConfig = device_->configuration()->getServiceConfig("firstrund");
    wifiStoragePath_ = getString(firstRunConfig, "wifiStoragePath");

    const auto iotConfig = device_->configuration()->getServiceConfig("iot");
}

IotEndpoint::~IotEndpoint() {
    server_->shutdown(); // close all connections
    cbQueue_.destroy();  // finish all outstanding work
}

void IotEndpoint::start() {
    server_->listenService();
}

void IotEndpoint::onClientConnected(ipc::IServer::IClientConnection& connection) {
    cbQueue_.add([this, connection = connection.share()] {
        proto::QuasarMessage message;
        auto& iotState = *message.mutable_iot_state();
        iotState.set_current_state(state_);

        connection->send(std::move(message));
    });
}

void IotEndpoint::onQuasarMessage(const proto::QuasarMessage& message) {
    if (message.has_iot_request()) {
        const auto& iotRequest = message.iot_request();
        if (iotRequest.has_start_discovery()) {
            cbQueue_.add([this, req = iotRequest.start_discovery()] {
                onStartDiscovery(req);
            });
            return;
        } else if (iotRequest.has_credentials()) {
            cbQueue_.add([this, req = iotRequest.credentials()] {
                onReceivingCredentials(req);
            });
            return;
        } else if (iotRequest.has_stop_discovery()) {
            cbQueue_.add([this, req = iotRequest.stop_discovery()] {
                onStopDiscovery(req);
            });
            return;
        }
    }
    YIO_LOG_WARN("Unknown iot incoming message: " << shortUtf8DebugString(message));
}

void IotEndpoint::onStartDiscovery(const proto::IotRequest::StartDiscovery& message) {
    ensureDiscoveryStopped(proto::IotDiscoveryResult::SUPERCEDED);

    try {
        auto wifiInfo = getWifiInfo();
        currentDiscovery_ = discoveryProvider_->createDiscovery();

        currentDiscoveryInfo_ = CurrentDiscoveryInfo();
        if (message.timeout_ms() > 0) {
            currentDiscoveryInfo_->timeout = std::chrono::milliseconds(message.timeout_ms());
        }
        currentDiscoveryInfo_->deviceType = message.device_type();
        currentDiscoveryInfo_->ssid = message.ssid();
        currentDiscoveryInfo_->password = wifiInfo.wifiPassword;
    } catch (const std::exception& err) {
        std::string reason = err.what();
        Json::Value value;
        value["reason"] = reason;
        device_->telemetry()->reportEvent("iotDiscoveryFailedToStart", jsonToString(value));
        YIO_LOG_ERROR_EVENT("IotEndpoint.FailedStartDiscovery", "Failed to start IoT discovery: " << reason);
        ensureDiscoveryStopped(proto::IotDiscoveryResult::NOT_STARTED);

        // TODO: consider special sound for this case
        filePlayerCapability_->playSoundFile("vins_error.wav", quasar::proto::AudioChannel::DIALOG_CHANNEL);

        return;
    }

    transitionState(proto::IotState::STARTING_DISCOVERY, /* result = */ nullptr);
    installCurrentDiscoveryTimeout(currentDiscoveryInfo_->timeout);
}

void IotEndpoint::onReceivingCredentials(const proto::IotRequest::Credentials& message) {
    if (state_ != proto::IotState::STARTING_DISCOVERY) {
        ensureDiscoveryStopped(proto::IotDiscoveryResult::CREDENTIALS_WITHOUT_DISCOVERY);
        return;
    }

    YandexIO::IIotDiscovery::PairingInfo pairingInfo;
    pairingInfo.wifiSSID = message.ssid();
    pairingInfo.wifiPassword = message.password();
    pairingInfo.token = message.token();
    pairingInfo.cipher = message.cipher();
    currentDiscovery_->start(pairingInfo);

    transitionState(proto::IotState::DISCOVERY_IN_PROGRESS, /* result = */ nullptr);
}

void IotEndpoint::onStopDiscovery(const proto::IotRequest::StopDiscovery& message) {
    ensureDiscoveryStopped(message.result());
}

IotEndpoint::WifiInfo IotEndpoint::getWifiInfo() {
    // FIXME: very bad workaround. We touch wifiStoragePath file, which is also touched by FirstRunEndpoint.
    //  May be touched at the same time, but it's not very likely to happen, so let it be for some time.
    //  How must be fixed: get wifi status from another service (e.g. wifid, when it'll keep password or networkd etc.).
    WifiInfo res;

    if (!fileExists(wifiStoragePath_)) {
        throw std::runtime_error("No wifi info");
    }
    std::string serialized = getFileContent(wifiStoragePath_);
    proto::WifiConnect currentWifi;
    if (!currentWifi.ParseFromString(TString(serialized))) {
        throw std::runtime_error(std::string("Cannot parse saved wifi network from ") + wifiStoragePath_);
    }
    if (!currentWifi.has_wifi_id() || !currentWifi.has_password()) {
        throw std::runtime_error("Parsed wifi doesn't have required fields");
    }
    res.wifiPassword = currentWifi.password();
    res.wifiSSID = currentWifi.wifi_id();
    return res;
}

void IotEndpoint::installCurrentDiscoveryTimeout(std::chrono::milliseconds timeout) {
    // create a weak reference for an expirable timeout
    std::weak_ptr<YandexIO::IIotDiscovery> weakDiscovery = currentDiscovery_;
    cbQueue_.addDelayed([this, weakDiscovery = std::move(weakDiscovery)] {
        auto discovery = weakDiscovery.lock();
        if (!discovery || discovery != currentDiscovery_) {
            // this discovery has already been destroyed
            return;
        }

        onDiscoveryTimeout();
    }, timeout);
}

void IotEndpoint::onDiscoveryTimeout() {
    device_->telemetry()->reportEvent("iotDiscoveryTimeout");

    ensureDiscoveryStopped(proto::IotDiscoveryResult::TIMEOUT);
}

void IotEndpoint::ensureDiscoveryStopped(const proto::IotDiscoveryResult::ResultCode resultCode) {
    proto::IotDiscoveryResult result;
    result.set_code(resultCode);
    ensureDiscoveryStopped(result);
}

void IotEndpoint::ensureDiscoveryStopped(const proto::IotDiscoveryResult& reason) {
    if (currentDiscovery_) {
        currentDiscovery_->stop();
        currentDiscovery_.reset();
    }

    transitionState(proto::IotState::IDLE, &reason);

    currentDiscoveryInfo_.reset();
}

void IotEndpoint::transitionState(proto::IotState::State newState, const proto::IotDiscoveryResult* result) {
    if (state_ == newState) {
        return;
    }

    YIO_LOG_INFO("Transition IOT state: from " << getStateName(state_) << " to " << getStateName(newState));

    // exit old state
    if (state_ == proto::IotState::STARTING_DISCOVERY) {
        device_->telemetry()->reportLatency(std::move(prepareLatencyPoint_), "iotDiscoveryPrepare");
    } else if (state_ == proto::IotState::DISCOVERY_IN_PROGRESS) {
        device_->telemetry()->reportLatency(std::move(waitLatencyPoint_), "iotDiscoveryStop");
    }

    // enter new state
    if (newState == proto::IotState::STARTING_DISCOVERY) {
        device_->telemetry()->reportEvent("iotDiscoveryStart");
        prepareLatencyPoint_ = device_->telemetry()->createLatencyPoint();
    } else if (newState == proto::IotState::DISCOVERY_IN_PROGRESS) {
        waitLatencyPoint_ = device_->telemetry()->createLatencyPoint();
    }

    // broadcast state transition
    proto::QuasarMessage message;
    auto& iotState = *message.mutable_iot_state();
    iotState.set_current_state(newState);
    iotState.set_prev_state(state_);
    if (result) {
        iotState.mutable_result()->CopyFrom(*result);
    }
    if (currentDiscoveryInfo_) {
        auto& discoveryInfo = *iotState.mutable_current_discovery_info();

        discoveryInfo.set_device_type(TString(currentDiscoveryInfo_->deviceType));
        discoveryInfo.set_ssid(TString(currentDiscoveryInfo_->ssid));
        discoveryInfo.set_password(TString(currentDiscoveryInfo_->password));
        discoveryInfo.set_timeout_ms(currentDiscoveryInfo_->timeout.count());
    }
    server_->sendToAll(std::move(message));

    state_ = newState;
}
