#pragma once

#include <yandex_io/capabilities/file_player/interfaces/i_file_player_capability.h>
#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/iot/i_iot_discovery.h>
#include <yandex_io/libs/iot/i_iot_discovery_provider.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <chrono>
#include <memory>
#include <optional>

namespace quasar {

    class IotEndpoint {
    public:
        static const std::string SERVICE_NAME;

        IotEndpoint(std::shared_ptr<YandexIO::IDevice> device,
                    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                    std::shared_ptr<YandexIO::IIotDiscoveryProvider> discoveryProvider,
                    std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability);
        ~IotEndpoint();

        void start();

    private:
        // Called from off-thread callbacks
        void onClientConnected(ipc::IServer::IClientConnection& connection);
        void onQuasarMessage(const proto::QuasarMessage& message);

        // All methods below are only called from event loop thread
        void onStartDiscovery(const proto::IotRequest::StartDiscovery& message);
        void onReceivingCredentials(const proto::IotRequest::Credentials& message);
        void onStopDiscovery(const proto::IotRequest::StopDiscovery& message);
        void onDiscoveryTimeout();

        void installCurrentDiscoveryTimeout(std::chrono::milliseconds timeout);
        void ensureDiscoveryStopped(const proto::IotDiscoveryResult& reason);
        void ensureDiscoveryStopped(proto::IotDiscoveryResult::ResultCode resultCode);
        void transitionState(proto::IotState::State newState, const proto::IotDiscoveryResult* result);

        struct WifiInfo {
            std::string wifiSSID;
            std::string wifiPassword;
        };

        WifiInfo getWifiInfo();

    private:
        // dependencies
        std::shared_ptr<YandexIO::IDevice> device_;
        std::shared_ptr<YandexIO::IIotDiscoveryProvider> discoveryProvider_;

        // lifecycle machinery
        std::shared_ptr<ipc::IServer> server_;
        const std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability_;
        NamedCallbackQueue cbQueue_{"IotEndpoint"};

        struct CurrentDiscoveryInfo {
            std::chrono::milliseconds timeout{std::chrono::minutes(1)};
            std::string deviceType;
            std::string ssid;
            std::string password;
        };

        // current state
        // NOTE: all access to current state must be serialized via `cbQueue_`
        std::shared_ptr<YandexIO::IIotDiscovery> currentDiscovery_;
        std::optional<CurrentDiscoveryInfo> currentDiscoveryInfo_;
        proto::IotState::State state_ = proto::IotState::IDLE;

        std::shared_ptr<const YandexIO::LatencyData> waitLatencyPoint_;
        std::shared_ptr<const YandexIO::LatencyData> prepareLatencyPoint_;

        std::string wifiStoragePath_;
    };

} // namespace quasar
