#include "network_endpoint.h"

#include <yandex_io/libs/logging/logging.h>

#include <chrono>

using namespace quasar;

const std::string NetworkEndpoint::SERVICE_NAME = "networkd";

NetworkEndpoint::NetworkEndpoint(std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , wifidConnector_(ipcFactory->createIpcConnector("wifid"))
{
    YIO_LOG_INFO("NetworkEndpoint started");

    wifidConnector_->setMessageHandler(std::bind(&NetworkEndpoint::handleWifiMessage, this, std::placeholders::_1));

    server_->setMessageHandler(std::bind(&NetworkEndpoint::processQuasarMessage, this, std::placeholders::_1, std::placeholders::_2));
    server_->setClientConnectedHandler([this](auto& connection)
                                       {
                                           std::lock_guard<std::mutex> lock(mutex_);
                                           if (currentNetworkStatus_.has_status()) {
                                               proto::QuasarMessage quasarMessage;
                                               *quasarMessage.mutable_network_status() = currentNetworkStatus_;
                                               connection.send(std::move(quasarMessage));
                                           }
                                       });

    wifidConnector_->connectToService();
    server_->listenService();
}

NetworkEndpoint::~NetworkEndpoint()
{
    server_->shutdown();
    wifidConnector_->shutdown();
}

void NetworkEndpoint::handleWifiMessage(const ipc::SharedMessage& message)
{
    if (message->has_wifi_status()) {
        proto::QuasarMessage networkMessage;
        {
            std::lock_guard<std::mutex> lock(mutex_);
            *currentNetworkStatus_.mutable_wifi_status() = message->wifi_status();
            currentNetworkStatus_.set_status(proto::NetworkStatus_Status(message->wifi_status().status()));
            currentNetworkStatus_.set_type(proto::CONNECTION_TYPE_WIFI);
            *networkMessage.mutable_network_status() = currentNetworkStatus_;
        }
        server_->sendToAll(std::move(networkMessage));
    }
}

void NetworkEndpoint::processQuasarMessage(const ipc::SharedMessage& message,
                                           ipc::IServer::IClientConnection& connection)
{
    if (message->has_reset_network()) {
        proto::QuasarMessage wifidRequest;
        wifidRequest.mutable_reset_wifi();
        proxyCall(message->request_id(), std::move(wifidRequest), connection);
    }

    if (message->has_network_config_reload()) {
        proto::QuasarMessage wifidRequest;
        wifidRequest.mutable_wifi_config_reload();
        proxyCall(message->request_id(), std::move(wifidRequest), connection);
    }

    if (message->has_networks_disable()) {
        proto::QuasarMessage wifidRequest;
        wifidRequest.mutable_wifi_networks_disable();
        proxyCall(message->request_id(), std::move(wifidRequest), connection);
    }

    if (message->has_networks_enable()) {
        proto::QuasarMessage wifidRequest;
        wifidRequest.mutable_wifi_networks_enable();
        proxyCall(message->request_id(), std::move(wifidRequest), connection);
    }
}

void NetworkEndpoint::proxyCall(const std::string& originalRequestId,
                                proto::QuasarMessage&& wifidRequest,
                                ipc::IServer::IClientConnection& connection) {
    auto rsp = wifidConnector_->sendRequestSync(std::move(wifidRequest), std::chrono::seconds(10));
    auto wifidResponse = *rsp;
    wifidResponse.set_request_id(TString(originalRequestId));
    connection.send(std::move(wifidResponse));
}

void NetworkEndpoint::waitUntilConnected() {
    wifidConnector_->waitUntilConnected();
}
