#include "notification_endpoint.h"

#include <yandex_io/capabilities/file_player/play_sound_file_listener.h>

#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <memory>

using namespace quasar;
using namespace proto;

struct NotificationEndpoint::Settings {
    const bool enableSyncVolume;
    const std::chrono::milliseconds syncVolumeTimeout;
    Settings(bool enableSyncVolume_, std::chrono::milliseconds syncVolumeTimeout_)
        : enableSyncVolume(enableSyncVolume_)
        , syncVolumeTimeout(syncVolumeTimeout_)
    {
    }
};

NotificationEndpoint::NotificationEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IStereoPairProvider> stereoPairProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider,
    std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , stereoPairProvider_(std::move(stereoPairProvider))
    , userConfigProvider_(std::move(userConfigProvider))
    , filePlayerCapability_(std::move(filePlayerCapability))
    , deviceContext_(std::make_shared<YandexIO::DeviceContext>(ipcFactory))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , toDoNotDisturb_(ipcFactory->createIpcConnector("do_not_disturb"))
    , asyncQueue_(std::make_shared<NamedCallbackQueue>("NotificationEndpoint"))
    , playSoundFileListener_(std::make_shared<YandexIO::PlaySoundFileListener>(nullptr, makeSafeCallback([this]() {
                                                                                   onNotificationSoundCompleted();
                                                                               }, lifetime_, asyncQueue_)))
{
    const auto config = device_->configuration()->getServiceConfig("notificationd");
    enableSyncVolume_ = tryGetBool(config, "enable_sync_with_maind", true);
    syncVolumeTimeout_ = std::chrono::milliseconds(tryGetInt64(config, "sync_timeout_ms", 500));
    defaultSettings_ = std::make_unique<Settings>(enableSyncVolume_, syncVolumeTimeout_);

    server_->setMessageHandler([this](const auto& message, auto& /*connection*/) {
        asyncQueue_->add([this, message]() {
            handleQuasarMessage(message);
        });
    });
    server_->setClientConnectedHandler([this](auto& connection) {
        asyncQueue_->add([this, connection = connection.share()]() {
            QuasarMessage msg;
            msg.mutable_notification_update_event()->set_count(notificationsCount_);
            msg.mutable_notification_update_event()->set_notification_mode(notificationState_);
            connection->send(std::move(msg));
        }, lifetime_);
    });

    toDoNotDisturb_->setMessageHandler([this](const auto& message) {
        asyncQueue_->add([this, message]() {
            handleDoNotDisturbMessage(message);
        });
    });

    deviceContext_->onPreparedForNotification =
        makeSafeCallback([this]() {
            if (pendingNotification_) {
                YIO_LOG_INFO("Maind is prepared for notification, play notification sound");
                handleNotifyDirective(pendingNotification_->notifyDirective);
                pendingNotification_.reset(nullptr);
            }
        }, lifetime_, asyncQueue_);

    server_->listenService();
    toDoNotDisturb_->connectToService();

    userConfigProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "notification_prefs").connect([this](const auto& json) {
        onConfigChanged(*json);
    }, lifetime_, asyncQueue_);

    authProvider_->ownerAuthInfo().connect(
        [this, uid = std::string()](const auto& authInfo) mutable {
            if (uid != authInfo->passportUid) {
                uid = authInfo->passportUid;
                handleAccountChange();
            }
        }, lifetime_, asyncQueue_);

    if (stereoPairProvider_) {
        stereoPairProvider_->stereoPairState().connect(
            [this](const auto& state) {
                updateStereoPairFollower(state->isFollower());
            }, lifetime_, asyncQueue_);
    }
}

NotificationEndpoint::~NotificationEndpoint() {
    lifetime_.die();
    asyncQueue_->destroy();

    server_->shutdown();
    toDoNotDisturb_->shutdown();
    deviceContext_->shutdown();
}

void NotificationEndpoint::handleQuasarMessage(const ipc::SharedMessage& message) {
    if (message->has_notify_directive()) {
        processNotifyDirective(message->notify_directive());
    }
}

void NotificationEndpoint::processNotifyDirective(const proto::NotifyDirective& notifyDirective) {
    if (!notifyDirective.has_versionid()) {
        return;
    }

    int64_t versionId = std::stoll(notifyDirective.versionid());
    if (versionId <= versionId_) {
        YIO_LOG_INFO("Notification version is outdated");
        return;
    }
    if (!notificationsEnabled_) {
        YIO_LOG_INFO("Notifications are disabled");
        return;
    }

    if (isStereoPairFollower_) {
        YIO_LOG_INFO("Ignore notification due stereo pair follower");
        return;
    }

    sendNotificationUpdateEvent(notifyDirective);
    versionId_ = versionId;
    // we only need sync with VolumeManager if notification should play sound
    if (enableSyncVolume_ && shouldPlaySound(notifyDirective)) {
        auto curTime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch());
        pendingNotification_ = std::make_unique<PendingNotification>(notifyDirective, curTime);
        deviceContext_->fireNotificationPending();
        // if maind doesn't reply to request, we still want to play directive after syncVolumeTimeout_ ms
        asyncQueue_->addDelayed([this, curTime]() {
            if (pendingNotification_ && pendingNotification_->requestTimestamp == curTime) {
                YIO_LOG_INFO("Maind is not prepared for notification, play it anyway");
                handleNotifyDirective(pendingNotification_->notifyDirective);
                pendingNotification_.reset(nullptr);
            }
        }, syncVolumeTimeout_, lifetime_);
    } else {
        handleNotifyDirective(notifyDirective);
    }
}

void NotificationEndpoint::updateStereoPairFollower(bool isStereoPairFollower) {
    if (isStereoPairFollower_ == isStereoPairFollower) {
        return;
    }
    isStereoPairFollower_ = isStereoPairFollower;

    if (isStereoPairFollower_) {
        YIO_LOG_INFO("NotificationEndpoint disabled any notifications due stereo pair follower");
        pendingNotification_.reset(nullptr);
        notificationsCount_ = 0;
        notificationState_ = NotificationUpdateEvent_NotificationMode_NONE;
        QuasarMessage msg;
        msg.mutable_notification_update_event()->set_count(notificationsCount_);
        msg.mutable_notification_update_event()->set_notification_mode(notificationState_);
        server_->sendToAll(std::move(msg));
    } else {
        YIO_LOG_INFO("NotificationEndpoint enable all notifications");
    }
}

void NotificationEndpoint::handleNotifyDirective(const proto::NotifyDirective& notifyDirective)
{
    if (isStereoPairFollower_) {
        return;
    }

    if (shouldPlaySound(notifyDirective)) {
        playNotificationSound();
    }
    onUpdateNotifications(notifyDirective.notifications_size(), notifyDirective.ring());
}

void NotificationEndpoint::onUpdateNotifications(int count, NotifyDirective_RingType ringType) {
    notificationsCount_ = count;
    if (isNotificationActiveState(ringType, count, doNotDisturb_)) {
        notificationState_ = NotificationUpdateEvent_NotificationMode_ACTIVE;
    } else {
        notificationState_ = NotificationUpdateEvent_NotificationMode_NONE;
    }

    asyncQueue_->addDelayed([this]() {
        if (notificationState_ == NotificationUpdateEvent_NotificationMode_NONE) {
            return;
        }
        notificationState_ = NotificationUpdateEvent_NotificationMode_PASSIVE;
        QuasarMessage msg;
        msg.mutable_notification_update_event()->set_count(notificationsCount_);
        msg.mutable_notification_update_event()->set_notification_mode(notificationState_);
        server_->sendToAll(std::move(msg));
    }, TO_PASSIVE_MODE_MILLISECONDS, lifetime_);

    QuasarMessage msg;
    msg.mutable_notification_update_event()->set_count(notificationsCount_);
    msg.mutable_notification_update_event()->set_notification_mode(notificationState_);
    server_->sendToAll(std::move(msg));
}

bool NotificationEndpoint::shouldPlaySound(const proto::NotifyDirective& notifyDirective) const {
    if (isStereoPairFollower_) {
        return false;
    }
    if (!notifyDirective.has_ring()) {
        return false;
    }
    if (notifyDirective.ring() == NotifyDirective_RingType_PROACTIVE) {
        return true;
    }
    return !doNotDisturb_ && notifyDirective.ring() != NotifyDirective_RingType_NO_SOUND;
}

bool NotificationEndpoint::isNotificationActiveState(NotifyDirective_RingType ringType, int notificationCount, bool doNotDisturb) {
    return (!doNotDisturb || ringType == NotifyDirective_RingType_PROACTIVE) && notificationCount > 0;
}

std::string NotificationEndpoint::getRingTypeStr(NotifyDirective_RingType ringType) {
    /* NOTE: This array should be always synchronized with it's protobuf enum projection */
    static_assert(proto::NotifyDirective::RingType_ARRAYSIZE == ringTypeMap_.size(), "Missed RingType Enum value to string");

    return ringTypeMap_[ringType];
}

void NotificationEndpoint::playNotificationSound() {
    device_->telemetry()->reportEvent("notificationSoundPlayed");

    deviceContext_->fireNotificationStarted();
    filePlayerCapability_->playSoundFile("notification_sound.mp3", std::nullopt, std::nullopt, playSoundFileListener_);
}

void NotificationEndpoint::onNotificationSoundCompleted() {
    deviceContext_->fireNotificationEnd();
}

void NotificationEndpoint::sendNotificationUpdateEvent(const NotifyDirective& notifyDirective) {
    Json::Value eventBody;
    eventBody["dnd_enabled"] = doNotDisturb_;
    eventBody["new_count"] = notifyDirective.notifications_size();
    eventBody["old_count"] = notificationsCount_;
    eventBody["ring_type"] = getRingTypeStr(notifyDirective.ring());
    eventBody["should_play_sound"] = shouldPlaySound(notifyDirective);
    eventBody["notification_state"] = isNotificationActiveState(notifyDirective.ring(), notifyDirective.notifications_size(), doNotDisturb_) ? "active" : "none";

    device_->telemetry()->reportEvent("NotificationUpdateReceived", jsonToString(eventBody));
}

void NotificationEndpoint::setDoNotDisturbMode(bool doNotDisturb) {
    if (doNotDisturb_ == doNotDisturb) {
        return;
    }

    doNotDisturb_ = doNotDisturb;
    updateNotificationMode();
}

void NotificationEndpoint::onConfigChanged(const Json::Value& notificationPrefsConfig) {
    enableSyncVolume_ = tryGetBool(notificationPrefsConfig, "enable_sync_with_maind", defaultSettings_->enableSyncVolume);
    syncVolumeTimeout_ = std::chrono::milliseconds(tryGetInt64(notificationPrefsConfig, "sync_timeout_ms", defaultSettings_->syncVolumeTimeout.count()));
    bool notificationsEnabledState = tryGetBool(notificationPrefsConfig, "notifications_enabled", true);
    if (notificationsEnabled_ && !notificationsEnabledState) {
        notificationsCount_ = 0;
        notificationState_ = NotificationUpdateEvent_NotificationMode_NONE;
        QuasarMessage quasarMessage;
        quasarMessage.mutable_notification_update_event()->set_count(notificationsCount_);
        quasarMessage.mutable_notification_update_event()->set_notification_mode(notificationState_);
        server_->sendToAll(std::move(quasarMessage));
        notificationsEnabled_ = false;
        YIO_LOG_INFO("Disable notifications");
        versionId_ = -1;
    } else if (!notificationsEnabled_ && notificationsEnabledState) {
        notificationsEnabled_ = true;
        YIO_LOG_INFO("Enable notifications");
    }
}

void NotificationEndpoint::handleDoNotDisturbMessage(const ipc::SharedMessage& message) {
    if (message->has_do_not_disturb_event() && message->do_not_disturb_event().has_is_dnd_enabled()) {
        setDoNotDisturbMode(message->do_not_disturb_event().is_dnd_enabled());
    }
}

void NotificationEndpoint::handleAccountChange() {
    versionId_ = DEFAULT_VERSION_ID;
    notificationsCount_ = 0;
    notificationState_ = NotificationUpdateEvent_NotificationMode_NONE;
    QuasarMessage msg;
    msg.mutable_notification_update_event()->set_count(notificationsCount_);
    msg.mutable_notification_update_event()->set_notification_mode(notificationState_);
    server_->sendToAll(std::move(msg));
}

void NotificationEndpoint::updateNotificationMode() {
    if (notificationsCount_ <= 0) {
        return;
    }

    if (doNotDisturb_) {
        notificationState_ = NotificationUpdateEvent_NotificationMode_NONE;
    } else {
        notificationState_ = NotificationUpdateEvent_NotificationMode_PASSIVE;
    }

    QuasarMessage msg;
    msg.mutable_notification_update_event()->set_count(notificationsCount_);
    msg.mutable_notification_update_event()->set_notification_mode(notificationState_);
    server_->sendToAll(std::move(msg));
}
