#include "ntp_endpoint.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <memory>
#include <random>

YIO_DEFINE_LOG_MODULE("ntp");

using namespace quasar;
using namespace quasar::proto;

namespace {
    const char* LAST_CUSTOM_NTPD_CONFIG_FILE = "/data/quasar/data/ntpd_custom_config";
    std::string generateSid() noexcept {
        return rndMark(6);
    }
} // namespace

const std::string NtpEndpoint::SERVICE_NAME = "ntpd";

NtpEndpoint::NtpEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IUserConfigProvider> userConfigProvider)
    : server_(ipcFactory->createIpcServer(SERVICE_NAME))
{
    server_->setClientConnectedHandler(makeSafeCallback(
        [this](auto& connection) {
            if (hasNtpSync_.load()) {
                QuasarMessage message;
                message.mutable_ntp_sync_event()->set_is_ntp_sync_successful(isNtpSyncSuccessful_.load());
                connection.send(std::move(message));
            }
        }, lifetime_));
    server_->listenService();

    const auto& deviceNtpdConfig = device->configuration()->getServiceConfig("ntpd");
    const auto& customNtpdConfig = tryGetConfigFromFile(LAST_CUSTOM_NTPD_CONFIG_FILE, Json::Value{});
    ntpSync_ = std::make_unique<NtpSync>(device, deviceNtpdConfig, customNtpdConfig, generateSid());
    ntpSync_->setOnSyncStatusChanged(makeSafeCallback([this](bool isSyncSuccessful) {
        isNtpSyncSuccessful_ = isSyncSuccessful;
        hasNtpSync_ = true;
        QuasarMessage message;
        message.mutable_ntp_sync_event()->set_is_ntp_sync_successful(isSyncSuccessful);
        YIO_LOG_INFO("Ntp sync broadcast: " << message);
        server_->sendToAll(std::move(message));
    }, lifetime_));

    userConfigProvider->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "ntpd").connect([this](auto pJson) {
        if (pJson->isNull()) {
            std::remove(LAST_CUSTOM_NTPD_CONFIG_FILE);
            ntpSync_->reloadDefaultConfig();
        } else {
            ntpSync_->reloadConfig(*pJson);
            try {
                TransactionFile file(LAST_CUSTOM_NTPD_CONFIG_FILE);
                file.write(jsonToString(*pJson));
                file.commit();
            } catch (const std::exception& ex) {
                YIO_LOG_ERROR_EVENT("NtpEndpoint.FailedSaveConfig", "Write new ntpd config to " << LAST_CUSTOM_NTPD_CONFIG_FILE << " failed with exception: " << ex.what());
            }
        }
    }, lifetime_);

    ntpSync_->start();
}

NtpEndpoint::~NtpEndpoint()
{
    lifetime_.die();
}
