#include "ntp_sync.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/ntp/time_over_http.h>
#include <yandex_io/libs/ping/gateway_monitor.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <boost/lexical_cast.hpp>
#include <boost/optional.hpp>

#include <build/scripts/c_templates/svnversion.h>

#include <ctime>
#include <fstream>
#include <memory>
#include <random>
#include <sstream>
#include <thread>

#include <sys/time.h>
#include <sys/timex.h>

YIO_DEFINE_LOG_MODULE("ntp");

using namespace quasar;
using namespace quasar::proto;

static_assert(NtpSync::SyncState::UNDEFINED < NtpSync::SyncState::MONOTONIC, "Invalid enum order");
static_assert(NtpSync::SyncState::MONOTONIC < NtpSync::SyncState::OUT_OF_SYNC, "Invalid enum order");
static_assert(NtpSync::SyncState::OFFLINE_BORDER < NtpSync::SyncState::OUT_OF_SYNC, "Invalid enum order");
static_assert(NtpSync::SyncState::OUT_OF_SYNC < NtpSync::SyncState::SUCCESS_SYNC, "Invalid enum order");
static_assert(NtpSync::SyncState::SUCCESS_SYNC <= NtpSync::SyncState::HTTP_SYNC, "Invalid enum order");
static_assert(NtpSync::SyncState::HTTP_SYNC < NtpSync::SyncState::EXPIRED_SYNC, "Invalid enum order");
static_assert(NtpSync::SyncState::EXPIRED_SYNC < NtpSync::SyncState::SYNC, "Invalid enum order");

namespace {
    constexpr std::chrono::minutes flushErrorPeriod{5};
    constexpr uint32_t maxIpCache = 12;
    constexpr uint32_t minUpdatePeriodSec = 15;
    constexpr uint32_t minFluctuationMs = 100;
    const NtpSync::Config defaultConfig = [] {
        NtpSync::Config config;
        config.syncParams.timeout = std::chrono::milliseconds{100};
        config.syncParams.sufficientMeasuringCount = 20;
        config.syncParams.maxMeasuringCount = 50;
        config.syncParams.ntpServers.emplace_back("0.ru.pool.ntp.org:123");
        config.syncParams.ntpServers.emplace_back("1.ru.pool.ntp.org:123");
        config.syncParams.ntpServers.emplace_back("2.ru.pool.ntp.org:123");
        config.syncParams.ntpServers.emplace_back("3.ru.pool.ntp.org:123");
        config.syncParams.ntpServers.emplace_back("europe.pool.ntp.org:123");
        return config;
    }();

    template <class T>
    int settimeOfDay(T timePoint) {
        auto timestamp = std::chrono::duration_cast<std::chrono::microseconds>(timePoint.time_since_epoch()).count();
        struct timeval tv {
            static_cast<time_t>(timestamp / 1000000), static_cast<suseconds_t>(timestamp % 1000000)
        };
        return ::settimeofday(&tv, nullptr);
    }

    const char* syncStateText(NtpSync::SyncState syncState)
    {
        switch (syncState) {
            case NtpSync::SyncState::UNDEFINED:
                return "UNDEFINED";
            case NtpSync::SyncState::MONOTONIC:
                return "MONOTONIC";
            case NtpSync::SyncState::OUT_OF_SYNC:
                return "OUT_OF_SYNC";
            case NtpSync::SyncState::HTTP_SYNC:
                return "HTTP_SYNC";
            case NtpSync::SyncState::EXPIRED_SYNC:
                return "EXPIRED_SYNC";
            case NtpSync::SyncState::SYNC:
                return "SYNC";
        }
        return "undefined";
    }

    const char* routerNtpText(NtpSync::RouterNtp routerNtp)
    {
        switch (routerNtp) {
            case NtpSync::RouterNtp::UNDEFINED:
                return "UNDEFINED";
            case NtpSync::RouterNtp::UNAVAILABLE:
                return "UNAVAILABLE";
            case NtpSync::RouterNtp::AVAILABLE:
                return "AVAILABLE";
        }
        return "undefined";
    }

} // namespace

bool NtpSync::Config::operator==(const Config& other) const {
    return syncParams == other.syncParams &&
           ipCacheFile == other.ipCacheFile &&
           monotonicClockFile == other.monotonicClockFile &&
           syncCheckPeriod == other.syncCheckPeriod &&
           timeOverHttpDelay == other.timeOverHttpDelay &&
           syncFluctuation == other.syncFluctuation &&
           syncEnabled == other.syncEnabled &&
           tryRouterNtpServer == other.tryRouterNtpServer &&
           syncEnabledMarkerFile == other.syncEnabledMarkerFile &&
           timeOverHttpUrl == other.timeOverHttpUrl;
}

bool NtpSync::Config::operator!=(const Config& other) const {
    return !(*this == other);
}

NtpSync::NtpSync(std::shared_ptr<YandexIO::IDevice> device, const Json::Value& deviceNtpdConfig, const Json::Value& customNtpdConfig, std::string sid)
    : device_(std::move(device))
    , deviceConfig_(parseConfig(defaultConfig, deviceNtpdConfig))
    , sid_(std::move(sid))
    , startTime_(std::chrono::steady_clock::now())
    , config_(deviceConfig_)
{
    onSyncStateChangedFunc_ = [](auto /*unused*/) {};

    if (!customNtpdConfig.empty()) {
        try {
            YIO_LOG_DEBUG("Load custom ntpd config: " << jsonToString(customNtpdConfig));
            config_ = parseConfig(deviceConfig_, customNtpdConfig);
        } catch (const std::exception& ex) {
            YIO_LOG_ERROR_EVENT("NtpSync.FailedParseConfig", "Fail to parse custom config");
        }
    }
    auto lastSystemTime = loadMonotonicClock(config_.monotonicClockFile);
    if (config_.syncEnabled && lastSystemTime) {
        auto currentTime = std::chrono::system_clock::now();
        if (*lastSystemTime > currentTime) {
            std::time_t timestamp = std::chrono::system_clock::to_time_t(*lastSystemTime);
            auto errorCode = settimeOfDay(*lastSystemTime);
            if (errorCode == -1) {
                auto e = errno;
                YIO_LOG_ERROR_EVENT("NtpSync.FailedSetSystemTime.Error", "Fail change system time to \"" << trim(std::ctime(&timestamp)) << "\", errno=" << e);
                if (e == EPERM) {
                    insufficientPrivilegeFlag_ = true;
                }
            } else {
                currentTime = std::chrono::system_clock::now();
                if (*lastSystemTime <= currentTime)
                {
                    syncState_ = SyncState::MONOTONIC;
                    YIO_LOG_INFO("NtpSync: Set monotonic time \"" << trim(std::ctime(&timestamp)) << "\"");
                } else {
                    std::time_t timestamp = std::chrono::system_clock::to_time_t(*lastSystemTime);
                    YIO_LOG_ERROR_EVENT("NtpSync.FailedSetSystemTime.Unchanged", "Fail change system time to \"" << trim(std::ctime(&timestamp)) << "\" by unknown reason");
                }
            }
        }
    }

    if (lastSystemTime && *lastSystemTime <= std::chrono::system_clock::now()) {
        syncState_ = SyncState::MONOTONIC;
    }

    createOrRemoveSyncEnabledMarkerFile(config_.syncEnabled, config_.syncEnabledMarkerFile);
    if (config_.syncEnabled) {
        flushMonotonicClock(std::chrono::system_clock::now(), config_.monotonicClockFile);
    }
    reportEvent(false, std::nullopt, false);
    logSyncState();
    reportExecutor_ = std::make_unique<PeriodicExecutor>(
        [this] { reportEvent(false, std::nullopt, false); },
        std::chrono::minutes{1},
        PeriodicExecutor::PeriodicType::SLEEP_FIRST);
}

NtpSync::~NtpSync()
{
    stopThread();
}

bool NtpSync::start() noexcept {
    try {
        return startThread();
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedStartThread", "NtpSync start failed: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedStartThread", "NtpSync start failed with unexpected exception");
    }
    return false;
}

bool NtpSync::stop() noexcept {
    try {
        return stopThread();
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedStopThread", "NtpSync stop failed: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedStopThread", "NtpSync stop failed with unexpected exception");
    }
    return false;
}

bool NtpSync::reloadConfig(const Json::Value& customNtpdConfig) noexcept {
    try {
        return reloadConfig(parseConfig(deviceConfig_, customNtpdConfig));
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedReloadOrParseConfig", "NtpSync: Reload config failed: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedReloadOrParseConfig", "NtpSync: Reload config failed with unexpected exception");
    }
    return false;
}

void NtpSync::reloadDefaultConfig() noexcept {
    reloadConfig(deviceConfig_);
}

void NtpSync::setOnSyncStatusChanged(std::function<void(bool isSyncSuccessful)> func) noexcept {
    onSyncStateChangedFunc_ = std::move(func);
}

void NtpSync::syncNow() noexcept {
    std::unique_lock<std::mutex> lock1(periodicExecutorMutex_);
    if (periodicExecutor_) {
        YIO_LOG_INFO("NtpSync: Immediately run ntp synchronization");
        periodicExecutor_->executeNow();
    }
}

NtpSync::Config NtpSync::parseConfig(const Config& deviceConfig, const Json::Value& ntpdConfig)
{
    Config config = deviceConfig;

    config.ipCacheFile = tryGetString(ntpdConfig, "ipCacheFile", config.ipCacheFile);
    config.monotonicClockFile = tryGetString(ntpdConfig, "monotonicClockFile", config.monotonicClockFile);
    config.syncInitialPeriod = std::chrono::seconds{tryGetUInt64(ntpdConfig, "syncInitialPeriodSec", config.syncInitialPeriod.count())};
    config.syncCheckPeriod = std::chrono::seconds{tryGetUInt64(ntpdConfig, "syncCheckPeriodSec", config.syncCheckPeriod.count())};
    config.timeOverHttpDelay = std::chrono::seconds{tryGetUInt64(ntpdConfig, "timeOverHttpDelaySec", config.timeOverHttpDelay.count())};
    config.syncFluctuation = std::chrono::milliseconds{tryGetUInt64(ntpdConfig, "syncFluctuationMSec", config.syncFluctuation.count())};
    config.syncEnabled = tryGetBool(ntpdConfig, "syncEnabled", config.syncEnabled);
    config.tryRouterNtpServer = tryGetBool(ntpdConfig, "tryRouterNtpServer", config.tryRouterNtpServer);
    config.syncEnabledMarkerFile = tryGetString(ntpdConfig, "syncEnabledMarkerFile", config.syncEnabledMarkerFile);
    config.timeOverHttpUrl = tryGetString(ntpdConfig, "timeOverHttpUrl", config.timeOverHttpUrl);

    const char* fSyncParams = "syncParams";
    if (ntpdConfig.isMember(fSyncParams) && ntpdConfig[fSyncParams].isObject()) {
        config.syncParams.tryParseJson(ntpdConfig[fSyncParams]);
    }

    config.syncCheckPeriod = std::max(config.syncCheckPeriod, std::chrono::seconds{minUpdatePeriodSec});
    config.syncFluctuation = std::chrono::milliseconds{std::min<int64_t>(std::max<int64_t>(std::abs(config.syncFluctuation.count()), minFluctuationMs), 0xFFFFFFFFll)};

    return config;
}

Json::Value NtpSync::configToJson(const Config& ntpdConfig)
{
    Json::Value json;
    json["ipCacheFile"] = ntpdConfig.ipCacheFile;
    json["monotonicClockFile"] = ntpdConfig.monotonicClockFile;
    json["syncInitialPeriodSec"] = static_cast<int64_t>(ntpdConfig.syncInitialPeriod.count());
    json["syncCheckPeriodSec"] = static_cast<int64_t>(ntpdConfig.syncCheckPeriod.count());
    json["timeOverHttpDelaySec"] = static_cast<int64_t>(ntpdConfig.timeOverHttpDelay.count());
    json["syncFluctuationMSec"] = static_cast<int64_t>(ntpdConfig.syncFluctuation.count());
    json["syncEnabled"] = ntpdConfig.syncEnabled;
    json["tryRouterNtpServer"] = ntpdConfig.tryRouterNtpServer;
    json["syncEnabledMarkerFile"] = ntpdConfig.syncEnabledMarkerFile;
    json["timeOverHttpUrl"] = ntpdConfig.timeOverHttpUrl;
    json["syncParams"] = ntpdConfig.syncParams.toJson();
    return json;
}

bool NtpSync::reloadConfig(Config newConfig) noexcept {
    try {
        {
            std::unique_lock<std::mutex> lock(configMutex_);
            if (newConfig == config_) {
                YIO_LOG_DEBUG("NtpSync: reload config - no changes");
                return true;
            }
        }

        YIO_LOG_INFO("NtpSync: reloading config: " << jsonToString(configToJson(newConfig)));
        bool threadStopped = stopThread();
        {
            std::unique_lock<std::mutex> lock(configMutex_);
            config_ = std::move(newConfig);
            createOrRemoveSyncEnabledMarkerFile(config_.syncEnabled, config_.syncEnabledMarkerFile);
        }
        if (threadStopped) {
            startThread();
        }
        return true;
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedReloadConfig", "NtpSync: Reload config failed: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedReloadConfig", "NtpSync: Reload config failed with unexpected exception");
    }
    return false;
}

bool NtpSync::startThread()
{
    YIO_LOG_DEBUG("NtpSync: starting worker thread");
    std::unique_lock<std::mutex> lock1(periodicExecutorMutex_);
    if (!periodicExecutor_) {
        syncStopping_ = false;
        std::unique_lock<std::mutex> lock2(configMutex_);
        std::unique_lock<std::mutex> lock3(dataMutex_);
        auto config = config_;
        auto period = syncState_ <= SyncState::OFFLINE_BORDER ? config.syncInitialPeriod : config.syncCheckPeriod;
        periodicExecutor_ = std::make_unique<PeriodicExecutor>(
            PeriodicExecutor::PECallback([this, config](PeriodicExecutor* executor) {
                syncWorker(config, executor);
            }),
            period,
            PeriodicExecutor::PeriodicType::CALLBACK_FIRST);
    } else {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedStartThread.AlreadyStarted", "NtpSync already started");
        throw std::runtime_error("NtpSync already started");
    }
    YIO_LOG_INFO("NtpSync: worker thread started");
    return true;
}

bool NtpSync::stopThread()
{
    YIO_LOG_DEBUG("NtpSync: stoping worker thread");
    bool result{false};
    std::unique_lock<std::mutex> lock1(periodicExecutorMutex_);
    if (periodicExecutor_) {
        syncStopping_ = true;
        periodicExecutor_.reset();
        result = true;
    }
    flushException(true);
    flushSyncError(true);
    YIO_LOG_INFO("NtpSync: worker thread stopped");
    return result;
}

void NtpSync::syncWorker(const Config& config, quasar::PeriodicExecutor* executor)
{
    bool recoveryMode{false};
    try {
        bool syncEnabled{false};
        bool needAccurateSync{false};
        bool checkRouterNtp{false};
        bool useHttpSync{false};
        std::vector<NtpClient::Addr> ipCachedNtpServers;
        {
            std::unique_lock<std::mutex> lock2(dataMutex_);
            std::chrono::milliseconds uptime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startTime_);
            syncEnabled = config.syncEnabled && !insufficientPrivilegeFlag_;
            needAccurateSync = syncEnabled && syncState_ <= SyncState::OUT_OF_SYNC;
            checkRouterNtp = (routerNtp_ != RouterNtp::AVAILABLE && config.tryRouterNtpServer);
            useHttpSync = (syncEnabled && !httpSynced_ && syncState_ < SyncState::OUT_OF_SYNC && !config.timeOverHttpUrl.empty() && uptime >= config.timeOverHttpDelay);
            if (!ntpResult_ && unsuccessfulAttempts_ && (unsuccessfulAttempts_ % 7 == 0)) {
                ipCachedNtpServers = loadIpCache(config.ipCacheFile);
                if (!ipCachedNtpServers.empty()) {
                    recoveryMode = true;
                    YIO_LOG_INFO("NtpSync: Recovery mode attempt, use cached IP addresses");
                }
            }
        }

        std::optional<NtpClient::NtpResult> ntpResult;
        std::string exceptionMessage;
        if (!needAccurateSync) {
            // If it is known that the time is not synchronized, then you can skip the verification phase and
            // immediately go to normal synchronization
            std::optional<NtpClientException> telemetryException;
            NtpClient::Params mParams = config.syncParams;
            mParams.sufficientMeasuringCount = 1;
            mParams.timeout = std::chrono::milliseconds{1000};
            if (recoveryMode) {
                mParams.ntpServers = ipCachedNtpServers;
            }
            NtpClient mNtpClient(mParams);
            for (size_t retry = 0; retry < 5; ++retry) {
                if (syncStopping_) {
                    break;
                } // for
                try {
                    ntpResult = mNtpClient.sync(NtpClient::SyncMode::DISCOVERY, syncStopping_);
                    if (ntpResult && ntpResult->serverTimeCredibility != NtpClient::ServerTimeCredibility::UNDEFINED) {
                        break;
                    }
                } catch (const NtpClientException& ex) {
                    telemetryException = ex;
                    exceptionMessage = ex.what();
                } catch (const std::exception& ex) {
                    telemetryException = std::nullopt;
                    exceptionMessage = ex.what();
                    continue; // for
                }
            }
            if (syncStopping_) {
                return;
            }
            if (telemetryException) {
                reportException(recoveryMode, *telemetryException);
            }
        }

        if (ntpResult && ntpResult->serverTimeCredibility != NtpClient::ServerTimeCredibility::UNDEFINED) {
            needAccurateSync = needAccurateSync || (syncEnabled && ntpResult->absSyncDiff() > config.syncFluctuation);
        }

        if (needAccurateSync) {
            NtpClient::Params mParams = config.syncParams;
            if (recoveryMode) {
                mParams.ntpServers = ipCachedNtpServers;
            }
            NtpClient sNtpClient(mParams);
            std::optional<NtpClient::NtpResult> accurateResult;
            try {
                // Making accurate time synchronization
                accurateResult = sNtpClient.sync(NtpClient::SyncMode::RANDOMIZE, syncStopping_);
            } catch (const NtpClientException& ex) {
                reportException(recoveryMode, ex);
                exceptionMessage = ex.what();
            } catch (const std::exception& ex) {
                exceptionMessage = ex.what();
            }

            if (accurateResult) {
                ntpResult = accurateResult;
            }
        }

        if (checkRouterNtp || (!ntpResult && config.tryRouterNtpServer)) {
            std::string gatewayIp = GatewayMonitor::gatewayIp();
            if (!gatewayIp.empty()) {
                NtpClient::Params rParams = config.syncParams;
                rParams.ntpServers = std::vector<NtpClient::Addr>{NtpClient::Addr{gatewayIp, 123}};
                NtpClient rNtpClient(rParams);
                std::optional<NtpClient::NtpResult> gatewayNtpResult;
                std::string routerNtpExceptionMessage;
                try {
                    gatewayNtpResult = rNtpClient.sync(NtpClient::SyncMode::DEFAULT, syncStopping_);
                } catch (const std::exception& ex) {
                    routerNtpExceptionMessage = ex.what();
                } catch (...) {
                    routerNtpExceptionMessage = "unexpected exception";
                }

                if (gatewayNtpResult) {
                    if (!ntpResult) {
                        ntpResult = gatewayNtpResult;
                        ntpResult->addr = NtpClient::Addr{"gateway", 123};
                    }
                    std::unique_lock<std::mutex> lock2(dataMutex_);
                    if (routerNtp_ != RouterNtp::AVAILABLE) {
                        YIO_LOG_DEBUG("Router NTP server is available at IP " << gatewayIp << " (success)");
                        routerNtp_ = RouterNtp::AVAILABLE;
                    }
                } else {
                    std::unique_lock<std::mutex> lock2(dataMutex_);
                    if (routerNtp_ != RouterNtp::UNAVAILABLE) {
                        YIO_LOG_DEBUG("Router NTP server is not available at IP " << gatewayIp
                                                                                  << (routerNtpExceptionMessage.empty() ? std::string{} : ": " + routerNtpExceptionMessage));
                        routerNtp_ = RouterNtp::UNAVAILABLE;
                    }
                }
            }
        }

        bool thisIsHttpSync = false;
        if (useHttpSync && !ntpResult) {
            try {
                auto tp = getUtcTimeOverHttp(config.timeOverHttpUrl, std::chrono::seconds{3});
                auto diffNs = std::chrono::nanoseconds{tp.time_since_epoch()} - std::chrono::nanoseconds{std::chrono::steady_clock::now().time_since_epoch()};
                ntpResult = NtpClient::NtpResult(
                    NtpClient::SyncMode::CUSTOM,
                    std::vector<NtpClient::Addr>{},
                    NtpClient::Addr{config.timeOverHttpUrl, 80},
                    "", "HTTP", NtpClient::ServerTimeCredibility::UNDEFINED,
                    std::chrono::duration_cast<NtpClient::duration>(diffNs));
                httpSynced_ = true;
                thisIsHttpSync = true;
            } catch (const std::exception& ex) {
                YIO_LOG_INFO("Try to sync over http failed: " << ex.what());
            }
        }

        bool needLogSyncState = false;
        bool reportFirstSync = false;
        SyncState newSyncState = SyncState::UNDEFINED;
        std::optional<std::chrono::nanoseconds> syncDiff;
        SyncState oldSyncState = SyncState::UNDEFINED;
        {
            std::unique_lock<std::mutex> lock(dataMutex_);
            oldSyncState = syncState_;
            if (!ntpResult) {
                if (ntpResult_) {
                    syncState_ = ntpResult_->absSyncDiff() < config.syncFluctuation
                                     ? (syncState_ == SyncState::SYNC ? SyncState::EXPIRED_SYNC : syncState_)
                                     : std::min(SyncState::OUT_OF_SYNC, syncState_);
                } else {
                    syncState_ = std::min(SyncState::OUT_OF_SYNC, syncState_);
                }
                ++unsuccessfulAttempts_;
                logSyncError(recoveryMode, std::move(exceptionMessage));
            } else {
                delayedError_ = std::nullopt;
                unsuccessfulAttempts_ = 0;
                if (!ntpResult_) {
                    reportFirstSync = true;
                }
                ntpResult_ = ntpResult;
                if (!recoveryMode) {
                    saveIpCache(config.ipCacheFile, ntpResult->resolvedIps);
                }
                auto diff = ntpResult->syncDiff();
                auto absDiff = std::chrono::nanoseconds{std::abs(diff.count())};
                if (syncEnabled && absDiff > config.syncFluctuation) {
                    // Make system clock synchronization
                    int syncErrNo = 0;
                    for (size_t retry = 0; !syncErrNo && retry < 5; ++retry) {
                        std::this_thread::yield();
                        auto retCode = settimeOfDay(ntpResult->syncTime());
                        syncErrNo = (retCode == -1 ? errno : 0);
                        if (!syncErrNo) {
                            // if at the time of the call to "settimeOfDay(...)" there was a context switch and our thread
                            // suspended, then the difference between syncTime() and what was set can be very large.
                            constexpr std::chrono::milliseconds callError{1};
                            if (ntpResult->absSyncDiff() > callError) {
                                continue; // for
                            } else {
                                needLogSyncState = true;
                                syncDiff = diff;
                                YIO_LOG_INFO("NtpSync: system time was synchronized: TimeDiffUs=" << (int64_t)std::chrono::duration_cast<std::chrono::microseconds>(diff).count()
                                                                                                  << "; Ntp server " << ntpResult->addr.host << ":" << ntpResult->addr.port << " (" << ntpResult->ipAddress << ")");

                                setKernelSyncStatus();
                            }
                        }
                        break; // for
                    }

                    if (syncErrNo == EPERM) {
                        insufficientPrivilegeFlag_ = true;
                        YIO_LOG_ERROR_EVENT("NtpSync.FailedSetSystemTime.InvalidPermissions", "NtpSync: fail to change system time, insufficient privileges to change system time");
                    } else if (syncErrNo) {
                        YIO_LOG_ERROR_EVENT("NtpSync.FailedSetSystemTime.Error", "NtpSync: fail to change system time, errno=" << syncErrNo);
                    }
                }

                syncState_ = ntpResult_->absSyncDiff() < config.syncFluctuation
                                 ? (thisIsHttpSync ? SyncState::HTTP_SYNC : SyncState::SYNC)
                                 : SyncState::OUT_OF_SYNC;
            }
            needLogSyncState = needLogSyncState || (syncState_ != oldSyncState);
            newSyncState = syncState_;
        }

        bool isSyncSuccessful = (newSyncState >= SyncState::SUCCESS_SYNC);
        onSyncStateChangedFunc_(isSyncSuccessful);

        if (oldSyncState <= SyncState::OFFLINE_BORDER && newSyncState > SyncState::OFFLINE_BORDER) {
            if (config.syncCheckPeriod != config.syncInitialPeriod) {
                YIO_LOG_INFO("NtpSync: switch ntp sync period from " << config.syncInitialPeriod.count() << "s to " << config.syncCheckPeriod.count() << "s");
                executor->setPeriodTime(config.syncCheckPeriod);
            }
        }

        if (syncEnabled) {
            flushMonotonicClock(std::chrono::system_clock::now(), config.monotonicClockFile);
        }

        if (needLogSyncState || reportFirstSync || syncDiff) {
            reportEvent(recoveryMode, syncDiff, reportFirstSync);
        }

        if (needLogSyncState) {
            logSyncState();
        }
    } catch (const std::exception& ex) {
        reportException(recoveryMode, std::string{ex.what()});
        YIO_LOG_ERROR_EVENT("NtpSync.UnknownWorkerException", "NtpSync: worker thread catch unexpected exception: " << ex.what());
    } catch (...) {
        reportException(recoveryMode, std::string{"..."});
    }
}

void NtpSync::logSyncState() const {
    SyncState syncState = SyncState::UNDEFINED;
    std::optional<NtpClient::NtpResult> ntpResult;
    NtpClient::time_point systemTime;
    std::chrono::milliseconds syncFluctuation{0};

    Config config;
    {
        std::unique_lock<std::mutex> lock(configMutex_);
        syncFluctuation = config_.syncFluctuation;
    }

    {
        std::unique_lock<std::mutex> lock(dataMutex_);
        systemTime = NtpClient::NtpResult::systemTime();
        ntpResult = ntpResult_;
        syncState = syncState_;
    }

    std::stringstream ss;
    auto timePointToText =
        [](const NtpClient::time_point& tp)
    {
        std::time_t ttp = std::chrono::system_clock::to_time_t(
            std::chrono::system_clock::time_point{
                std::chrono::duration_cast<std::chrono::system_clock::duration>(
                    tp.time_since_epoch()),
            });
        std::stringstream result;
        result << std::ctime(&ttp);
        return trim(result.str());
    };

    ss << "NtpSync: System time \"" << timePointToText(systemTime) << "\"";
    if (ntpResult) {
        ss << ", ntp time \"" << timePointToText(ntpResult->syncTime()) << "\"";
        auto ns1 = std::chrono::duration_cast<std::chrono::milliseconds>(ntpResult->absSyncDiff());
        if (ns1 > syncFluctuation) {
            ss << ", time delta " << std::chrono::duration_cast<std::chrono::milliseconds>(ntpResult->syncDiff()).count() << "ms";
        }
        ss << ", ntp server " << ntpResult->addr.host << ":" << ntpResult->addr.port << " (" << ntpResult->ipAddress << ")";
    } else {
        ss << ", ntp time unknown";
    }
    ss << ", sync state [" << syncStateText(syncState) << "]";
    YIO_LOG_INFO(ss.str());
}

void NtpSync::reportEvent(bool recoveryMode, const std::optional<NtpClient::duration>& syncDiff, bool firstSync) const {
    SyncState syncState = SyncState::UNDEFINED;
    RouterNtp routerNtp = RouterNtp::UNDEFINED;
    std::optional<NtpClient::NtpResult> ntpResult;
    std::chrono::milliseconds uptime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startTime_);
    bool insufficientPrivilegeFlag = false;
    {
        std::unique_lock<std::mutex> lock(dataMutex_);
        ntpResult = ntpResult_;
        syncState = syncState_;
        insufficientPrivilegeFlag = insufficientPrivilegeFlag_;
        routerNtp = routerNtp_;
    }

    Json::Value attributes;
    if (!sid_.empty()) {
        attributes["sid"] = sid_;
    }
    if (recoveryMode) {
        attributes["recoveryMode"] = recoveryMode;
    }
    attributes["uptimeMs"] = static_cast<uint64_t>(uptime.count());
    attributes["syncState"] = syncStateText(syncState);
    attributes["systemTimestampMs"] = static_cast<uint64_t>(
        std::chrono::duration_cast<std::chrono::milliseconds>(
            NtpClient::NtpResult::systemTime().time_since_epoch())
            .count());
    if (routerNtp != RouterNtp::UNDEFINED) {
        attributes["routerNtp"] = routerNtpText(routerNtp);
    }
    if (ntpResult) {
        attributes["ntpTimestampMs"] = static_cast<uint64_t>(
            std::chrono::duration_cast<std::chrono::milliseconds>(
                ntpResult->syncTime().time_since_epoch())
                .count());
        attributes["ntpHost"] = ntpResult_->addr.host;
        attributes["ntpPort"] = ntpResult_->addr.port;
        attributes["ntpIp"] = ntpResult_->ipAddress;
    }

    if (syncDiff) {
        attributes["syncDiffUs"] = static_cast<int64_t>(
            std::chrono::duration_cast<std::chrono::microseconds>(*syncDiff).count());
    }

    if (firstSync) {
        attributes["firstSync"] = true;
    }

    if (insufficientPrivilegeFlag) {
        attributes["insufficientPrivilege"] = insufficientPrivilegeFlag;
    }

    auto jsonString = jsonToString(attributes, true);
    YIO_LOG_INFO("Ntp report: " << jsonString);
    device_->telemetry()->reportEvent("ntpSync", jsonString);
}

void NtpSync::reportException(bool recoveryMode, const NtpClientException& ex) const {
    SyncState syncState = SyncState::UNDEFINED;
    RouterNtp routerNtp = RouterNtp::UNDEFINED;
    std::chrono::milliseconds uptime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startTime_);
    {
        std::unique_lock<std::mutex> lock(dataMutex_);
        syncState = syncState_;
        routerNtp = routerNtp_;
    }
    Json::Value attributes;
    if (!sid_.empty()) {
        attributes["sid"] = sid_;
    }
    if (recoveryMode) {
        attributes["recoveryMode"] = recoveryMode;
    }
    attributes["uptimeMs"] = static_cast<uint64_t>(uptime.count());
    attributes["syncState"] = syncStateText(syncState);
    if (routerNtp != RouterNtp::UNDEFINED) {
        attributes["routerNtp"] = routerNtpText(routerNtp);
    }
    attributes["exception"] = ex.message;
    if (!ex.host.empty()) {
        attributes["ntpHost"] = ex.host;
    }
    if (!ex.ipAddress.empty()) {
        attributes["ntpIp"] = ex.ipAddress;
    }
    if (!ex.host.empty() || !ex.ipAddress.empty()) {
        attributes["ntpPort"] = ex.port;
    }
    reportException(attributes);
}

void NtpSync::reportException(bool recoveryMode, const std::string& message) const {
    SyncState syncState = SyncState::UNDEFINED;
    RouterNtp routerNtp = RouterNtp::UNDEFINED;
    std::chrono::milliseconds uptime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startTime_);
    {
        std::unique_lock<std::mutex> lock(dataMutex_);
        syncState = syncState_;
        routerNtp = routerNtp_;
    }
    Json::Value attributes;
    if (!sid_.empty()) {
        attributes["sid"] = sid_;
    }
    if (recoveryMode) {
        attributes["recoveryMode"] = recoveryMode;
    }
    attributes["uptimeMs"] = static_cast<uint64_t>(uptime.count());
    attributes["syncState"] = syncStateText(syncState);
    if (routerNtp != RouterNtp::UNDEFINED) {
        attributes["routerNtp"] = routerNtpText(routerNtp);
    }
    attributes["exception"] = message;
    reportException(attributes);
}

void NtpSync::reportException(const Json::Value& attributes) const {
    std::string key = attributes["syncState"].asString() + attributes["exception"].asString();
    auto it = delayedExceptions_.find(key);
    if (it != delayedExceptions_.end()) {
        ++it->second.counter;
        if (it->second.counter == 1) {
            it->second.message = jsonToString(attributes);
        }
    } else {
        device_->telemetry()->reportEvent("ntpException", jsonToString(attributes));
        auto& rec = delayedExceptions_[key];
        rec.counter = 0;
        rec.message = jsonToString(attributes);
        rec.timePoint = std::chrono::steady_clock::now();
    }
    flushException(false);
}

void NtpSync::flushException(bool force) const {
    std::chrono::steady_clock::time_point now = std::chrono::steady_clock::now();
    for (auto it = delayedExceptions_.begin(); it != delayedExceptions_.end();) {
        if (force || it->second.timePoint + flushErrorPeriod < now) {
            if (it->second.counter > 0) {
                if (auto json = tryParseJson(it->second.message)) {
                    (*json)["repeated"] = it->second.counter;
                    device_->telemetry()->reportEvent("ntpException", jsonToString(*json));
                }
            }
            it = delayedExceptions_.erase(it);
        } else {
            ++it;
        }
    }
}

void NtpSync::logSyncError(bool recoveryMode, std::string message) const {
    if (recoveryMode) {
        message += " (Recovery mode)";
    }
    std::chrono::steady_clock::time_point now = std::chrono::steady_clock::now();
    if (!delayedError_ || message != delayedError_->message || delayedError_->timePoint + flushErrorPeriod < now) {
        flushSyncError(true);
        if (!delayedError_) {
            delayedError_.emplace();
        }
        delayedError_->message = std::move(message);
        delayedError_->timePoint = now;
        delayedError_->counter = 0;
        YIO_LOG_WARN("ERROR: NtpSync cannot get synchronized time: " << delayedError_->message);
    } else {
        ++delayedError_->counter;
    }
}

void NtpSync::flushSyncError(bool force) const {
    if (!delayedError_ || delayedError_->counter == 0) {
        return;
    }

    std::chrono::steady_clock::time_point now = std::chrono::steady_clock::now();
    if (force || delayedError_->timePoint + flushErrorPeriod < now) {
        YIO_LOG_WARN("ERROR: NtpSync cannot get synchronized time: " << delayedError_->message << " (x" << std::to_string(delayedError_->counter) << " times)");
    }
}

std::vector<NtpClient::Addr> NtpSync::loadIpCache(const std::string& filename)
{
    std::vector<NtpClient::Addr> result;

    if (!filename.empty()) {
        try {
            auto json = parseJson(getFileContent(filename));
            auto ips = tryGetArray(json, "ips");
            result.reserve(ips.size());
            for (Json::ArrayIndex i = 0; i < ips.size(); ++i) {
                result.emplace_back(NtpClient::Addr(ips[i].asString()));
            }
        } catch (...) {
        }
    }
    return result;
}

void NtpSync::saveIpCache(const std::string& filename, const std::vector<NtpClient::Addr>& addrs)
{
    try {
        std::vector<NtpClient::Addr> ips(loadIpCache(filename));
        ips.insert(ips.end(), addrs.begin(), addrs.end());
        std::sort(ips.begin(), ips.end());
        ips.erase(std::unique(ips.begin(), ips.end()), ips.end());
        if (ips.size() > maxIpCache) {
            std::shuffle(ips.begin(), ips.end(), std::default_random_engine(std::chrono::system_clock::now().time_since_epoch().count()));
            ips.resize(maxIpCache);
            std::sort(ips.begin(), ips.end());
        }

        Json::Value json;
        for (const auto& ip : ips) {
            json["ips"].append(Json::Value(ip.host + ":" + std::to_string(ip.port)));
        }
        AtomicFile file(filename);
        file.write(jsonToString(json) + "\n");
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedSaveIpCache", "NtpSync: fail to save ip cache: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedSaveIpCache", "NtpSync: fail to save ip cache: unexpected exception");
    }
}

void NtpSync::createOrRemoveSyncEnabledMarkerFile(bool syncEnabled, const std::string& syncEnabledMarkerFile)
{
    if (syncEnabledMarkerFile.empty()) {
        return;
    }

    if (syncEnabled) {
        if (!fileExists(syncEnabledMarkerFile)) {
            try {
                PersistentFile file(syncEnabledMarkerFile, PersistentFile::Mode::APPEND);
            } catch (std::exception& ex) {
                YIO_LOG_ERROR_EVENT("NtpSync.FailedCreateMarkerFile", "Fail to create " << syncEnabledMarkerFile << " marker file");
            }
        }
    } else {
        if (fileExists(syncEnabledMarkerFile)) {
            std::remove(syncEnabledMarkerFile.c_str());
            if (fileExists(syncEnabledMarkerFile)) {
                YIO_LOG_ERROR_EVENT("NtpSync.FailedRemoveMarkerFile", "Fail to remove " << syncEnabledMarkerFile << " marker file");
            }
        }
    }
}

void NtpSync::flushMonotonicClock(std::chrono::system_clock::time_point tp, const std::string& filename) noexcept {
    // Just in case, hard-coded is the inability to synchronize time. Disabling this #if is planned by another commit
    if (filename.empty()) {
        return;
    }

    int64_t timestamp = std::chrono::duration_cast<std::chrono::seconds>(tp.time_since_epoch()).count();
    if (timestamp < 0) {
        return;
    }

    try {
        AtomicFile file(filename);
        file.write(std::to_string(timestamp) + "\n");
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedSaveMonotonicTimestamp", "NtpSync: fail to save monotonic timestamp: " << ex.what());
    } catch (...) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedSaveMonotonicTimestamp", "NtpSync: fail to save monotonic timestamp: unexpected exception");
    }
}

std::optional<std::chrono::system_clock::time_point> NtpSync::loadMonotonicClock(const std::string& filename) noexcept {
    try {
        int64_t timestamp = boost::lexical_cast<int64_t>(trim(getFileContent(filename)));
        const int64_t memorableDate = std::max<int64_t>(1577836800, GetProgramBuildTimestamp()); // 2020-01-01
        if (timestamp > memorableDate) {
            return std::chrono::system_clock::time_point{std::chrono::seconds{timestamp}};
        }
    } catch (const std::exception& ex) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedToLoadMonotonicTimestamp", "NtpSync: failed to load monotonic timestamp: " << ex.what());
    }
    return std::nullopt;
}

void NtpSync::setKernelSyncStatus() {
#ifndef __ANDROID__
    struct timex txc;
    memset(&txc, 0, sizeof(txc));

    int ret = adjtimex(&txc);
    if (ret < 0) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedSetSyncStatus", "NtpSync: failed to get kernel clock status, errno=" << errno);
        return;
    }

    txc.modes = ADJ_STATUS;
    txc.status &= ~STA_UNSYNC;

    ret = adjtimex(&txc);
    if (ret < 0) {
        YIO_LOG_ERROR_EVENT("NtpSync.FailedSetSyncStatus", "NtpSync: failed to clear STA_UNSYNC bit, errno=" << errno);
        return;
    }

    YIO_LOG_INFO("NtpSync: set kernel sync status");
#endif
}
