#pragma once

#include <yandex_io/libs/base/linux_timer.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/ntp/ntp_client.h>
#include <yandex_io/libs/ntp/ntp_client_exception.h>
#include <yandex_io/libs/threading/blocking_queue.h>
#include <yandex_io/libs/threading/periodic_executor.h>

#include <atomic>
#include <mutex>
#include <optional>
#include <unordered_map>

namespace quasar {

    class NtpSync {
    public:
        struct Config {
            NtpClient::Params syncParams;
            std::string ipCacheFile;
            std::string monotonicClockFile;
            std::chrono::seconds syncInitialPeriod = std::chrono::seconds{5};
            std::chrono::seconds syncCheckPeriod = std::chrono::minutes{1};
            std::chrono::seconds timeOverHttpDelay = std::chrono::minutes{1};
            std::chrono::milliseconds syncFluctuation{1000};
            bool syncEnabled{true};
            bool tryRouterNtpServer{true};
            std::string syncEnabledMarkerFile;
            std::string timeOverHttpUrl;

            bool operator==(const Config& other) const;
            bool operator!=(const Config& other) const;
        };

        enum class SyncState {
            UNDEFINED,
            MONOTONIC,

            OFFLINE_BORDER = MONOTONIC,

            OUT_OF_SYNC,

            SUCCESS_SYNC,

            HTTP_SYNC = SUCCESS_SYNC,
            EXPIRED_SYNC,
            SYNC,
        };

        enum RouterNtp {
            UNDEFINED,
            UNAVAILABLE,
            AVAILABLE,
        };

        NtpSync(std::shared_ptr<YandexIO::IDevice> device, const Json::Value& deviceNtpdConfig, const Json::Value& customNtpdConfig, std::string sid);
        ~NtpSync();

        bool start() noexcept;
        bool stop() noexcept;
        bool reloadConfig(const Json::Value& customNtpdConfig) noexcept;
        void reloadDefaultConfig() noexcept;
        void syncNow() noexcept;
        void setOnSyncStatusChanged(std::function<void(bool isSyncSuccessful)> func) noexcept;

    private:
        static Config parseConfig(const Config& deviceConfig, const Json::Value& ntpdConfig);
        static Json::Value configToJson(const Config& ntpdConfig);

        bool reloadConfig(Config newConfig) noexcept;
        bool startThread();
        bool stopThread();

        void syncWorker(const Config& config, quasar::PeriodicExecutor* executor);
        void logSyncState() const;

        void reportEvent(bool recoveryMode, const std::optional<std::chrono::nanoseconds>& syncDiff, bool firstSync) const;
        void reportException(bool recoveryMode, const NtpClientException& ex) const;
        void reportException(bool recoveryMode, const std::string& message) const;
        void reportException(const Json::Value& attributes) const;
        void flushException(bool force) const;

        void logSyncError(bool recoveryMode, std::string message) const;
        void flushSyncError(bool force) const;

        static std::vector<NtpClient::Addr> loadIpCache(const std::string& filename);
        static void saveIpCache(const std::string& filename, const std::vector<NtpClient::Addr>& addrs);
        static void createOrRemoveSyncEnabledMarkerFile(bool syncEnabled, const std::string& syncEnabledMarkerFile);

        static void flushMonotonicClock(std::chrono::system_clock::time_point tp, const std::string& filename) noexcept;
        static std::optional<std::chrono::system_clock::time_point> loadMonotonicClock(const std::string& filename) noexcept;

        static void setKernelSyncStatus();

        std::function<void(bool isSyncSuccessful)> onSyncStateChangedFunc_;

    private:
        const std::shared_ptr<YandexIO::IDevice> device_;
        const Config deviceConfig_;
        const std::string sid_;
        const std::chrono::steady_clock::time_point startTime_;
        std::unique_ptr<PeriodicExecutor> reportExecutor_;
        std::atomic<bool> syncStopping_{false};
        std::atomic<bool> httpSynced_{false};

        mutable std::mutex periodicExecutorMutex_;
        std::unique_ptr<PeriodicExecutor> periodicExecutor_;

        mutable std::mutex configMutex_;
        Config config_;

        mutable std::mutex dataMutex_;
        bool insufficientPrivilegeFlag_{false};
        SyncState syncState_{SyncState::UNDEFINED};
        std::optional<NtpClient::NtpResult> ntpResult_;
        size_t unsuccessfulAttempts_{0};
        struct ErrorRecord {
            std::string message;
            size_t counter{0};
            std::chrono::steady_clock::time_point timePoint;
        };
        mutable std::optional<ErrorRecord> delayedError_;
        mutable std::unordered_map<std::string, ErrorRecord> delayedExceptions_;
        mutable RouterNtp routerNtp_{RouterNtp::UNDEFINED};
    };

} // namespace quasar
