#include "push_endpoint.h"

#include "xiva_operations.h"

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <util/generic/scope.h>

#include <chrono>
#include <cstdlib>
#include <cstring>
#include <ctime>
#include <iostream>
#include <memory>

YIO_DEFINE_LOG_MODULE("push");

using std::placeholders::_1;
using std::placeholders::_2;

using namespace quasar;

const std::string PushEndpoint::SERVICE_NAME = "pushd";

// must be called under xivaClientThreadMutex_
bool PushEndpoint::stopMainLoopThread() {
    {
        std::lock_guard<std::mutex> guard(mutex_);
        stopped_ = true;
    }
    wakeupVar_.notify_one();

    if (xivaClientThread_.joinable()) {
        xivaClientThread_.join();
        YIO_LOG_DEBUG("Main loop thread joined");
        return true;
    }
    return false;
}

PushEndpoint::PushEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider,
    bool sslVerification)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , userConfigProvider_(std::move(userConfigProvider))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , xivaClient_(device_->telemetry())
    , deviceContext_(ipcFactory)
    , connectionStatsSender_(device_->telemetry(), "xiva")
{
    Y_VERIFY(authProvider_);
    Y_VERIFY(userConfigProvider_);

    YIO_LOG_INFO("Pushd starting...");

    if (sslVerification) {
        auto commonConfig = device_->configuration()->getServiceConfig("common");
        wsSettings_.tls.crtFilePath = getString(commonConfig, "caCertsFile");
    } else {
        wsSettings_.tls.disabled = true;
    }

    auto serviceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
    const auto reconnectTimeoutSecBase = tryGetInt(serviceConfig, "reconnectTimeoutSecBase", 1);
    const auto reconnectTimeoutSecMax = tryGetInt(serviceConfig, "reconnectTimeoutSecMax", 60);
    httpClientTimeoutSec_ = tryGetInt(serviceConfig, "httpClientTimeoutSec", httpClientTimeoutSec_);

    const auto period = std::chrono::seconds(reconnectTimeoutSecBase);
    const auto maxPeriod = std::chrono::seconds(reconnectTimeoutSecMax);
    backoffHelper_ = BackoffRetriesWithRandomPolicy(getCrc32(device_->deviceId()));
    backoffHelper_.initCheckPeriod(period, period, maxPeriod);

    xivaServices_ = tryGetString(serviceConfig, "xivaServices", "quasar-realtime,messenger-prod");

    xivaSubscribeUrl_ = getString(serviceConfig, "xivaSubscribeUrl");

    xivaClient_.setOnConnectHandler([this]() {
        onXivaWsStatus(WebsocketClientState::CONNECTED);
    });

    xivaClient_.setOnDisconnectHandler([this](const quasar::Websocket::ConnectionInfo& /* info */) {
        onXivaWsStatus(WebsocketClientState::DISCONNECTED);
    });

    xivaClient_.setOnFailHandler([this](const quasar::Websocket::ConnectionInfo& /* info */) {
        onXivaWsStatus(WebsocketClientState::DISCONNECTED);
    });

    sslVerification_ = sslVerification;

    server_->setMessageHandler(std::bind(&PushEndpoint::handleQuasarMessageServer, this, ::_1, ::_2));

    server_->setClientConnectedHandler([this](auto& connection) {
        std::lock_guard<std::mutex> g(xivaConfigMutex_);

        QuasarMessage msg;
        const auto subscriptions = getXivaSubscriptionsNoLock();
        *msg.mutable_xiva_subscriptions()->mutable_xiva_subscriptions() = {subscriptions.begin(), subscriptions.end()};

        connection.send(std::move(msg));
    });

    server_->listenService();
    authProvider_->ownerAuthInfo().connect(
        [this](const auto& authInfo)
        {
            std::unique_lock lock(mutex_);
            if (authToken_ != authInfo->authToken || passportUid_ != authInfo->passportUid) {
                YIO_LOG_INFO("Auth state changed ...");
                authToken_ = authInfo->authToken;
                passportUid_ = authInfo->passportUid;
                lock.unlock();

                std::lock_guard lockGuard(xivaClientThreadMutex_);
                stopMainLoopThread();
                if (authInfo->isAuthorized()) {
                    stopped_ = false;
                    xivaClientThread_ = std::thread(&PushEndpoint::mainLoop, this);
                }
            }
        }, lifetime_);

    userConfigProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "/").connect([this](const auto& jsonSystemConfig) {
        onSystemConfigChanged(jsonSystemConfig);
    }, lifetime_);
}

PushEndpoint::~PushEndpoint() {
    YIO_LOG_INFO("Pushd stopping...");
    lifetime_.die();
    std::lock_guard lockGuard(xivaClientThreadMutex_);
    stopMainLoopThread();
}

void PushEndpoint::mainLoop() {
    std::unique_lock lock(mutex_);
    while (!stopped_) {
        try {
            lock.unlock();
            bool needLock = true;
            Y_DEFER {
                if (needLock) {
                    lock.lock();
                }
            };

            // try to get subscribe info from quasar backend. Will throw on error...
            auto subscribeInfo = getSubscribeInfo(device_->deviceId(), authToken_, passportUid_);
            const std::string subscribeUrl = getXivaSubscribeUrl(subscribeInfo);

            needLock = false;
            lock.lock();

            subscribeToXiva(subscribeUrl);

            xivaSecretSignDeadlineTs_ = atoi(subscribeInfo.signDeadlineTs.c_str());
            // Successfully  got subscribe info and connected to xiva. Reset backoff delay to default
            backoffHelper_.resetDelayBetweenCallsToDefault();
            time_t currentTs = time(nullptr);
            YIO_LOG_INFO("Time to deadline " << xivaSecretSignDeadlineTs_ - currentTs << " secs");
            while (xivaPingDeadlineTs_ > currentTs && xivaSecretSignDeadlineTs_ > currentTs && !stopped_) {
                time_t waitTime = std::min(xivaPingDeadlineTs_, xivaSecretSignDeadlineTs_) - currentTs;
                YIO_LOG_DEBUG("Ping timer set, waiting for: " << waitTime);
                wakeupVar_.wait_until(lock, std::chrono::steady_clock::now() + std::chrono::seconds(waitTime));
                currentTs = time(nullptr);
            }
            if (xivaPingDeadlineTs_ <= currentTs) {
                YIO_LOG_WARN("No ping messages for long");
            }
            if (xivaSecretSignDeadlineTs_ <= currentTs) {
                YIO_LOG_INFO("Xiva secret sign is outdated: " << xivaSecretSignDeadlineTs_ << " <= " << currentTs);
            }
        } catch (const std::exception& e) {
            YIO_LOG_WARN("Exception in main loop " << e.what());
            backoffHelper_.increaseDelayBetweenCalls();
        }
        if (!stopped_) {
            const std::chrono::milliseconds delay = backoffHelper_.getDelayBetweenCalls();
            YIO_LOG_INFO("Reconnecting in " << delay.count() << "milliseconds");
            wakeupVar_.wait_for(lock, delay, [this]() {
                return stopped_.load();
            });
        }
        /* Disconnect without lock, otherwise onXivaMessage will not be able to lock
         * and disconnectPromise will not be ever set up (because message handler should be done first)
         */
        lock.unlock();
        std::promise<void> disconnectPromise;
        xivaClient_.disconnectAsync([&disconnectPromise]() {
            disconnectPromise.set_value();
        });
        disconnectPromise.get_future().get();
        lock.lock();
    }
    YIO_LOG_INFO("MainLoop is out");
}

std::string PushEndpoint::getXivaSubscribeUrl(const SubscribeInfo& subscribeInfo) const {
    std::string deviceId = device_->deviceId();
    YIO_LOG_INFO("Authorized: uuid=" << passportUid_ << ", deviceId=" << deviceId);

    std::lock_guard<std::mutex> lockGuard(xivaConfigMutex_);
    return xivaSubscribeUrl_ + "?" + "&service=" + xivaServices_ + "&client=q" + "&session=" + urlEncode(deviceId) + "&sign=" + urlEncode(subscribeInfo.sign) + "&ts=" + urlEncode(subscribeInfo.signDeadlineTs) + "&user=" + urlEncode(passportUid_);
}

PushEndpoint::SubscribeInfo
PushEndpoint::getSubscribeInfo(const std::string& /* deviceId */, const std::string& authToken,
                               const std::string& uuid) const {
    YIO_LOG_INFO("Getting subscribe info");
    auto commonConfig = device_->configuration()->getServiceConfig("common");
    std::string url;
    {
        std::lock_guard<std::mutex> g(xivaConfigMutex_);
        url = getString(commonConfig, "backendUrl") + "/push_subscribe?device_id=" + urlEncode(device_->deviceId()) + "&platform=" + urlEncode(device_->platform()) + "&uuid=" + urlEncode(uuid) + "&service=" + xivaServices_;
    }
    HttpClient::Headers headers = {{"Authorization", "OAuth " + authToken}};
    HttpClient httpClient("pushd-quasar-backend", device_);
    httpClient.setTimeout(std::chrono::seconds{httpClientTimeoutSec_});
    const auto response = httpClient.post("push-subscribe", url, {}, headers);
    const auto body = parseJson(response.body);
    if (isSuccessHttpCode(response.responseCode) && "ok" == body["status"].asString()) {
        YIO_LOG_INFO("Got subscribe info successfully: ts=" << getString(body, "ts"));
        return {getString(body, "sign"), getString(body, "ts")};
    }
    if (response.responseCode == 403 && body["message"] == "AUTH_TOKEN_INVALID") {
        authProvider_->requestAuthTokenUpdate("PushEndpoint push-subscribe");
    }
    throw std::runtime_error("Cannot subscribe to push notifications: " + getString(body, "message"));
}

void PushEndpoint::subscribeToXiva(const std::string& subscribeUrl) {
    YIO_LOG_INFO("Subscribing by url: '" + subscribeUrl + "'");
    xivaClient_.setOnMessageHandler(std::bind(&PushEndpoint::onXivaMessage, this, ::_1));
    xivaClient_.setOnBinaryMessageHandler(std::bind(&PushEndpoint::onBinaryXivaMessage, this, ::_1));
    wsSettings_.url = subscribeUrl;
    onXivaWsStatus(WebsocketClientState::CONNECTING);
    xivaClient_.connectSync(wsSettings_);
    xivaPingDeadlineTs_ = time(nullptr) + 120;
}

void PushEndpoint::sendToAllWebsocketSubscribersNoLock(QuasarMessage&& message) {
    ipc::SharedMessage sharedMessage(std::move(message));
    for (auto it = websocketSubscribers_.begin(); it != websocketSubscribers_.end();) {
        auto conn = it->lock();

        if (conn) {
            conn->send(sharedMessage);
            ++it;
        } else {
            it = websocketSubscribers_.erase(it);
        }
    }
}

void PushEndpoint::onBinaryXivaMessage(const std::string& msg) {
    QuasarMessage message;

    message.set_xiva_websocket_recv_binary_message(TString(msg));

    std::lock_guard<std::mutex> g(lock_);

    sendToAllWebsocketSubscribersNoLock(std::move(message));
}

void PushEndpoint::onXivaWsStatus(const WebsocketClientState::State& st) {
    std::lock_guard<std::mutex> g(lock_);

    if (state_ != st) {
        YIO_LOG_INFO("Xiva websocket state changed: " << (int)state_ << "->" << (int)st);

        connectionStatsSender_.onConnectionStateChanged(st == WebsocketClientState::CONNECTED);

        state_ = st;

        QuasarMessage message;
        message.mutable_xiva_websocket_client_state()->set_state(st);

        sendToAllWebsocketSubscribersNoLock(std::move(message));
    }
}

std::vector<std::string> PushEndpoint::getXivaSubscriptionsNoLock() const {
    return split(xivaServices_, ",");
}

void PushEndpoint::onXivaMessage(const std::string& msg) {
    Json::Value xivaMessage;
    try {
        xivaMessage = parseJson(msg);
    } catch (const Json::Exception& e) {
        YIO_LOG_WARN("Can't parse Xiva message: " << msg << ". Drop it");
        device_->telemetry()->reportEvent("xiva_broken_json", msg);
        YIO_LOG_INFO("Send metrica");
        return;
    }

    // Skip if message has any specified service except of "quasar-realtime"
    if (const auto service = tryGetString(xivaMessage, "service", ""); !service.empty() && service != "quasar-realtime") {
        YIO_LOG_WARN("Skip xiva message with unknown service: " << service);
        return;
    }

    const std::string operation = tryGetString(xivaMessage, "operation", "");
    if (operation.empty()) {
        YIO_LOG_WARN("Skip xiva message with empty operation");
        return;
    }

    /* Check if Quasar declared this operation */
    if (!XivaOperations::isDeclaredXivaOperation(operation)) {
        YIO_LOG_WARN("Unknown Xiva Operation: '" << operation << "'. Drop it");
        device_->telemetry()->reportEvent("unknown_xiva_operation", msg);
        return;
    }

    if (operation == XivaOperations::PING) {
        YIO_LOG_DEBUG("onXivaMessage: " + msg);
        if (xivaMessage.isMember("server-interval-sec") && xivaMessage["server-interval-sec"].isInt()) {
            std::unique_lock lock(mutex_);
            xivaPingDeadlineTs_ = time(nullptr) + getInt(xivaMessage, "server-interval-sec") * 2.1 + 1;
            wakeupVar_.notify_one();
        } else {
            YIO_LOG_WARN("Incorrect " << XivaOperations::PING << " operation. Drop it");
        }
    } else {
        YIO_LOG_INFO("onXivaMessage: " + msg);
        device_->telemetry()->reportEvent("pushReceived", msg);
        processPush(operation, xivaMessage);
    }
}

void PushEndpoint::processPush(const std::string& operation, const Json::Value& xivaMessage) {
    const std::string messageStr = tryGetString(xivaMessage, "message", "{}");

    QuasarMessage quasarMessage;
    quasarMessage.mutable_push_notification()->set_message(TString(messageStr));
    quasarMessage.mutable_push_notification()->set_operation(TString(operation));
    server_->sendToAll(std::move(quasarMessage));
    deviceContext_.firePushNotification(operation, messageStr);
}

std::string PushEndpoint::getCurrentConnectionUrl() const {
    return wsSettings_.url;
}

int PushEndpoint::getPort() const {
    return server_->port();
}

void PushEndpoint::onSystemConfigChanged(const std::shared_ptr<const Json::Value>& json) {
    const Json::Value& systemConfig = *json;

    if (!systemConfig.isNull()) {
        bool changedXivaSubscribeUrl = false;

        if (!systemConfig["xivaSubscribeUrl"].isNull()) {
            std::lock_guard<std::mutex> lockGuard(xivaConfigMutex_);
            if (xivaSubscribeUrl_ != tryGetString(systemConfig, "xivaSubscribeUrl", xivaSubscribeUrl_)) {
                xivaSubscribeUrl_ = getString(systemConfig, "xivaSubscribeUrl");
                customXivaSubscribeUrl_ = true;
                changedXivaSubscribeUrl = true;
                YIO_LOG_INFO("Received custom xivaSubscribeUrl from system config, new url: " << xivaSubscribeUrl_);
            }
        } else if (customXivaSubscribeUrl_) {
            const auto serviceConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
            YIO_LOG_INFO("Restored default xivaSubscribeUrl from service config");
            std::lock_guard<std::mutex> lockGuard(xivaConfigMutex_);
            xivaSubscribeUrl_ = getString(serviceConfig, "xivaSubscribeUrl");
            customXivaSubscribeUrl_ = false;
            changedXivaSubscribeUrl = true;
        }

        const Json::Value& serviceConfig = systemConfig[SERVICE_NAME];
        bool changedXivaServices = false;

        if (!serviceConfig.isNull() && !serviceConfig["xivaServices"].isNull()) {
            const std::string xivaServices = tryGetString(serviceConfig, "xivaServices", "quasar-realtime,messenger-prod");

            std::lock_guard<std::mutex> lockGuard(xivaConfigMutex_);

            if (xivaServices_ != xivaServices) {
                YIO_LOG_INFO("Received new xivaServices: " << xivaServices_ << "->" << xivaServices);
                xivaServices_ = xivaServices;
                changedXivaServices = true;
            }

        } else {
            const std::string defaultXivaServices = tryGetString(
                device_->configuration()->getServiceConfig(SERVICE_NAME),
                "xivaServices",
                "quasar-realtime,messenger-prod");

            std::lock_guard<std::mutex> lockGuard(xivaConfigMutex_);

            if (xivaServices_ != defaultXivaServices) {
                YIO_LOG_INFO("Restored default xivaServices from service config: " << xivaServices_ << "->" << defaultXivaServices);
                xivaServices_ = defaultXivaServices;
                changedXivaServices = true;
            }
        }

        if (changedXivaServices) {
            std::lock_guard<std::mutex> g(xivaConfigMutex_);

            QuasarMessage msg;
            const auto subscriptions = getXivaSubscriptionsNoLock();
            *msg.mutable_xiva_subscriptions()->mutable_xiva_subscriptions() = {subscriptions.begin(), subscriptions.end()};

            server_->sendToAll(std::move(msg));
        }

        // check this to avoid creating mainLoop before auth does
        if (changedXivaSubscribeUrl || changedXivaServices) {
            YIO_LOG_INFO("Stopping mainLoop thread as xivaSubscribeUrl was changed");
            std::lock_guard lockGuard(xivaClientThreadMutex_);
            if (stopMainLoopThread()) {
                stopped_ = false;
                xivaClientThread_ = std::thread(&PushEndpoint::mainLoop, this);
            }
        }
    }
}

void PushEndpoint::handleQuasarMessageServer(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection) {
    if (message->has_xiva_websocket_send_binary_message()) {
        xivaClient_.unsafeSendBinary(message->xiva_websocket_send_binary_message());
    }

    if (message->has_list_xiva_subscriptions()) {
        QuasarMessage response;

        response.set_request_id(message->request_id());

        {
            std::lock_guard<std::mutex> g(xivaConfigMutex_);
            const auto subscriptions = getXivaSubscriptionsNoLock();
            *response.mutable_xiva_subscriptions()->mutable_xiva_subscriptions() = {subscriptions.begin(), subscriptions.end()};
        }

        connection.send(std::move(response));
    }

    if (message->has_subscribe_to_pushd_xiva_websocket_events()) {
        std::lock_guard<std::mutex> g(lock_);
        websocketSubscribers_.insert(connection.share());

        QuasarMessage message;
        message.mutable_xiva_websocket_client_state()->set_state(state_);
        connection.send(std::move(message));
    }
}
