#pragma once

#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/interfaces/user_config/i_user_config_provider.h>
#include <yandex_io/libs/delay_timings_policy/delay_timings_policy.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/connection_stats_sender/connection_stats_sender.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/libs/websocket/websocket_client.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/private/device_context.h>

#include <mutex>
#include <thread>

namespace quasar {

    class PushEndpoint {
        using QuasarMessage = quasar::proto::QuasarMessage;
        using WebsocketClientState = quasar::proto::WebsocketClientState;

    public:
        PushEndpoint(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<IAuthProvider> authProvider,
            std::shared_ptr<IUserConfigProvider> userConfigProvider,
            bool sslVerification = true);
        ~PushEndpoint();
        static const std::string SERVICE_NAME;

        int getPort() const;
        std::string getCurrentConnectionUrl() const;

    private:
        struct SubscribeInfo {
            std::string sign;
            std::string signDeadlineTs; // https://wiki.yandex-team.ru/yxiva/serverapi/
        };

        Lifetime lifetime_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<IAuthProvider> authProvider_;
        const std::shared_ptr<IUserConfigProvider> userConfigProvider_;

        time_t httpClientTimeoutSec_ = 10;
        void mainLoop();
        time_t xivaPingDeadlineTs_;
        time_t xivaSecretSignDeadlineTs_;
        std::atomic_bool stopped_{true};
        bool sslVerification_;

        std::shared_ptr<ipc::IServer> server_;
        WebsocketClient xivaClient_;

        std::mutex xivaClientThreadMutex_;
        std::thread xivaClientThread_;

        std::mutex mutex_;
        quasar::SteadyConditionVariable wakeupVar_;

        std::vector<std::string> getXivaSubscriptionsNoLock() const;

        void onXivaMessage(const std::string& msg);

        void sendToAllWebsocketSubscribersNoLock(QuasarMessage&& message);

        void onBinaryXivaMessage(const std::string& msg);

        void onXivaWsStatus(const WebsocketClientState::State& st);

        void subscribeToXiva(const std::string& subscribeUrl);
        void processPush(const std::string& operation, const Json::Value& xivaMessage);
        SubscribeInfo getSubscribeInfo(
            const std::string& deviceId, const std::string& authToken, const std::string& uuid) const;
        std::string getXivaSubscribeUrl(const SubscribeInfo& subscribeInfo) const;

        /**
         * @brief must be called under xivaClientThreadMutex_
         * @return if thread was joined
         */
        bool stopMainLoopThread();

        std::string authToken_;
        std::string passportUid_;
        WebsocketClient::Settings wsSettings_;

        bool customXivaSubscribeUrl_ = false;
        mutable std::mutex xivaConfigMutex_;
        std::string xivaSubscribeUrl_;
        std::string xivaServices_;

        void handleQuasarMessageServer(const ipc::SharedMessage& message, ipc::IServer::IClientConnection& connection);

        void onSystemConfigChanged(const std::shared_ptr<const Json::Value>& json);

        using WeakConnection = std::weak_ptr<ipc::IServer::IClientConnection>;
        using WeakConnectionComparator = std::owner_less<WeakConnection>;
        using Connections = std::set<WeakConnection, WeakConnectionComparator>;

        std::mutex lock_;
        Connections websocketSubscribers_;
        WebsocketClientState::State state_ = WebsocketClientState::DISCONNECTED;

        BackoffRetriesWithRandomPolicy backoffHelper_;

        YandexIO::DeviceContext deviceContext_;

        ConnectionStatsSender connectionStatsSender_;
    };
} // namespace quasar
