#include "setup_endpoint.h"

#include <yandex_io/capabilities/file_player/file_player_capability_proxy.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ble_decoder/ble_decoder.h>
#include <yandex_io/libs/cryptography/digest.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/setup_parser/setup_parser.h>
#include <yandex_io/libs/setup_parser/wifi_type.h>

YIO_DEFINE_LOG_MODULE("setup");

using namespace quasar;
using namespace quasar::proto;
using namespace quasar::SetupParser;

const std::string SetupEndpoint::SERVICE_NAME = "setupd";

SetupEndpoint::SetupEndpoint(std::shared_ptr<YandexIO::IDevice> device,
                             std::shared_ptr<ipc::IIpcFactory> ipcFactory,
                             std::shared_ptr<YandexIO::IFilePlayerCapability> filePlayerCapability)
    : device_(std::move(device))
    , latencyReporter_(device_->telemetry())
    , filePlayerCapability_(std::move(filePlayerCapability))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , wifiConnector_(ipcFactory->createIpcConnector("wifid"))
    , bleInitdConnector_(ipcFactory->createIpcConnector("ble_initd"))
    , updaterConnector_(ipcFactory->createIpcConnector("updatesd"))
    , deviceContext_(ipcFactory)
    , secretCommands_(device_->configuration()->getServiceConfig(SERVICE_NAME))
{
    server_->setMessageHandler(std::bind(&SetupEndpoint::handleQuasarMessage, this, std::placeholders::_1));

    updaterConnector_->setMessageHandler(std::bind(&SetupEndpoint::handleQuasarMessage, this, std::placeholders::_1));

    wifiConnector_->connectToService();

    updaterConnector_->connectToService();

    if (device_->hal()->getBluetoothCapabilities().hasBle()) {
        bleInitdConnector_->connectToService();
    }

    auto cryptographyConfig = getJson(device_->configuration()->getServiceConfig("common"), "cryptography");
    deviceCryptography_ = device_->hal()->createDeviceCryptography(cryptographyConfig);

    server_->listenService();
    YIO_LOG_INFO("setupd started");
}

SetupEndpoint::~SetupEndpoint() {
    server_->shutdown();
}

void SetupEndpoint::handleQuasarMessage(const ipc::SharedMessage& message) {
    if (message->has_setup_credentials_message()) {
        const auto& msg = message->setup_credentials_message();
        YIO_LOG_DEBUG("has_setup_credentials_message");
        std::vector<byte> credentials{
            msg.setup_credentials().begin(),
            msg.setup_credentials().end(),
        };

        onDataReceived(credentials, msg.source());
    }

    if (message->has_encrypted_setup_credentials_message()) {
        const auto& msg = message->encrypted_setup_credentials_message();
        YIO_LOG_DEBUG("has_encrypted_setup_credentials_message");
        std::vector<byte> decryptedCredentials;
        if (decryptCredentials(msg, msg.source(), decryptedCredentials)) {
            // Consider encrypted credentials as initial protocol version
            onDataReceived(decryptedCredentials, msg.source());
        }
    }

    if (message->has_update_state()) {
        if (updateState_ != message->update_state().state()) {
            if (message->update_state().state() == UpdateState::APPLYING) {
                notifySetupStatusViaBle(SetupStatusMessage::UPDATE_DOWNLOAD_FINISH);
                notifySetupStatusViaBle(SetupStatusMessage::OK);
            } else if (message->update_state().state() == UpdateState::DOWNLOADING) {
                notifySetupStatusViaBle(SetupStatusMessage::UPDATE_DOWNLOAD_START);
            }
            updateState_ = message->update_state().state();
        }
    }
}

bool SetupEndpoint::decryptCredentials(const proto::EncryptedSetupCredentialsMessage& msg,
                                       SetupSource source, std::vector<byte>& result) {
    try {
        result = decryptBleCredentials(msg, *deviceCryptography_);
        return true;
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("SetupEndpoint.FailedDecryptCredentials", "Failed to decrypt credentials: " << e.what());
        scheduleError(SetupStatusMessage::SOMETHING_WENT_WRONG, 0, Credentials());

        Json::Value event;
        event["exception"] = std::string(e.what());
        device_->telemetry()->reportEvent(getEventName(source, "CredentialsDecryptFail"),
                                          jsonToString(event));
        return false;
    }
}

std::string SetupEndpoint::getEventName(SetupSource source, const std::string& name) {
    switch (source) {
        case SetupSource::BLE:
            return "bleInit" + name;
        case SetupSource::SOUND:
            return "soundInit" + name;
        default:
            return "unknownInit" + name;
    };
}

Json::Value SetupEndpoint::getEventBody(const Credentials& credentials) {
    std::string networkType;

    if (credentials.isInitialized) {
        if (credentials.wifiType == WifiType::WIFI_TYPE_NONE) {
            networkType = "ethernet";
        } else {
            networkType = "wifi";
        }
    } else {
        networkType = "unknown";
    }

    Json::Value result;
    result["networkType"] = networkType;

    return result;
}

bool SetupEndpoint::handleExplicitWifiType(const Credentials& credentials, SetupSource source) {
    device_->telemetry()->reportEvent(getEventName(source, "DataHiddenNetwork"));
    latencyReporter_.createLatencyPoint(getEventName(source, "ConnectSent"));
    bool connected = requestConnect(credentials);

    device_->telemetry()->reportEvent(getEventName(source, "DataSuccessHidden"));
    if (connected) {
        YIO_LOG_INFO("Successfully initialized.");
        latencyReporter_.reportLatency(getEventName(source, "ConnectSent"),
                                       getEventName(source, "ConnectSuccessHidden"), true);
        return true;
    } else {
        YIO_LOG_INFO("Unsuccessfully initialized.");
        latencyReporter_.reportLatency(getEventName(source, "ConnectSent"),
                                       getEventName(source, "ConnectFailHidden"), true);
        return false;
    }
}

bool SetupEndpoint::handleUnknownWifiType(Credentials& credentials, SetupSource source) {
    device_->telemetry()->reportEvent(getEventName(source, "DataVisibleNetwork"));
    ipc::SharedMessage response;
    for (int i = 0; i < RETRIES_COUNT; ++i) {
        try {
            proto::QuasarMessage msg;
            msg.mutable_wifi_list_request();
            response = wifiConnector_->sendRequestSync(std::move(msg), std::chrono::seconds(5));

            break;
        } catch (const std::exception& e) {
            YIO_LOG_WARN("Can't get wifi list response in " << i << " attempts." << e.what());
        }
    }
    if (!response || !response->IsInitialized() || !response->has_wifi_list()) {
        YIO_LOG_ERROR_EVENT("SetupEndpoint.FailedGetWifiList", "Can't get wifi list response");
        scheduleError(SetupStatusMessage::CANNOT_CONNECT_TO_THE_WIFI, MIN_ERROR_TIMEOUT_MS, credentials);

        device_->telemetry()->reportEvent(getEventName(source, "DataWifiNetworksNotFound"));
        return false;
    }

    try {
        credentials.fillSSIDs(response->wifi_list().hotspots());
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("SetupEndpoint.FailedFillSSIDs", "exception &: " << e.what());
        Json::Value event;
        event["exception"] = std::string(e.what());
        device_->telemetry()->reportEvent(getEventName(source, "DataWifiSsidNotFound"), jsonToString(event));
        scheduleError(SetupStatusMessage::CANNOT_CONNECT_TO_THE_WIFI, MIN_ERROR_TIMEOUT_MS, credentials);
        return false;
    }

    if (credentials.SSIDs.empty()) {
        YIO_LOG_ERROR_EVENT("SetupEndpoint.MissingWifiCredentials", "Cannot find SSID with " << credentials.SSIDHashCode);
        scheduleError(SetupStatusMessage::CANNOT_CONNECT_TO_THE_WIFI, MIN_ERROR_TIMEOUT_MS, credentials);
        device_->telemetry()->reportEvent(getEventName(source, "DataSSIDListEmpty"));
        return false;
    }
    latencyReporter_.createLatencyPoint(getEventName(source, "ConnectSent"));
    const bool connected = requestConnect(credentials);
    device_->telemetry()->reportEvent(getEventName(source, "DataSuccessVisible"));
    if (connected) {
        YIO_LOG_INFO("Successfully initialized.");
        latencyReporter_.reportLatency(getEventName(source, "ConnectSent"),
                                       getEventName(source, "ConnectSuccessVisible"), true);
        return true;
    } else {
        YIO_LOG_INFO("Unsuccessfully initialized.");
        latencyReporter_.reportLatency(getEventName(source, "ConnectSent"),
                                       getEventName(source, "ConnectFailVisible"), true);
        return false;
    }
}

void SetupEndpoint::onDataReceived(const std::vector<byte>& bytes, SetupSource source) {
    device_->telemetry()->reportEvent(getEventName(source, "DataReceived"));
    latencyReporter_.reportLatency(getEventName(source, "TransferStart"),
                                   getEventName(source, "TransferEnd"), false);

    bool success = false;
    Credentials credentials;

    try {
        const std::string bytes_string = bytesToString(bytes);
        const std::string digest = calcSHA1Digest(bytes_string);

        YIO_LOG_DEBUG("Digest = " << digest);

        if (const auto optCommand = secretCommands_.getCommand(digest); optCommand.has_value()) {
            YIO_LOG_INFO("Secret string received");
            device_->telemetry()->reportEvent(getEventName(source, "AdminStringReceived"));
            /* Found secret string. Show Error leds to avoid any wrong behavior */
            deviceContext_.fireSetupError();

            const auto& command = optCommand.value();
            YIO_LOG_INFO("Running cmd: " << command.command << " is runtime: " << command.isRuntime);
            if (command.isRuntime) {
                deviceContext_.fireSoundCommand(command.command);
            } else {
                std::ignore = system(command.command.data());
            }
            success = true;
        } else {
            credentials = parseInitData(bytes, source);
            success = connectWithCredentials(credentials, source);
        }
    } catch (const std::exception& e) {
        Json::Value event;
        event["exception"] = std::string(e.what());
        device_->telemetry()->reportError(getEventName(source, "InputHandlingError"), jsonToString(event));
    }

    proto::QuasarMessage message;
    message.mutable_setup_credentials_handling_completed();
    server_->sendToAll(std::move(message));

    if (success) {
        device_->telemetry()->reportEvent(getEventName(source, "DataSuccess"), jsonToString(getEventBody(credentials)));
        latencyReporter_.reportLatency(getEventName(source, "TransferStart"),
                                       getEventName(source, "ActivationSuccess"), true);
    } else {
        device_->telemetry()->reportEvent(getEventName(source, "DataError"), jsonToString(getEventBody(credentials)));
        latencyReporter_.reportLatency(getEventName(source, "TransferStart"),
                                       getEventName(source, "ActivationError"), true);
    }
}

bool SetupEndpoint::connectWithCredentials(Credentials credentials, SetupSource source) {
    if (credentials.isInitialized) {
        if (credentials.wifiType == WifiType::WIFI_TYPE_UNKNOWN) {
            return handleUnknownWifiType(credentials, source);
        } else {
            return handleExplicitWifiType(credentials, source);
        }
    }

    return false;
}

Credentials SetupEndpoint::parseInitData(const std::vector<byte>& bytes, SetupSource source) {
    try {
        auto credentials = SetupParser::parseInitData(bytes);
        device_->telemetry()->reportEvent(getEventName(source, "DataParsedSuccessfully"), jsonToString(getEventBody(credentials)));
        YIO_LOG_INFO("Successful parsing: " << credentials.toString());
        return credentials;
    } catch (const std::exception& e) {
        YIO_LOG_ERROR_EVENT("SetupEndpoint.FailedParseInitData", "std::exception & caught during parsing init data. " << e.what());
        Json::Value event;
        event["exception"] = std::string(e.what());
        device_->telemetry()->reportEvent(getEventName(source, "DataParsingError"), jsonToString(event));
        scheduleError(SetupStatusMessage::SOMETHING_WENT_WRONG, 0, Credentials{});
    }
    return Credentials{};
}

bool SetupEndpoint::requestConnect(const Credentials& credentials) {
    const int port = getInt(device_->configuration()->getServiceConfig("firstrund"), "httpPort");
    const std::string baseUrl = "localhost:" + std::to_string(port);
    HttpClient client("local-firstrun-api", device_);
    client.setTimeout(std::chrono::seconds{getInt(device_->configuration()->getServiceConfig("sound_initd"), "firstrundHttpClientTimeoutSec")});

    Json::Value connect;
    auto& ssid = connect["ssid"];
    ssid = Json::arrayValue;
    for (size_t i = 0; i < credentials.SSIDs.size(); ++i) {
        ssid[(int)i] = credentials.SSIDs[i];
    }
    connect["password"] = credentials.password;
    connect["xtoken_code"] = credentials.tokenCode;

    connect["wifi_type"] = WifiType::WIFI_TYPES[credentials.wifiType];
    connect["plain"] = true;
    connect["stopAccessPoint"] = false;

    YIO_LOG_INFO("Sending wifi connect request to firstrund: " << jsonToString(connect));
    const auto response = client.post("connect", baseUrl + "/connect", jsonToString(connect));

    if (response.responseCode == 200 && parseJson(response.body)["status"].asString() == "ok") {
        return true;
    } else {
        YIO_LOG_ERROR_EVENT("SetupEndpoint.FailedConnectRequest", "Failed to connect to wifi with supported credentials. Firstrun connect response: " << response.body);
        return false;
    }
}

void SetupEndpoint::scheduleError(SetupStatusMessage::SetupStatus status, long timeoutMs, const Credentials& credentials) {
    YIO_LOG_INFO("scheduleError: " << status);
    std::this_thread::sleep_for(std::chrono::milliseconds(timeoutMs));
    deviceContext_.fireSetupError();
    filePlayerCapability_->playSoundFile(getSoundName(status), proto::AudioChannel::DIALOG_CHANNEL);
    notifySetupStatusViaBle(status);

    auto eventBody = getEventBody(credentials);
    eventBody["status"] = status;

    /* Event name should be the same as the one in FirstRunEndpoint where we don't know
     * whether we are configuring by Sound or by BLE */
    device_->telemetry()->reportEvent("initConnectError", jsonToString(eventBody));
}

std::string SetupEndpoint::getSoundName(SetupStatusMessage::SetupStatus status) {
    switch (status) {
        case SetupStatusMessage::SOMETHING_WENT_WRONG:
            return "something_went_wrong.wav";
        case SetupStatusMessage::CANNOT_CONNECT_TO_THE_WIFI:
            return "cannot_connect_to_the_wifi.wav";
        default:
            return "something_went_wrong.wav";
    }
}

void SetupEndpoint::notifySetupStatusViaBle(SetupStatusMessage::SetupStatus status) {
    if (device_->hal()->getBluetoothCapabilities().hasBle()) {
        proto::QuasarMessage message;
        message.mutable_setup_status_message()->set_status(status);
        bleInitdConnector_->sendMessage(std::move(message));
    }
}
