#include "stereo_pair_endpoint.h"

#include <yandex_io/libs/base/named_callback_queue.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/protobuf_utils/debug.h>
#include <yandex_io/libs/telemetry/telemetry.h>
#include <yandex_io/sdk/private/device_context.h>

using namespace quasar;
using namespace YandexIO;

YIO_DEFINE_LOG_MODULE("stereo_pair");

namespace {
    constexpr auto SEND_METRICA_PEREOD = std::chrono::seconds{120};
    constexpr auto SEND_METRICA_DELAY = std::chrono::seconds{2};
    constexpr auto MAX_HEARTBEAT_STATE_TIMEOUT = std::chrono::seconds{47};
    constexpr auto IS_CONNECTED_TIMEOUT = 2 * MAX_HEARTBEAT_STATE_TIMEOUT;
    constexpr auto TWOWAY_CONNECTIVITY_CHECK_PEREOD = IS_CONNECTED_TIMEOUT + std::chrono::seconds{1};
    constexpr auto HEARTBEAT_STATE_PEREOD = std::chrono::seconds{15};
    constexpr auto INITIAL_PAIRING_PERIOD = std::chrono::seconds{7};
    constexpr auto EVENT_VOLUME_CHANGED_THRESHOLD = std::chrono::seconds{10};
    const char* PHRASE_NO_CONNECTION_WITH_LEADER = "Сдается мне, я потеряла свою стереопару. Пожалуйста, проверьте подключение к сети или разъедините и снова соедините нас в приложении Яндекса.";
    const char* PHRASE_STEREO_PLAYER_NOT_READY_LEFT = "Одну минуту, я проверяю соединение со своей левой колонкой. Спросите меня чуть позже, пожалуйста.";
    const char* PHRASE_STEREO_PLAYER_NOT_READY_RIGHT = "Одну минуту, я проверяю соединение со своей правой колонкой. Спросите меня чуть позже, пожалуйста.";
    const char* PHRASE_STEREO_PLAYER_NOT_READY_ALL = "Одну минуту, я проверяю соединение с другой колонкой. Спросите меня чуть позже, пожалуйста.";

    proto::StereoPair::Role convertRole(StereoPairEndpoint::Role role)
    {
        switch (role) {
            case StereoPairEndpoint::Role::STANDALONE:
                return proto::StereoPair::STANDALONE;
            case StereoPairEndpoint::Role::LEADER:
                return proto::StereoPair::LEADER;
            case StereoPairEndpoint::Role::FOLLOWER:
                return proto::StereoPair::FOLLOWER;
            case StereoPairEndpoint::Role::UNDEFINED:
                break;
        }
        throw std::runtime_error("Unexpected Role role=" + std::to_string((int)role));
    }

    Json::Value createRepeatEvent(const std::string& text) {
        auto json = parseJson(
            R"({
            "name": "update_form",
            "payload": {
                "form_update": {
                "name": "personal_assistant.scenarios.quasar.iot.repeat_phrase",
                "slots": [
                    {
                    "name": "phrase_to_repeat",
                    "optional": false,
                    "type": "string",
                    "value": ""
                    }
                ]
                },
                "resubmit": true
            },
            "type": "server_action"
        })");
        json["payload"]["form_update"]["slots"][0]["value"] = text;

        return json;
    }

    Json::Value createStereoPairReadyEvent() {
        return parseJson(R"({
            "name": "@@mm_semantic_frame",
            "payload": {
                "typed_semantic_frame": {
                    "media_play_semantic_frame": {
                        "tune_id": {
                            "string_value": "stereopair_ready"
                        },
                        "location_id": {
                            "string_value": "stereopair"
                        }
                    }
                },
                "analytics": {
                    "origin": "SmartSpeaker",
                    "purpose": "stereopair"
                }
            },
            "type": "server_action"
        })");
    }

} // namespace

StereoPairEndpoint::StereoPairEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IClockTowerProvider> clockTowerProvider,
    std::shared_ptr<IGlagolClusterProvider> glagolCluster,
    std::shared_ptr<ISpectrogramAnimationProvider> spectrogramAnimationProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider,
    std::shared_ptr<IVolumeManagerProvider> volumeManagerProvider,
    std::shared_ptr<YandexIO::SDKInterface> sdk)
    : StereoPairEndpoint(
          std::move(device),
          std::move(clockTowerProvider),
          std::move(glagolCluster),
          std::move(spectrogramAnimationProvider),
          std::move(userConfigProvider),
          std::move(volumeManagerProvider),
          ipcFactory->createIpcServer("stereo_pair"),
          std::make_shared<NamedCallbackQueue>("stereo_pair"),
          std::move(sdk))
{
}

StereoPairEndpoint::StereoPairEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<IClockTowerProvider> clockTowerProvider,
    std::shared_ptr<IGlagolClusterProvider> glagolCluster,
    std::shared_ptr<ISpectrogramAnimationProvider> spectrogramAnimationProvider,
    std::shared_ptr<IUserConfigProvider> userConfigProvider,
    std::shared_ptr<IVolumeManagerProvider> volumeManagerProvider,
    std::shared_ptr<ipc::IServer> stereoPairServer,
    std::shared_ptr<ICallbackQueue> lifecycle,
    std::shared_ptr<YandexIO::SDKInterface> sdk)
    : device_(std::move(device))
    , clockTowerProvider_(std::move(clockTowerProvider))
    , glagolCluster_(std::move(glagolCluster))
    , spectrogramAnimationProvider_(std::move(spectrogramAnimationProvider))
    , volumeManagerProvider_(std::move(volumeManagerProvider))
    , stereoPairServer_(std::move(stereoPairServer))
    , lifecycle_(std::move(lifecycle))
    , hearbeatCallback_(lifecycle_)
    , twowayConnectivityCheckCallback_(lifecycle_)
    , initialPairingCallback_(lifecycle_)
    , metricaCallback_(lifecycle_)
    , clockTowerState_(ClockTowerState::createDefault())
    , sdk_(std::move(sdk))
{
    Y_VERIFY(clockTowerProvider_);

    stereoPairServer_->setClientConnectedHandler(makeSafeCallback(
        [this](auto& client) {
            lifecycle_->add([this, client = client.share()] { onClientConnected(client); }, lifetime_);
        }, lifetime_));

    stereoPairServer_->setMessageHandler(makeSafeCallback(
        [this](const auto& message, auto& client) {
            lifecycle_->add([this, message, client = client.share()] { onMessageReceived(client, message); }, lifetime_);
        }, lifetime_));

    userConfigProvider->jsonChangedSignal(IUserConfigProvider::ConfigScope::MERGED, "stereo_pair")
        .connect([this](const auto& json) {
            onUserConfigChanged(json);
        }, lifetime_, lifecycle_);

    if (spectrogramAnimationProvider_) {
        spectrogramAnimationProvider_->spectrogramAnimationState()
            .connect([this](const auto& state) {
                if (role_ == Role::LEADER) {
                    ++syncId_;
                }
                if (role_ != Role::FOLLOWER && state->source == SpectrogramAnimationState::Source::EXTERNAL) {
                    spectrogramAnimationProvider_->setExternalPresets("", "", "");
                }
                spectrogramAnimationState_ = state;
                hearbeatCallback_.execute([this] { sendStateHeartbeat(true); }, lifetime_);
            }, lifetime_, lifecycle_);
    }

    if (volumeManagerProvider_) {
        volumeManagerProvider_->volumeManagerState()
            .connect([this](const auto& state) {
                if (state->source != partnerDeviceId_) {
                    if (role_ == Role::LEADER || syncId_ > 0) {
                        ++syncId_;
                    }
                }
                volumeManagerState_ = state;
                hearbeatCallback_.execute([this] { sendStateHeartbeat(true); }, lifetime_);
            }, lifetime_, lifecycle_);
    }

    clockTowerProvider_->clockTowerState()
        .connect([this](const auto& state) {
            clockTowerState_ = state;
            updateStereoPlayerReady();
        }, lifetime_, lifecycle_);

    glagolCluster_->deviceList()
        .connect([this](auto deviceList) {
            onDeviceList(deviceList);
        }, lifetime_, lifecycle_);

    stereoPairServer_->listenService();

    lifecycle_->add(
        [this] {
            updateStereoPlayerReady();
            heartbeatReady_ = true;
            hearbeatCallback_.execute([this] { sendStateHeartbeat(true); }, lifetime_);
        }, lifetime_);
}

StereoPairEndpoint::~StereoPairEndpoint()
{
    lifetime_.die();
}

void StereoPairEndpoint::onDeviceList(const std::shared_ptr<const std::vector<std::string>>& deviceList)
{
    Y_ENSURE_THREAD(lifecycle_);
    if (!partnerDeviceId_.empty() && !isConnected()) {
        for (const auto& deviceId : *deviceList) {
            if (deviceId == partnerDeviceId_) {
                auto state = makeStateMessage();
                YIO_LOG_DEBUG("Device list changed. Sending own state to " << partnerDeviceId_ << ": " << state);
                glagolCluster_->send(partnerDeviceId_, "stereo_pair", state);
            }
        }
    }
}

void StereoPairEndpoint::onClientConnected(const std::shared_ptr<ipc::IServer::IClientConnection>& client)
{
    Y_ENSURE_THREAD(lifecycle_);
    auto state = makeStateMessage();
    YIO_LOG_DEBUG("Client connected. Sending own state: " << state);
    client->send(state);
}

void StereoPairEndpoint::onMessageReceived(const std::shared_ptr<ipc::IServer::IClientConnection>& client, const ipc::SharedMessage& message)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (message->has_alice_message()) {
        onMessageAlice(message->alice_message());
        return;
    }

    if (!message->has_stereo_pair_message()) {
        YIO_LOG_ERROR_EVENT("StereoPairEndpoint.UnexcpectedMessage", "Unexpected message for stereo pair controller");
        return;
    }
    const auto& spm = message->stereo_pair_message();

    if (spm.has_request_state()) {
        onMessageRequestState(client, spm.request_state());
    } else if (spm.has_state()) {
        onMessageState(spm.state());
    } else if (spm.has_start_conversation_request()) {
        onMessageStartConversationRequest(spm.start_conversation_request());
    } else if (spm.has_stop_conversation_request()) {
        onMessageStopConversationRequest(spm.stop_conversation_request());
    } else if (spm.has_toggle_conversation_request()) {
        onMessageToggleConversationRequest(spm.toggle_conversation_request());
    } else if (spm.has_finish_conversation_request()) {
        onMessageFinishConversationRequest(spm.finish_conversation_request());
    } else if (spm.has_initial_pairing_request()) {
        onMessageInitialPairingRequest(spm.initial_pairing_request());
    } else if (spm.has_initial_pairing_answer()) {
        onMessageInitialPairingAnswer(spm.initial_pairing_answer());
    } else if (spm.has_override_channel_request()) {
        onMessageOverrideChannelRequest(spm.override_channel_request());
    } else if (spm.has_speak_not_ready_notification_request()) {
        onMessageSpeakNotReadyNotificationRequest(spm.speak_not_ready_notification_request());
    } else if (spm.has_user_event_request()) {
        onMessageUserEventRequest(spm.user_event_request());
    } else if (spm.has_user_event_signal()) {
        onMessageUserEventSignal(message);
    } else {
        YIO_LOG_INFO("Unknown message for stereo pair controller: " << message);
    }
}

void StereoPairEndpoint::onUserConfigChanged(const std::shared_ptr<const Json::Value>& json)
{
    Y_ENSURE_THREAD(lifecycle_);

    YIO_LOG_INFO("Load stereo pair config: " << jsonToString(*json, true));

    auto oldPartnerDeviceId = partnerDeviceId_;
    auto oldRole = role_;
    auto oldChannel = channel_;

    partnerDeviceId_ = tryGetString(*json, "partnerDeviceId");
    if (partnerDeviceId_.empty()) {
        role_ = Role::STANDALONE;
    } else {
        auto roleName = tryGetString(*json, "role");
        if (roleName == "leader") {
            role_ = Role::LEADER;
        } else if (roleName == "follower") {
            role_ = Role::FOLLOWER;
        } else if (roleName.empty() || roleName == "standalone") {
            role_ = Role::STANDALONE;
        } else {
            YIO_LOG_ERROR_EVENT("StereoPairEndpoint.InvalidRoleName", "Invalie stereo pair role \"" << roleName << "\"");
        }
    }

    auto playJingle = tryGetBool(*json, "playJingle", true);
    if (playJingle_.exchange(playJingle) != playJingle) {
        YIO_LOG_INFO("Play jingle: " << playJingle);
    }

    auto nowaitSync = tryGetBool(*json, "nowaitSync", false);
    if (nowaitSync_.exchange(nowaitSync) != nowaitSync) {
        YIO_LOG_INFO("NowaitSync: " << nowaitSync);
    }

    channel_ = StereoPairState::parseChannel(tryGetString(*json, "channel"));
    if (channel_ == Channel::UNDEFINED) {
        channel_ = Channel::ALL;
    }

    if (oldPartnerDeviceId == partnerDeviceId_ && oldRole == role_ && oldChannel == channel_) {
        return;
    }

    YIO_LOG_INFO("Stereo pair configuration of this device changed: role=" << StereoPairState::roleName(role_) << ", partnerDeviceId=" << partnerDeviceId_);

    lastPartnerStateReceived_ = decltype(lastPartnerStateReceived_){};

    if (role_ != oldRole) {
        YIO_LOG_INFO("Stereo pair role changed from " << StereoPairState::roleName(oldRole) << " to " << StereoPairState::roleName(role_));
        auto oldIsStereoPair = (oldRole == Role::LEADER || oldRole == Role::FOLLOWER);
        auto newIsStereoPair = (role_ == Role::LEADER || role_ == Role::FOLLOWER);
        if (!newIsStereoPair) {
            startUpTime_ = decltype(startUpTime_){};
            awatingStereoPlayerReadyEvent_ = false;
        } else if (oldIsStereoPair != newIsStereoPair) {
            startUpTime_ = std::chrono::steady_clock::now();
        }

        if (role_ != Role::LEADER) {
            initialPairingUuid_.clear();
            initialPairingJingleUuid_.clear();
            initialPairingStarted_ = std::chrono::steady_clock::time_point{};
        }

        if (oldRole == Role::FOLLOWER) {
            if (spectrogramAnimationProvider_) {
                spectrogramAnimationProvider_->setExternalPresets("", "", "");
            }
        }

        if (role_ == Role::FOLLOWER) {
            syncId_ = 0;
            auto message = ipc::buildMessage([](auto& msg) {
                msg.mutable_stereo_pair_message()->mutable_request_state();
            });
            glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);
        } else if (role_ == Role::LEADER) {
            if (oldRole == Role::STANDALONE) {
                sendInitialParingRequest();
            }
            ++syncId_;
        }

        if (oldRole != Role::UNDEFINED) {
            YIO_LOG_INFO("Reset alice dialog, clear directive queue and turn off media");
            sdk_->getAliceCapability()->cancelDialogAndClearQueue();
        }
    }
    updateStereoPlayerReady();
    hearbeatCallback_.execute([this] { sendStateHeartbeat(true); }, lifetime_);
    metricaCallback_.executeDelayed([this] { sendMetrica(); }, SEND_METRICA_DELAY, lifetime_);
}

void StereoPairEndpoint::onMessageState(const proto::StereoPair::State& state)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::STANDALONE || role_ == Role::UNDEFINED || partnerDeviceId_.empty()) {
        YIO_LOG_ERROR_EVENT("StereoPairEndpoint.SettingsDesynchronization", "Ignore state from " << state.device_id() << " beacause this device in STANDALONE mode");
        return;
    }

    if (partnerDeviceId_ != state.device_id()) {
        YIO_LOG_ERROR_EVENT("StereoPairEndpoint.SettingsDesynchronization", "Ignore state from " << state.device_id() << " beacause parner missmatch");
        return;
    }

    if (convertRole(role_) == state.role()) {
        YIO_LOG_ERROR_EVENT("StereoPairEndpoint.SettingsDesynchronization", "Ignore state from " << state.device_id() << " beacause parner role is same");
        return;
    }
    bool changed = false;
    bool oldIsConnected = isConnected();
    bool oldPartnerPlayerRead = partnerPlayerReady_;
    bool oldTwowayConnectivity = twowayConnectivity_;

    lastPartnerStateReceived_ = std::chrono::steady_clock::now();
    twowayConnectivity_ = isConnected() && (state.connectivity() == proto::StereoPair::ONEWAY || state.connectivity() == proto::StereoPair::TWOWAY);

    changed = changed || (oldIsConnected != isConnected());

    if (role_ == Role::LEADER) {
        changed = onMessageStateForLeader(state) || changed;
    } else if (role_ == Role::FOLLOWER) {
        changed = onMessageStateForFollower(state) || changed;
    }

    partnerUptime_ = (state.has_uptime_ms() ? std::optional<int64_t>(state.uptime_ms()) : std::nullopt);
    partnerPlayerReady_ = (state.stereo_player_status() == proto::StereoPair::PLAYER_PARTNER_NOT_READY ||
                           state.stereo_player_status() == proto::StereoPair::PLAYER_READY);
    changed = changed || (oldPartnerPlayerRead != partnerPlayerReady_);

    if (changed) {
        hearbeatCallback_.execute([this] { sendStateHeartbeat(true); }, lifetime_);
    }

    if (oldTwowayConnectivity != twowayConnectivity_) {
        onTwowayConnectivityChanged();
        twowayConnectivityCheckCallback_.executeDelayed([this] { checkTwowayConnectivity(); }, TWOWAY_CONNECTIVITY_CHECK_PEREOD, lifetime_);
    } else if (awatingStereoPlayerReadyEvent_) {
        if (partnerPlayerReady_ && stereoPlayerReady_) {
            reportEvent("stereoPairStereoPlayerTimeout", {{"timeout", std::to_string(stereoPairUptimeMs())}});
            awatingStereoPlayerReadyEvent_ = false;
        }
    }
}

bool StereoPairEndpoint::onMessageStateForLeader(const proto::StereoPair::State& state)
{
    Y_ENSURE_THREAD(lifecycle_);

    bool changed = false;
    if (state.sync_id() > syncId_) {
        syncId_ = state.sync_id();
        if (volumeManagerProvider_ && state.has_volume() && !state.volume().source().empty() && state.volume().source() != device_->deviceId()) {
            auto vms = volumeManagerProvider_->volumeManagerState().value();
            bool mutedChanged = (state.volume().has_is_muted() && state.volume().is_muted() != vms->isMuted);
            if (state.platform() == device_->platform()) {
                if (mutedChanged || state.volume().has_platform_volume() && state.volume().platform_volume() != vms->platformVolume) {
                    volumeManagerProvider_->setPlatformVolume(state.volume().platform_volume(), state.volume().is_muted(), state.device_id());
                    reportEvent("stereoPairUseFollowerVolume", EVENT_VOLUME_CHANGED_THRESHOLD);
                }
            } else {
                if (mutedChanged || state.volume().has_alice_volume() && state.volume().alice_volume() != vms->aliceVolume) {
                    volumeManagerProvider_->setAliceVolume(state.volume().alice_volume(), state.volume().is_muted(), state.device_id());
                    reportEvent("stereoPairUseFollowerVolume", EVENT_VOLUME_CHANGED_THRESHOLD);
                }
            }
        }
        changed = true;
    }
    return changed;
}

bool StereoPairEndpoint::onMessageStateForFollower(const proto::StereoPair::State& state)
{
    Y_ENSURE_THREAD(lifecycle_);

    bool changed = false;
    if (state.sync_id() > syncId_) {
        syncId_ = state.sync_id();
        if (volumeManagerProvider_ && state.has_volume() && !state.volume().source().empty() && state.volume().source() != device_->deviceId()) {
            auto vms = volumeManagerProvider_->volumeManagerState().value();
            bool mutedChanged = (state.volume().has_is_muted() && state.volume().is_muted() != vms->isMuted);
            if (state.platform() == device_->platform()) {
                if (mutedChanged || state.volume().has_platform_volume() && state.volume().platform_volume() != vms->platformVolume) {
                    volumeManagerProvider_->setPlatformVolume(state.volume().platform_volume(), state.volume().is_muted(), state.device_id());
                    reportEvent("stereoPairUseLeaderVolume", EVENT_VOLUME_CHANGED_THRESHOLD);
                }
            } else {
                if (mutedChanged || state.volume().has_alice_volume() && state.volume().alice_volume() != vms->aliceVolume) {
                    volumeManagerProvider_->setAliceVolume(state.volume().alice_volume(), state.volume().is_muted(), state.device_id());
                    reportEvent("stereoPairUseLeaderVolume", EVENT_VOLUME_CHANGED_THRESHOLD);
                }
            }
        }

        if (spectrogramAnimationProvider_ && state.has_spectrogram()) {
            const auto& spectrogram = state.spectrogram();
            if (spectrogramAnimationState_->source != SpectrogramAnimationState::Source::EXTERNAL ||
                spectrogram.configs() != spectrogramAnimationState_->configs ||
                spectrogram.current() != spectrogramAnimationState_->current ||
                spectrogram.extra_data() != spectrogramAnimationState_->extraData) {
                YIO_LOG_INFO("Synchronize leader spectrogram animation: current=" << spectrogram.current() << ", extraData=" << spectrogram.extra_data());
                spectrogramAnimationProvider_->setExternalPresets(spectrogram.configs(), spectrogram.current(), spectrogram.extra_data());
            }
        }
        changed = true;
    }
    return changed;
}

void StereoPairEndpoint::onMessageRequestState(const std::shared_ptr<ipc::IServer::IClientConnection>& client, const proto::StereoPair::RequestState& rs)
{
    Y_ENSURE_THREAD(lifecycle_);
    auto state = makeStateMessage();
    if (rs.has_requester() && !rs.requester().empty()) {
        YIO_LOG_DEBUG("Request state from " << rs.requester() << ". Sending own state: " << state);
        glagolCluster_->send(rs.requester(), "stereo_pair", state);
    } else {
        YIO_LOG_DEBUG("Request state from client. Sending own state: " << state);
        client->send(state);
    }
}

void StereoPairEndpoint::onMessageStartConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest)
{
    Y_ENSURE_THREAD(lifecycle_);
    if (role_ == Role::LEADER) {
        reportEvent("stereoPairFollowerStartConversationInvoke", {{"requestId", conversationRequest.request_id()}});
        sdk_->getAliceCapability()->startConversation(VinsRequest::createHardwareButtonClickEventSource());
    }
}

void StereoPairEndpoint::onMessageStopConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest)
{
    Y_ENSURE_THREAD(lifecycle_);
    if (role_ == Role::LEADER) {
        reportEvent("stereoPairFollowerStopConversationInvoke", {{"requestId", conversationRequest.request_id()}});
        sdk_->getAliceCapability()->stopConversation();
    }
}

void StereoPairEndpoint::onMessageToggleConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest)
{
    Y_ENSURE_THREAD(lifecycle_);
    if (role_ == Role::LEADER) {
        reportEvent("stereoPairFollowerToggleConversationInvoke", {{"requestId", conversationRequest.request_id()}});
        sdk_->getAliceCapability()->toggleConversation(VinsRequest::createHardwareButtonClickEventSource());
    }
}

void StereoPairEndpoint::onMessageFinishConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest)
{
    Y_ENSURE_THREAD(lifecycle_);
    if (role_ == Role::LEADER) {
        reportEvent("stereoPairFollowerFinishConversationInvoke", {{"requestId", conversationRequest.request_id()}});
        sdk_->getAliceCapability()->finishConversationVoiceInput();
    }
}

void StereoPairEndpoint::onMessageInitialPairingRequest(const proto::StereoPair::InitialPairingRequest& initialPairingRequest)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::FOLLOWER &&
        device_->deviceId() == initialPairingRequest.follower_device_id() &&
        partnerDeviceId_ == initialPairingRequest.leader_device_id() &&
        isTwowayConnectivity()) {
        if (nowaitSync_ || (stereoPlayerReady_ && partnerPlayerReady_)) {
            auto message = ipc::buildMessage([&](auto& msg) {
                auto& answer = *msg.mutable_stereo_pair_message()->mutable_initial_pairing_answer();
                answer.set_follower_device_id(TString(device_->deviceId()));
                answer.set_uuid(TString(initialPairingRequest.uuid()));
            });
            sendStateHeartbeat(true);
            glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);
            YIO_LOG_INFO("Send initial pairing answer: uuid=" << initialPairingRequest.uuid());
        } else {
            YIO_LOG_INFO("Ignore initial pairing request because stereo player is not ready: uuid=" << initialPairingRequest.uuid());
        }
    }
}

void StereoPairEndpoint::onMessageInitialPairingAnswer(const proto::StereoPair::InitialPairingAnswer& initialPairingAnswer)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::LEADER &&
        initialPairingUuid_ == initialPairingAnswer.uuid() &&
        partnerDeviceId_ == initialPairingAnswer.follower_device_id()) {
        if (!nowaitSync_ && (!stereoPlayerReady_ || !partnerPlayerReady_)) {
            YIO_LOG_INFO("Ignore initial pairing answer because stereo player is not ready: uuid=" << initialPairingAnswer.uuid()
                                                                                                   << ", stereoPlayerReady=" << stereoPlayerReady_ << ", partnerPlayerReady=" << partnerPlayerReady_);
            return;
        }

        YIO_LOG_INFO("Receive initial pairing answer from partner " << initialPairingAnswer.follower_device_id() << ", uuid=" << initialPairingAnswer.uuid());
        auto initialPairingTime = std::chrono::steady_clock::now() - initialPairingStarted_;
        auto initialPairingJingleUuid = std::move(initialPairingJingleUuid_);
        initialPairingUuid_.clear();
        initialPairingJingleUuid_.clear();
        initialPairingStarted_ = std::chrono::steady_clock::time_point{};
        initialPairingTimepoint_ = std::chrono::system_clock::now();
        hearbeatCallback_.execute([this, initialPairingJingleUuid] {
            sendStateHeartbeat(true);
            Json::Value event;
            if (playJingle_ && partnerPlayerReady_) {
                event = createStereoPairReadyEvent();
            } else {
                event = createRepeatEvent("Поздравляю, стереопара создана.");
            }

            auto request = std::make_shared<VinsRequest>(std::move(event), VinsRequest::createSoftwareDirectiveEventSource(), initialPairingJingleUuid);
            sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);
        }, lifetime_);
        initialPairingCallback_.reset();
        Json::Value json;
        json["jingleUuid"] = initialPairingJingleUuid;
        json["initialPairingTimeMs"] = static_cast<int64_t>(std::chrono::duration_cast<std::chrono::milliseconds>(initialPairingTime).count());
        reportEvent("stereoPairFinishInitialPairing", std::move(json));
    }
}

void StereoPairEndpoint::onMessageOverrideChannelRequest(const proto::StereoPair::OverrideChannelRequest& overrideChannelRequest)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (!overrideChannelRequest.has_channel()) {
        YIO_LOG_WARN("OverrideChannelRequest.channel is not specified");
        return;
    }

    Channel overridedChannel = StereoPairState::parseChannel(overrideChannelRequest.channel());
    if (overridedChannel != overridedChannel_) {
        auto oldEffectiveChannel = effectiveChannel();
        YIO_LOG_INFO("Change overrided sound channel from " << StereoPairState::channelName(overridedChannel_) << " to " << StereoPairState::channelName(overridedChannel));
        overridedChannel_ = overridedChannel;
        auto newEffectiveChannel = effectiveChannel();
        if (oldEffectiveChannel != newEffectiveChannel) {
            YIO_LOG_INFO("Effective sound channel changed from " << StereoPairState::channelName(oldEffectiveChannel) << " to " << StereoPairState::channelName(newEffectiveChannel));
            hearbeatCallback_.execute([this] { sendStateHeartbeat(true); }, lifetime_);
        }
    }
}

void StereoPairEndpoint::onMessageSpeakNotReadyNotificationRequest(const proto::StereoPair::SpeakNotReadyNotificationReqest& speakNotReadyNotificationReqest)
{
    YIO_LOG_INFO("Stereo pair process \"speak not ready notification request\"");

    Json::Value event;
    switch (speakNotReadyNotificationReqest.reason()) {
        case proto::StereoPair::SpeakNotReadyNotificationReqest::NO_CONNECTION:
            event = createRepeatEvent(PHRASE_NO_CONNECTION_WITH_LEADER);
            reportEvent("stereoPairNoConnectionNotification", {{"source", std::string_view("no_connection")}});
            break;
        case proto::StereoPair::SpeakNotReadyNotificationReqest::PLAYER_NOT_READY:
            if (channel_ == Channel::LEFT) {
                event = createRepeatEvent(PHRASE_STEREO_PLAYER_NOT_READY_LEFT);
            } else if (channel_ == Channel::RIGHT) {
                event = createRepeatEvent(PHRASE_STEREO_PLAYER_NOT_READY_RIGHT);
            } else {
                event = createRepeatEvent(PHRASE_STEREO_PLAYER_NOT_READY_ALL);
            }
            reportEvent("stereoPairNoConnectionNotification", {{"source", std::string_view("player_not_ready")}});
            break;
        default:
            event = createRepeatEvent(PHRASE_NO_CONNECTION_WITH_LEADER);
            reportEvent("stereoPairNoConnectionNotification", {{"source", std::string_view("external")}});
            break;
    }

    auto request = std::make_shared<VinsRequest>(std::move(event), VinsRequest::createSoftwareDirectiveEventSource());
    sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);
}

void StereoPairEndpoint::onMessageUserEventRequest(const proto::StereoPair::UserEvent& userEvent)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::LEADER || role_ == Role::FOLLOWER) {
        auto message = ipc::buildMessage([&](auto& message) {
            message.mutable_stereo_pair_message()->mutable_user_event_signal()->CopyFrom(userEvent);
        });
        glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);
        if (twowayConnectivity_) {
            YIO_LOG_INFO("Send user event: id=" << userEvent.event_id() << ", payload=" << userEvent.payload());
            reportEvent("stereoPairUserEvent", {{"eventId", userEvent.event_id()}, {"success", "true"}});
        } else {
            YIO_LOG_INFO("Try to send user event: id=" << userEvent.event_id() << ", payload=" << userEvent.payload());
            reportEvent("stereoPairUserEvent", {{"eventId", userEvent.event_id()}, {"success", "false"}});
        }
    }
}

void StereoPairEndpoint::onMessageUserEventSignal(const ipc::SharedMessage& message)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::LEADER || role_ == Role::FOLLOWER) {
        const auto& userEvent = message->stereo_pair_message().user_event_signal();
        YIO_LOG_INFO("Receive user event: id=" << userEvent.event_id() << ", payload=" << userEvent.payload());
        stereoPairServer_->sendToAll(message);
    }
}

void StereoPairEndpoint::onMessageAlice(const proto::Alice& message)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (message.has_start_conversation_signal()) {
        startConversationOnLeader();
    } else if (message.has_stop_conversation_signal()) {
        stopConversationOnLeader();
    } else if (message.has_toggle_conversation_signal()) {
        toggleConversationOnLeader();
    } else if (message.has_finish_conversation_signal()) {
        finishConversationOnLeader();
    }
}

void StereoPairEndpoint::onTwowayConnectivityChanged()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (twowayConnectivity_) {
        YIO_LOG_INFO("Twoway connectivity established.");
        if (role_ == Role::LEADER) {
            auto timeoutMs = std::to_string(stereoPairUptimeMs());
            reportEvent("stereoPairTwowayConnectivityTimeout", {{"timeout", timeoutMs}});
            if (partnerPlayerReady_ && stereoPlayerReady_) {
                reportEvent("stereoPairStereoPlayerTimeout", {{"timeout", timeoutMs}});
                awatingStereoPlayerReadyEvent_ = false;
            } else {
                awatingStereoPlayerReadyEvent_ = true;
            }
        }
    } else {
        YIO_LOG_INFO("Twoway connectivity lost. My state is " << (isConnected() ? "connected" : "not connected"));
        awatingStereoPlayerReadyEvent_ = false;
        startUpTime_ = std::chrono::steady_clock::now();
    }
    metricaCallback_.executeDelayed([this] { sendMetrica(); }, SEND_METRICA_DELAY, lifetime_);
}

bool StereoPairEndpoint::isConnected() const {
    Y_ENSURE_THREAD(lifecycle_);
    return role_ != Role::STANDALONE && role_ != Role::UNDEFINED && !partnerDeviceId_.empty() && (std::chrono::steady_clock::now() - lastPartnerStateReceived_ < IS_CONNECTED_TIMEOUT);
}

bool StereoPairEndpoint::isTwowayConnectivity() const {
    Y_ENSURE_THREAD(lifecycle_);
    return twowayConnectivity_ && isConnected();
}

ipc::SharedMessage StereoPairEndpoint::makeStateMessage() const {
    Y_ENSURE_THREAD(lifecycle_);

    return ipc::buildMessage([this](auto& message) {
        auto& state = *message.mutable_stereo_pair_message()->mutable_state();
        state.set_device_id(TString(device_->deviceId()));
        state.set_platform(TString(device_->platform()));
        if (role_ != Role::UNDEFINED) {
            state.set_role(convertRole(role_));
        }

        switch (effectiveChannel()) {
            case StereoPairState::Channel::UNDEFINED:
            case StereoPairState::Channel::ALL:
                state.set_channel(proto::StereoPair::CH_ALL);
                break;
            case StereoPairState::Channel::RIGHT:
                state.set_channel(proto::StereoPair::CH_RIGHT);
                break;
            case StereoPairState::Channel::LEFT:
                state.set_channel(proto::StereoPair::CH_LEFT);
                break;
        }

        if (role_ == Role::STANDALONE || role_ == Role::UNDEFINED) {
            state.set_connectivity(proto::StereoPair::INAPPLICABLE);
        } else {
            if (startUpTime_ != decltype(startUpTime_){}) {
                state.set_uptime_ms(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startUpTime_).count());
            }
            state.set_connectivity(isTwowayConnectivity()
                                       ? proto::StereoPair::TWOWAY
                                       : (isConnected() ? proto::StereoPair::ONEWAY : proto::StereoPair::NO_CONNECTION));
        }
        state.set_partner_device_id(TString(partnerDeviceId_));

        if (role_ == Role::STANDALONE || role_ == Role::UNDEFINED) {
            state.set_stereo_player_status(proto::StereoPair::PLAYER_UNDEFINED);
        } else if (!stereoPlayerReady_) {
            state.set_stereo_player_status(proto::StereoPair::PLAYER_NO_SYNC);
        } else if (!partnerPlayerReady_ || !isTwowayConnectivity()) {
            state.set_stereo_player_status(proto::StereoPair::PLAYER_PARTNER_NOT_READY);
        } else {
            state.set_stereo_player_status(proto::StereoPair::PLAYER_READY);
        }

        state.set_sync_id(syncId_);
        if (initialPairingTimepoint_ != std::chrono::system_clock::time_point{}) {
            auto timeMs = std::chrono::duration_cast<std::chrono::milliseconds>(initialPairingTimepoint_.time_since_epoch()).count();
            state.set_initial_pairing_time_ms(timeMs);
        }

        if (spectrogramAnimationState_) {
            auto& spectrogram = *state.mutable_spectrogram();
            spectrogram.set_source(
                spectrogramAnimationState_->source == SpectrogramAnimationState::Source::EXTERNAL
                    ? proto::SpectrogramAnimation::State::EXTERNAL
                    : proto::SpectrogramAnimation::State::LOCAL);
            spectrogram.set_configs(TString(spectrogramAnimationState_->configs));
            spectrogram.set_current(TString(spectrogramAnimationState_->current));
            spectrogram.set_extra_data(TString(spectrogramAnimationState_->extraData));
        }

        if (volumeManagerState_) {
            auto& volume = *state.mutable_volume();
            volume.set_platform_volume(volumeManagerState_->platformVolume);
            volume.set_alice_volume(volumeManagerState_->aliceVolume);
            volume.set_is_muted(volumeManagerState_->isMuted);
            volume.set_source(TString(volumeManagerState_->source));
            volume.set_set_bt_volume(volumeManagerState_->setBtVolume);
        }
    });
}

void StereoPairEndpoint::sendStateHeartbeat(bool verbose)
{
    Y_ENSURE_THREAD(lifecycle_);

    if (!heartbeatReady_) {
        return;
    }

    auto state = makeStateMessage();
    stereoPairServer_->sendToAll(state);
    if (!partnerDeviceId_.empty()) {
        if (verbose) {
            YIO_LOG_INFO("Heartbeat of stereo pair. Sending own state: " << state);
        } else {
            YIO_LOG_DEBUG("Heartbeat of stereo pair. Sending own state: " << state);
        }
        glagolCluster_->send(partnerDeviceId_, "stereo_pair", state);
        /*
         * Reschedule sendStateHeartbeat only in StereoPair mode but not in StandAlone
         */
        hearbeatCallback_.executeDelayed([this] { sendStateHeartbeat(false); }, HEARTBEAT_STATE_PEREOD, lifetime_);
    }
}

void StereoPairEndpoint::checkTwowayConnectivity()
{
    Y_ENSURE_THREAD(lifecycle_);

    auto oldTwowayConnectivity = twowayConnectivity_;
    twowayConnectivity_ = twowayConnectivity_ && isConnected();
    if (twowayConnectivity_ != oldTwowayConnectivity) {
        onTwowayConnectivityChanged();
    }
    twowayConnectivityCheckCallback_.executeDelayed([this] { checkTwowayConnectivity(); }, TWOWAY_CONNECTIVITY_CHECK_PEREOD, lifetime_);
}

void StereoPairEndpoint::sendInitialParingRequest()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ != Role::LEADER) {
        return;
    }

    if (initialPairingStarted_ == std::chrono::steady_clock::time_point{}) {
        initialPairingStarted_ = std::chrono::steady_clock::now();
        initialPairingJingleUuid_ = makeUUID();
        Json::Value json;
        json["jingleUuid"] = initialPairingJingleUuid_;
        reportEvent("stereoPairStartInitialPairing", std::move(json));
    }

    if (isTwowayConnectivity() && stereoPlayerReady_) {
        initialPairingUuid_ = makeUUID();

        auto message = ipc::buildMessage([&](auto& msg) {
            auto& req = *msg.mutable_stereo_pair_message()->mutable_initial_pairing_request();
            req.set_leader_device_id(TString(device_->deviceId()));
            req.set_follower_device_id(TString(partnerDeviceId_));
            req.set_uuid(TString(initialPairingUuid_));
        });
        sendStateHeartbeat(true);
        glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);

        YIO_LOG_INFO("Send initial pairing request: uuid=" << initialPairingUuid_);
    } else {
        YIO_LOG_INFO("We are waiting for our device to be ready for full-fledged operation in stereo pair mode: "
                     << "isTwowayConnectivity=" << isTwowayConnectivity() << ", stereoPlayerReady=" << stereoPlayerReady_);
    }

    initialPairingCallback_.executeDelayed([this] { sendInitialParingRequest(); }, INITIAL_PAIRING_PERIOD, lifetime_);
}

StereoPairEndpoint::Channel StereoPairEndpoint::effectiveChannel() const {
    Y_ENSURE_THREAD(lifecycle_);

    return (overridedChannel_ != Channel::UNDEFINED ? overridedChannel_ : channel_);
}

int64_t StereoPairEndpoint::stereoPairUptimeMs() const {
    // For older versions, the uptime is calculated by its own time, and for the latest
    // firmwares, the minimum time from two columns is taken, i.e. lifetime of working SP
    Y_ENSURE_THREAD(lifecycle_);

    if (startUpTime_ == decltype(startUpTime_){} || !(role_ == Role::LEADER || role_ == Role::FOLLOWER)) {
        return 0;
    }
    int64_t myUptime = std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - startUpTime_).count();
    return (partnerUptime_ ? std::min<int64_t>(*partnerUptime_, myUptime) : myUptime);
}

void StereoPairEndpoint::updateStereoPlayerReady()
{
    Y_ENSURE_THREAD(lifecycle_);

    bool ready = false;
    if (role_ == Role::LEADER) {
        ready = true;
    } else if (!partnerDeviceId_.empty()) {
        if (const auto partnerClock = clockTowerState_->findClockByDeviceId(partnerDeviceId_)) {
            ready = (partnerClock->syncLevel() >= IClock::SyncLevel::STRONG);
        }
    }

    if (ready == stereoPlayerReady_) {
        return;
    }
    stereoPlayerReady_ = ready;
    hearbeatCallback_.execute([this] { sendStateHeartbeat(true); }, lifetime_);
    metricaCallback_.executeDelayed([this] { sendMetrica(); }, SEND_METRICA_DELAY, lifetime_);
    YIO_LOG_INFO("Stereo player status changed to " << (stereoPlayerReady_ ? "READY" : "NOT READY"));
}

void StereoPairEndpoint::sendMetrica() {
    if (role_ == Role::LEADER || role_ == Role::FOLLOWER) {
        Json::Value values;
        Json::Value& clocks = values["clocks"];
        for (const auto& [clockId, remoteClock] : clockTowerState_->remoteClocks) {
            if (remoteClock->syncLevel() > IClock::SyncLevel::NONE && !remoteClock->expired()) {
                Json::Value p = clocks[clockId];
                p["deviceId"] = remoteClock->deviceId();
                p["peer"] = remoteClock->peer();
                p["syncLevel"] = remoteClock->syncLevelAsText();
            }
        }
        reportEvent("stereoPairClocks", std::move(values));
        metricaCallback_.executeDelayed([this] { sendMetrica(); }, SEND_METRICA_PEREOD, lifetime_);
    }
}

void StereoPairEndpoint::reportEvent(const std::string& eventName, Json::Value values)
{
    Y_ENSURE_THREAD(lifecycle_);

    auto& ctx = values["ctx"];
    ctx["role"] = StereoPairState::roleName(role_);
    ctx["partnerDeviceId"] = partnerDeviceId_;
    ctx["channel"] = StereoPairState::channelName(channel_);
    if (overridedChannel_ != Channel::UNDEFINED) {
        ctx["overridedChannel"] = StereoPairState::channelName(overridedChannel_);
    }
    ctx["stereoPlayerReady"] = stereoPlayerReady_;
    ctx["partnerPlayerReady"] = partnerPlayerReady_;
    if (role_ == Role::STANDALONE || role_ == Role::UNDEFINED) {
        ctx["connectivity"] = StereoPairState::connectivityName(StereoPairState::Connectivity::INAPPLICABLE);
    } else {
        ctx["connectivity"] = StereoPairState::connectivityName(
            isTwowayConnectivity()
                ? StereoPairState::Connectivity::TWOWAY
                : (isConnected() ? StereoPairState::Connectivity::ONEWAY : StereoPairState::Connectivity::NO_CONNECTION));
    }
    device_->telemetry()->reportEvent(eventName, jsonToString(values));

    lastReportedEvents_[eventName] = std::chrono::steady_clock::now();
    YIO_LOG_DEBUG("REPORT EVENT: event=" << eventName << ", json=" << jsonToString(values));
}

void StereoPairEndpoint::reportEvent(const std::string& eventName, std::map<std::string_view, std::string_view> values)
{
    Y_ENSURE_THREAD(lifecycle_);

    Json::Value json;
    for (const auto& [key, value] : values) {
        json[std::string(key)] = std::string(value);
    }
    reportEvent(eventName, std::move(json));
}

void StereoPairEndpoint::reportEvent(const std::string& eventName, std::chrono::milliseconds threshold)
{
    Y_ENSURE_THREAD(lifecycle_);

    auto it = lastReportedEvents_.find(eventName);
    if (it != lastReportedEvents_.end()) {
        if (it->second + threshold > std::chrono::steady_clock::now()) {
            lastReportedEvents_[eventName] = std::chrono::steady_clock::now();
            return;
        }
    }
    reportEvent(eventName, Json::Value{});
}

void StereoPairEndpoint::startConversationOnLeader()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::FOLLOWER) {
        auto requestId = makeUUID();
        if (twowayConnectivity_) {
            YIO_LOG_INFO("Send directive to start conversation on leader device");
            reportEvent("stereoPairFollowerStartConversationRequest", {{"requestId", requestId}, {"success", "true"}});
        } else {
            YIO_LOG_INFO("Try to send directive to start conversation on leader device (no twoway connectivity)");
            reportEvent("stereoPairFollowerStartConversationRequest", {{"requestId", requestId}, {"success", "false"}});
            reportEvent("stereoPairNoConnectionNotification", {{"source", std::string_view("start_conversation")}});

            auto request = std::make_shared<VinsRequest>(createRepeatEvent(PHRASE_NO_CONNECTION_WITH_LEADER), VinsRequest::createSoftwareDirectiveEventSource());
            sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);

            return;
        }
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_stereo_pair_message()->mutable_start_conversation_request()->set_request_id(std::move(requestId));
        });
        glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);
    }
}

void StereoPairEndpoint::stopConversationOnLeader()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::FOLLOWER) {
        auto requestId = makeUUID();
        if (twowayConnectivity_) {
            YIO_LOG_INFO("Send directive to stop conversation on leader device");
            reportEvent("stereoPairFollowerStopConversationRequest", {{"requestId", requestId}, {"success", "true"}});
        } else {
            YIO_LOG_INFO("Try to send directive to stop conversation on leader device (no twoway connectivity)");
            reportEvent("stereoPairFollowerStopConversationRequest", {{"requestId", requestId}, {"success", "false"}});
        }
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_stereo_pair_message()->mutable_stop_conversation_request()->set_request_id(std::move(requestId));
        });
        glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);
    }
}

void StereoPairEndpoint::toggleConversationOnLeader()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::FOLLOWER) {
        auto requestId = makeUUID();
        if (twowayConnectivity_) {
            YIO_LOG_INFO("Send directive to toggle conversation on leader device");
            reportEvent("stereoPairFollowerToggleConversationRequest", {{"requestId", requestId}, {"success", "true"}});
        } else {
            YIO_LOG_INFO("Try to send directive to toggle conversation on leader device (no twoway connectivity)");
            reportEvent("stereoPairFollowerToggleConversationRequest", {{"requestId", requestId}, {"success", "false"}});
            reportEvent("stereoPairNoConnectionNotification", {{"source", std::string_view("toggle_conversation")}});

            auto request = std::make_shared<VinsRequest>(createRepeatEvent(PHRASE_NO_CONNECTION_WITH_LEADER), VinsRequest::createSoftwareDirectiveEventSource());
            sdk_->getAliceCapability()->startRequest(std::move(request), nullptr);

            return;
        }
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_stereo_pair_message()->mutable_toggle_conversation_request()->set_request_id(std::move(requestId));
        });
        glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);
    }
}

void StereoPairEndpoint::finishConversationOnLeader()
{
    Y_ENSURE_THREAD(lifecycle_);

    if (role_ == Role::FOLLOWER) {
        auto requestId = makeUUID();
        if (twowayConnectivity_) {
            YIO_LOG_INFO("Send directive to finish conversation on leader device");
            reportEvent("stereoPairFollowerFinishConversationRequest", {{"requestId", requestId}, {"success", "true"}});
        } else {
            YIO_LOG_INFO("Try to send directive to finish conversation on leader device (no twoway connectivity)");
            reportEvent("stereoPairFollowerFinishConversationRequest", {{"requestId", requestId}, {"success", "false"}});
        }
        auto message = ipc::buildMessage([&](auto& msg) {
            msg.mutable_stereo_pair_message()->mutable_finish_conversation_request()->set_request_id(std::move(requestId));
        });
        glagolCluster_->send(partnerDeviceId_, "stereo_pair", message);
    }
}
