#pragma once

#include <yandex_io/interfaces/clock_tower/i_clock_tower_provider.h>
#include <yandex_io/interfaces/glagol/i_glagol_cluster_provider.h>
#include <yandex_io/interfaces/spectrogram_animation/i_spectrogram_animation_provider.h>
#include <yandex_io/interfaces/stereo_pair/i_stereo_pair_provider.h>
#include <yandex_io/interfaces/user_config/i_user_config_provider.h>
#include <yandex_io/interfaces/volume_manager/i_volume_manager_provider.h>
#include <yandex_io/libs/device/i_device.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/libs/threading/unique_callback.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/sdk_interface.h>

#include <chrono>
#include <optional>

namespace quasar {

    class StereoPairEndpoint {
    public:
        using Role = StereoPairState::Role;
        using Channel = StereoPairState::Channel;
        using StereoPlayerStatus = StereoPairState::StereoPlayerStatus;

    public:
        StereoPairEndpoint(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<IClockTowerProvider> clockTowerProvider,
            std::shared_ptr<IGlagolClusterProvider> glagolCluster,
            std::shared_ptr<ISpectrogramAnimationProvider> spectrogramAnimationProvider,
            std::shared_ptr<IUserConfigProvider> userConfigProvider,
            std::shared_ptr<IVolumeManagerProvider> volumeManagerProvider,
            std::shared_ptr<YandexIO::SDKInterface> sdk);

        StereoPairEndpoint(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<IClockTowerProvider> clockTowerProvider,
            std::shared_ptr<IGlagolClusterProvider> glagolCluster,
            std::shared_ptr<ISpectrogramAnimationProvider> spectrogramAnimationProvider,
            std::shared_ptr<IUserConfigProvider> userConfigProvider,
            std::shared_ptr<IVolumeManagerProvider> volumeManagerProvider,
            std::shared_ptr<ipc::IServer> stereoPairServer,
            std::shared_ptr<ICallbackQueue> lifecycle,
            std::shared_ptr<YandexIO::SDKInterface> sdk); // This constructor for internal and testing purpose ONLY

        ~StereoPairEndpoint();

    private:
        void onDeviceList(const std::shared_ptr<const std::vector<std::string>>& deviceList);
        void onClientConnected(const std::shared_ptr<ipc::IServer::IClientConnection>& client);
        void onMessageReceived(const std::shared_ptr<ipc::IServer::IClientConnection>& client, const ipc::SharedMessage& message);
        void onUserConfigChanged(const std::shared_ptr<const Json::Value>& /*json*/);

        void onMessageState(const proto::StereoPair::State& state);
        bool onMessageStateForLeader(const proto::StereoPair::State& state);
        bool onMessageStateForFollower(const proto::StereoPair::State& state);
        void onMessageRequestState(const std::shared_ptr<ipc::IServer::IClientConnection>& client, const proto::StereoPair::RequestState& rs);
        void onMessageStartConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest);
        void onMessageStopConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest);
        void onMessageToggleConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest);
        void onMessageFinishConversationRequest(const proto::StereoPair::ConversationRequest& conversationRequest);
        void onMessageInitialPairingRequest(const proto::StereoPair::InitialPairingRequest& initialPairingRequest);
        void onMessageInitialPairingAnswer(const proto::StereoPair::InitialPairingAnswer& initialPairingAnswer);
        void onMessageOverrideChannelRequest(const proto::StereoPair::OverrideChannelRequest& overrideChannelRequest);
        void onMessageSpeakNotReadyNotificationRequest(const proto::StereoPair::SpeakNotReadyNotificationReqest& speakNotReadyNotificationReqest);
        void onMessageUserEventRequest(const proto::StereoPair::UserEvent& userEvent);
        void onMessageUserEventSignal(const ipc::SharedMessage& message);
        void onMessageAlice(const proto::Alice& message);
        void onTwowayConnectivityChanged();

        bool isConnected() const;
        bool isTwowayConnectivity() const;
        ipc::SharedMessage makeStateMessage() const;
        void sendStateHeartbeat(bool verbose);
        void checkTwowayConnectivity();
        void sendInitialParingRequest();
        Channel effectiveChannel() const;
        int64_t stereoPairUptimeMs() const;
        void updateStereoPlayerReady();
        void sendMetrica();
        void reportEvent(const std::string& eventName, Json::Value values);
        void reportEvent(const std::string& eventName, std::map<std::string_view, std::string_view> values);
        void reportEvent(const std::string& eventName, std::chrono::milliseconds threshold);

        void startConversationOnLeader();
        void stopConversationOnLeader();
        void toggleConversationOnLeader();
        void finishConversationOnLeader();

    private:
        Lifetime lifetime_;
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<IClockTowerProvider> clockTowerProvider_;
        const std::shared_ptr<IGlagolClusterProvider> glagolCluster_;
        const std::shared_ptr<ISpectrogramAnimationProvider> spectrogramAnimationProvider_;
        const std::shared_ptr<IVolumeManagerProvider> volumeManagerProvider_;
        const std::shared_ptr<ipc::IServer> stereoPairServer_;
        const std::shared_ptr<ICallbackQueue> lifecycle_;

        UniqueCallback hearbeatCallback_;
        UniqueCallback twowayConnectivityCheckCallback_;
        UniqueCallback initialPairingCallback_;
        UniqueCallback metricaCallback_;

        /*
         * Configuration
         */
        Role role_{Role::UNDEFINED};
        std::string partnerDeviceId_;
        Channel channel_{Channel::ALL};
        std::atomic<bool> playJingle_{true};
        std::atomic<bool> nowaitSync_{false};

        /*
         * State
         */
        std::shared_ptr<const ClockTowerState> clockTowerState_;
        bool heartbeatReady_{false};
        int64_t syncId_{0};
        Channel overridedChannel_{Channel::UNDEFINED};
        std::optional<int64_t> partnerUptime_;
        bool stereoPlayerReady_{false};
        bool partnerPlayerReady_{false};
        bool twowayConnectivity_{false};
        std::string initialPairingUuid_;
        std::string initialPairingJingleUuid_;
        std::chrono::steady_clock::time_point startUpTime_;
        bool awatingStereoPlayerReadyEvent_{false};
        std::chrono::steady_clock::time_point initialPairingStarted_;
        std::chrono::system_clock::time_point initialPairingTimepoint_;
        std::chrono::steady_clock::time_point lastPartnerStateReceived_;
        SpectrogramAnimationStateConstPtr spectrogramAnimationState_;
        VolumeManagerStateConstPtr volumeManagerState_;

        std::map<std::string, std::chrono::steady_clock::time_point> lastReportedEvents_;
        std::shared_ptr<YandexIO::SDKInterface> sdk_;
    };

} // namespace quasar
