#include "sync_endpoint.h"

#include <yandex_io/services/pushd/xiva_operations.h>

#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/glagol_sdk/account_devices.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/sdk/converters/glagol_device_list_converter.h>

#include <util/system/yassert.h>

#include <cstdlib>
#include <iostream>

YIO_DEFINE_LOG_MODULE("sync");

using namespace quasar;
using namespace quasar::proto;

namespace {
    // leave seconds here because of config compatibility
    constexpr std::chrono::seconds DEFAULT_UPDATE_PERIOD = std::chrono::minutes(10);
    constexpr std::chrono::milliseconds MAX_UPDATE_PERIOD = std::chrono::minutes(15);
    constexpr std::chrono::milliseconds MIN_UPDATE_PERIOD = std::chrono::minutes(1);

    void fillDevicesList(QuasarMessage& message, const glagol::BackendApi::DevicesMap& devices) {
        auto devicesList = YandexIO::convertDevicesListToProto(devices);
        message.mutable_account_devices_list()->Swap(&devicesList);
    }

    const Json::Value& getSyncdConfig(YandexIO::IDevice& device) {
        return device.configuration()->getServiceConfig(SyncEndpoint::SERVICE_NAME);
    }

    bool updateFieldsHasNetworkInfo(const Json::Value& v) {
        if (v.isMember("update_fields")) {
            const auto& arr = v["update_fields"];
            if (arr.isArray()) {
                for (const auto& item : arr) {
                    if (item.isString() && item.asString() == "networkInfo") {
                        return true;
                    }
                }
            }
        }
        return false;
    }
} // namespace

SyncEndpoint::SyncEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
    std::shared_ptr<IBackoffRetries> delayTimingsPolicy)
    : SyncEndpoint(std::move(device), std::move(ipcFactory), std::move(authProvider), std::move(deviceStateProvider), std::move(delayTimingsPolicy),
                   [](std::shared_ptr<glagol::BackendApi> backendApi, std::string cacheFile) {
                       return std::make_unique<glagol::AccountDevices>(std::move(backendApi), std::move(cacheFile), true);
                   }, MAX_UPDATE_PERIOD, MIN_UPDATE_PERIOD)
{
}

SyncEndpoint::SyncEndpoint(
    std::shared_ptr<YandexIO::IDevice> device,
    std::shared_ptr<ipc::IIpcFactory> ipcFactory,
    std::shared_ptr<IAuthProvider> authProvider,
    std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
    std::shared_ptr<IBackoffRetries> delayTimingsPolicy,
    AccountDevicesFactory accountDevicesFactory,
    std::chrono::milliseconds maxUpdatePeriod,
    std::chrono::milliseconds minUpdatePeriod)
    : device_(std::move(device))
    , authProvider_(std::move(authProvider))
    , deviceStateProvider_(std::move(deviceStateProvider))
    , pollPeriodDefault_(tryGetSeconds(getSyncdConfig(*device_), "updatePeriodSec", DEFAULT_UPDATE_PERIOD))
    , backendClient_("sync-quasar-backend", device_)
    , storageFile_(getString(getSyncdConfig(*device_), "configPath"))
    , pushdConnector_(ipcFactory->createIpcConnector("pushd"))
    , server_(ipcFactory->createIpcServer(SERVICE_NAME))
    , backendUrl_(device_->configuration()->getBackendUrl())
    , lifecycle_(std::make_shared<NamedCallbackQueue>("SyncdEndpoint_lifecycle"))
    , deviceContext_(ipcFactory, nullptr, false)
    , delayTimingsPolicy_(std::move(delayTimingsPolicy))
    , authTokenNotNeeded_(tryGetBool(getSyncdConfig(*device_), "authTokenNotNeeded", false))
{
    Y_VERIFY(authProvider_);
    storageFile_.Parent().MkDirs();
    configStorage_ = loadConfigStorage(storageFile_.GetPath());

    currentUserId_ = configStorage_.current_passport_uid();
    for (const auto& userConfig : configStorage_.user_config()) {
        if (userConfig.passport_uid() == currentUserId_) {
            currentConfig_ = userConfig;
            cachedConfig_ = currentConfig_;
            syncReceived_ = true;
            processNewConfig();
        }
    }
    bool found = false;
    for (const auto& subscriptionState : configStorage_.subscription_state()) {
        if (subscriptionState.passport_uid() == currentUserId_) {
            currentSubscriptionInfo_ = subscriptionState.subscription_info();
            lastUpdateTimeSec_ = subscriptionState.last_update_time();
            found = true;
        }
    }

    if (syncReceived_ && !found) {
        YIO_LOG_ERROR_EVENT("SyncEndpoint.MissingSubscriptionState", "Can't find subscription info for passportUid " << currentUserId_);
        syncReceived_ = false;
    }

    pushdConnector_->setMessageHandler(std::bind(&SyncEndpoint::processQuasarMessage, this, std::placeholders::_1));
    pushdConnector_->connectToService();
    backendClient_.setTimeout(std::chrono::seconds(20));
    backendClient_.setRetriesCount(0);
    backendClient_.setDnsCacheTimeout(std::chrono::seconds(0)); // disable dns cache

    delayTimingsPolicy_->initCheckPeriod(pollPeriodDefault_, pollPeriodDefault_,
                                         maxUpdatePeriod, minUpdatePeriod);

    backendClient_.setCalcRetryDelayFunction([this](int retryNum) {
        return delayTimingsPolicy_->calcHttpClientRetriesDelay(retryNum);
    });

    auto backendApi = std::make_shared<glagol::BackendApi>(
        authProvider_,
        glagol::BackendApi::Settings{backendUrl_, {}},
        device_);
    accountDevices_ = accountDevicesFactory(std::move(backendApi),
                                            tryGetString(getSyncdConfig(*device_), "deviceCacheFile"));

    server_->setClientConnectedHandler([this](auto& connection) {
        std::lock_guard lock(mutex_);
        QuasarMessage message;
        if (syncReceived_) {
            message.mutable_user_config_update()->CopyFrom(currentConfig_);

            message.mutable_subscription_state()->set_subscription_info(TString(currentSubscriptionInfo_));
            message.mutable_subscription_state()->set_last_update_time(lastUpdateTimeSec_);
            message.mutable_subscription_state()->set_passport_uid(TString(currentUserId_));
            message.mutable_subscription_state()->set_is_saved_state(!backendUpdateReceived_);
        } else {
            message.mutable_cached_user_config()->CopyFrom(cachedConfig_);
        }

        if (authInvalid_) {
            message.mutable_auth_failed();
        }

        if (hasDevicesList_.load()) {
            YIO_LOG_DEBUG("Filling devices list");
            fillDevicesList(message, accountDevices_->devices());
        }

        connection.send(std::move(message));
    });
    server_->setMessageHandler([this](const auto& message, auto& /*connection*/) {
        if (message->has_rerequest_devices_list()) {
            accountDevices_->scheduleUpdate();
        }
    });
    server_->listenService();

    accountDevices_->deviceListChangedSignal().connect(
        [this](const auto& srcDevices) {
            YIO_LOG_INFO("Devices list came. " << srcDevices.size() << " devices");
            if (srcDevices.size() != 0) {
                hasDevicesList_.store(true);
                QuasarMessage message;
                fillDevicesList(message, srcDevices);
                server_->sendToAll(std::move(message));
                YIO_LOG_INFO("broadcasted devices");
            } else {
                hasDevicesList_.store(false);
            }
        }, lifetime_, lifecycle_);

    deviceContext_.onSubscribeToDeviceConfig = [this](const std::string& configName) {
        std::lock_guard<std::mutex> guard(mutex_);
        subscribedDeviceConfigs_.insert(configName);
        if (const auto parsedConfig = tryParseJson(currentConfig_.config()); parsedConfig) {
            if (auto deviceConfig = tryGetJson(*parsedConfig, "device_config"); !deviceConfig.empty() && deviceConfig.isMember(configName)) {
                deviceContext_.fireDeviceConfig(configName, jsonToString(deviceConfig[configName], true));
            }
        }
    };

    deviceContext_.onSubscribeToSystemConfig = [this](const std::string& configName) {
        std::lock_guard<std::mutex> guard(mutex_);
        subscribedSystemConfigs_.insert(configName);
        if (const auto parsedConfig = tryParseJson(currentConfig_.config()); parsedConfig) {
            if (auto systemConfig = tryGetJson(*parsedConfig, "system_config"); !systemConfig.empty() && systemConfig.isMember(configName)) {
                deviceContext_.fireSystemConfig(configName, jsonToString(systemConfig[configName], true));
            }
        }
    };

    deviceContext_.onSubscribeToAccountConfig = [this](const std::string& configName) {
        std::lock_guard<std::mutex> guard(mutex_);
        subscribedAccountConfigs_.insert(configName);
        if (const auto parsedConfig = tryParseJson(currentConfig_.config()); parsedConfig) {
            if (auto accountConfig = tryGetJson(*parsedConfig, "account_config"); !accountConfig.empty() && accountConfig.isMember(configName)) {
                deviceContext_.fireAccountConfig(configName, jsonToString(accountConfig[configName], true));
            }
        }
    };

    deviceContext_.onUnsubscribeFromDeviceConfig = [this](const std::string& configName) {
        std::lock_guard<std::mutex> guard(mutex_);
        subscribedDeviceConfigs_.erase(configName);
    };

    deviceContext_.onUnsubscribeFromSystemConfig = [this](const std::string& configName) {
        std::lock_guard<std::mutex> guard(mutex_);
        subscribedSystemConfigs_.erase(configName);
    };

    deviceContext_.onUnsubscribeFromAccountConfig = [this](const std::string& configName) {
        std::lock_guard<std::mutex> guard(mutex_);
        subscribedAccountConfigs_.erase(configName);
    };

    deviceContext_.connectToSDK();

    authProvider_->ownerAuthInfo().connect(
        [this](const auto& authInfo) {
            std::unique_lock lock(mutex_);
            YIO_LOG_INFO("OwnerAuthInfo changed:"
                         << " old uid=" << currentUserId_
                         << " new uid=" << authInfo->passportUid
                         << " old token=" << maskToken(authToken_)
                         << " new token=" << maskToken(authInfo->authToken)
                         << " old tag=" << tag_
                         << " new tag=" << authInfo->tag);
            if (currentUserId_ != authInfo->passportUid || authInfo->authToken != authToken_ || authInfo->tag != tag_) {
                updateState(State::Updating);
                wakeupVar_.notify_one();
            }
        }, lifetime_);

    deviceStateProvider_->configurationChangedSignal().connect(
        [this](std::shared_ptr<const DeviceState> newDeviceState) {
            std::unique_lock lock(mutex_);
            if (newDeviceState->configuration == DeviceState::Configuration::CONFIGURED) {
                YIO_LOG_INFO("Device state was changed to CONFIGURED");
                updateState(State::Updating);
                wakeupVar_.notify_one();
            }
        }, lifetime_);

    updateThread_ = std::thread(&SyncEndpoint::updateLoop, this);
}

SyncEndpoint::~SyncEndpoint()
{
    // Explicitly reset connector to pushd to pervent state changing from processQuasarMessage
    pushdConnector_.reset();
    lifetime_.die();

    std::unique_lock lock(mutex_);
    updateState(State::Stopped);
    wakeupVar_.notify_one();
    lock.unlock();
    backendClient_.cancelRetries();
    updateThread_.join();
}

void SyncEndpoint::updateLoop() noexcept {
    std::unique_lock lock(mutex_);

    while (state_ != State::Stopped) {
        updateState(State::Processing);

        const auto owner = authProvider_->ownerAuthInfo().value();
        const bool hasOwner = owner != nullptr;

        quasar::AuthInfo2 ownerInfo;
        if (hasOwner) {
            ownerInfo = *owner;
        }
        const auto deviceState = deviceStateProvider_->deviceState().value();
        YIO_LOG_DEBUG("Update auth info: hasOwner: " << hasOwner << ", uid=" << ownerInfo.passportUid << ", token=" << maskToken(ownerInfo.authToken) << " tag=" << ownerInfo.tag);
        // get_sync_info can be requested without authorization, but we have to distinct the state - device isn't authorized from auth hasn't been set
        // So, first of all, we check that any (empty is also valid) auth info has been set and only if this true, we check is device authorized or not
        const auto isAuthTokenOk = !ownerInfo.authToken.empty() || authTokenNotNeeded_;
        if (deviceState->configuration == DeviceState::Configuration::CONFIGURED && isAuthTokenOk) {
            lock.unlock();
            const auto syncResult = sync(ownerInfo.authToken, ownerInfo.passportUid, ownerInfo.tag);

            deviceContext_.fireGetSyncInfoStatus(syncResult.status != SyncResult::Status::NETWORK_ERROR);

            const bool configUpdated = syncResult.configUpdated;
            const bool subscriptionInfoUpdated = syncResult.subscriptionUpdated;

            lock.lock();
            QuasarMessage message;
            if (configUpdated) {
                processNewConfig();
                message.mutable_user_config_update()->CopyFrom(currentConfig_);
            }

            if (subscriptionInfoUpdated) {
                YIO_LOG_INFO("Subscription info updated. New state: " << currentSubscriptionInfo_);

                message.mutable_subscription_state()->set_subscription_info(TString(currentSubscriptionInfo_));
                message.mutable_subscription_state()->set_passport_uid(TString(currentUserId_));
                message.mutable_subscription_state()->set_last_update_time(lastUpdateTimeSec_);
                message.mutable_subscription_state()->set_is_saved_state(!backendUpdateReceived_);
            }

            if (configUpdated || subscriptionInfoUpdated) {
                server_->sendToAll(std::move(message));
                YIO_LOG_INFO("Saving new config to " << storageFile_);
                saveConfigStorage(storageFile_.GetPath());
            }
        }
        wakeupVar_.wait_for(
            lock,
            delayTimingsPolicy_->getDelayBetweenCalls(),
            [this]() { return state_ != State::Processing; });
    }
}

SyncEndpoint::SyncResult SyncEndpoint::sync(const std::string& authToken, const std::string& passportUid, long tag)
{
    try {
        YIO_LOG_DEBUG("backend request get_sync_info: passportUid=" << passportUid << " authToken=" << maskToken(authToken));

        bool updateAll = false;
        if (currentUserId_ != passportUid) {
            YIO_LOG_INFO("change user uuid from " << currentUserId_ << " to " << passportUid);
            currentUserId_ = passportUid;
            updateAll = true;
        }

        if (authToken_ != authToken) {
            YIO_LOG_INFO("change auth token from " << maskToken(authToken_) << " to " << maskToken(authToken));
            authToken_ = authToken;
            if (accountDevices_->setSettings({backendUrl_, authToken})) {
                accountDevices_->resumeUpdate();
            }
            updateAll = true;
        }

        // reset account devices on every registration attempt (different tag received)
        if (tag_ != tag) {
            YIO_LOG_INFO("change tag from " << tag_ << " to " << tag);
            tag_ = tag;

            YIO_LOG_INFO("Schedule update of device list");
            accountDevices_->scheduleUpdate();
            updateAll = true;
        }

        std::stringstream getSyncInfo;
        getSyncInfo << backendUrl_ << "/get_sync_info?device_id=" << urlEncode(device_->deviceId()) << "&platform=" << urlEncode(device_->platform());
        const auto revision = device_->platformRevision();
        if (!revision.empty()) {
            getSyncInfo << "&revision=" << urlEncode(revision);
        }

        HttpClient::Headers headers;
        if (!authToken.empty()) {
            headers.emplace("Authorization", "OAuth " + authToken);
        }

        const auto response = backendClient_.get("get-sync-info", getSyncInfo.str(), headers);

        if (response.responseCode != 200) {
            YIO_LOG_ERROR_EVENT("SyncEndpoint.FailedGetSyncInfoRequest",
                                "Cannot update config for passportUid=" << passportUid << " authToken=" << maskToken(authToken)
                                                                        << ". Backend returned code: " << response.responseCode << ". Body: " << response.body);

            if (403 == response.responseCode) {
                authInvalid_ = true;
                QuasarMessage message;
                message.mutable_auth_failed();
                server_->sendToAll(std::move(message));
                delayTimingsPolicy_->resetDelayBetweenCallsToDefault();

                if (getStringSafe(response.body, ".message") == "AUTH_TOKEN_INVALID") {
                    authProvider_->requestAuthTokenUpdate("SyncEndpoint get-sync-info");
                } else {
                    deviceContext_.fireAuthenticationIsInvalid(currentUserId_);
                }
            } else {
                delayTimingsPolicy_->increaseDelayBetweenCalls();
            }
            return {.configUpdated = false, .subscriptionUpdated = false, .status = SyncResult::Status::HTTP_ERROR_CODE};
        } else {
            delayTimingsPolicy_->resetDelayBetweenCallsToDefault();
        }

        const auto optJson = tryParseJson(response.body);
        if (!optJson.has_value()) {
            YIO_LOG_ERROR_EVENT("SyncEndpoint.InvalidResponseJson", "Invalid response json: " << response.body);
            return {.configUpdated = false, .subscriptionUpdated = false, .status = SyncResult::Status::RESPONSE_ERROR};
        }

        const auto& jsonResponse = *optJson;
        if (jsonResponse["status"].asString() != "ok") {
            YIO_LOG_ERROR_EVENT("SyncEndpoint.BadGetSyncInfoResponse", "Cannot update config. Message: " << jsonResponse["message"].asString());
            return {.configUpdated = false, .subscriptionUpdated = false, .status = SyncResult::Status::RESPONSE_ERROR};
        }

        UserConfig newConfig;
        newConfig.set_passport_uid(TString(passportUid)); // NB: current uid becomes it anyway
        newConfig.set_config(jsonToString(jsonResponse["config"]));
        newConfig.set_group_config(jsonToString(jsonResponse.get("group", Json::objectValue))); // treat absent key as empty value

        std::string newSubscriptionInfo = jsonToString(jsonResponse["subscription"]);
        std::lock_guard lock(mutex_);
        syncReceived_ = true;
        bool configUpdated = false;

        if (!currentConfig_.IsInitialized() || (newConfig.SerializeAsString() != currentConfig_.SerializeAsString())) {
            currentConfig_ = newConfig;
            configUpdated = true;
        }

        bool subscriptionUpdated = false;
        if (newSubscriptionInfo != currentSubscriptionInfo_) {
            currentSubscriptionInfo_ = std::move(newSubscriptionInfo);
            lastUpdateTimeSec_ = getNowTimestampMs() / 1000;
            subscriptionUpdated = true;
        }

        if (!backendUpdateReceived_) {
            backendUpdateReceived_ = true;
            updateAll = true;
        }

        if (authInvalid_) {
            authInvalid_ = false;
            updateAll = true;
        }

        if (updateAll) {
            configUpdated = true;
            subscriptionUpdated = true;
        }
        return {.configUpdated = configUpdated, .subscriptionUpdated = subscriptionUpdated, .status = SyncResult::Status::OK};

    } catch (const std::exception& e) {
        YIO_LOG_WARN("Cannot update config: " << e.what());
    }

    return {.configUpdated = false, .subscriptionUpdated = false, .status = SyncResult::Status::NETWORK_ERROR};
}

void SyncEndpoint::processQuasarMessage(const ipc::SharedMessage& message)
{
    if (message->has_push_notification()) {
        const auto& operation = message->push_notification().operation();
        if (XivaOperations::UPDATE_CONFIG == operation) {
            std::unique_lock lock(mutex_);
            YIO_LOG_INFO("Received update_config push. Updating config.");
            updateState(State::Updating);
            wakeupVar_.notify_one();
        } else if (XivaOperations::UPDATE_DEVICE_STATE == operation) {
            const auto& pushMsg = message->push_notification().message();
            YIO_LOG_INFO("Update device state " << pushMsg);
            auto jsonMsg = tryParseJson(pushMsg);
            if (jsonMsg && updateFieldsHasNetworkInfo(*jsonMsg)) {
                accountDevices_->scheduleUpdate();
            }
        }
    }
}

quasar::proto::ConfigStorage SyncEndpoint::loadConfigStorage(const std::string& fileName)
{
    quasar::proto::ConfigStorage result;
    if (!fileExists(fileName)) { // Storage is not created yet
        return result;
    }

    const std::string serialized = getFileContent(fileName);
    if (!result.ParseFromString(TString(serialized))) {
        YIO_LOG_ERROR_EVENT("SyncEndpoint.BadProto.ConfigStorage", "Cannot parse config storage data from " << fileName);
    }
    return result;
}

void SyncEndpoint::saveConfigStorage(const std::string& fileName)
{
    configStorage_.set_current_passport_uid(TString(currentUserId_));
    bool found = false;
    for (auto& userConfig : *configStorage_.mutable_user_config()) {
        if (userConfig.passport_uid() == currentUserId_) {
            userConfig.CopyFrom(currentConfig_);
            found = true;
        }
    }
    if (!found) {
        auto userConfig = configStorage_.add_user_config();
        userConfig->CopyFrom(currentConfig_);
    }

    found = false;
    for (auto& subscriptionState : *configStorage_.mutable_subscription_state()) {
        if (subscriptionState.passport_uid() == currentUserId_) {
            subscriptionState.set_subscription_info(TString(currentSubscriptionInfo_));
            subscriptionState.set_last_update_time(lastUpdateTimeSec_);
            found = true;
        }
    }

    if (!found) {
        auto subscriptionState = configStorage_.add_subscription_state();
        subscriptionState->set_passport_uid(TString(currentUserId_));
        subscriptionState->set_subscription_info(TString(currentSubscriptionInfo_));
        subscriptionState->set_last_update_time(lastUpdateTimeSec_);
    }

    PersistentFile storage(fileName, PersistentFile::Mode::TRUNCATE);
    if (!storage.write(configStorage_.SerializeAsString())) {
        YIO_LOG_ERROR_EVENT("SyncEndpoint.FailedDumpConfigStorage", "Can't dump Serialized ConfigStorage");
    }
}

void SyncEndpoint::processNewConfig()
{
    YIO_LOG_INFO("Config updated. New config: " << currentConfig_.config() << ", group_config:" << currentConfig_.group_config());

    if (currentConfig_.has_config()) {
        auto parsedConfig = tryParseJson(currentConfig_.config());
        if (parsedConfig) {
            const auto& envVals = (*parsedConfig)["system_config"]["appmetrikaReportEnvironment"];
            for (const auto& key : envVals.getMemberNames()) {
                // Here we look for string environment values sent from quasmodrom to be set locally
                if (envVals[key].isString()) {
                    device_->telemetry()->putAppEnvironmentValue(key, envVals[key].asString());
                }
            }

            const auto handleDeviceConfig = [this](const auto& subscribedConfig, const auto& config) {
                for (const auto& configName : subscribedConfig) {
                    // If entry disappear from device_config -> send nullValue
                    Json::Value entry = Json::nullValue;
                    if (config.isMember(configName)) {
                        entry = config[configName];
                    }
                    deviceContext_.fireDeviceConfig(configName, jsonToString(entry, true));
                }
            };
            if (auto config = tryGetJson(*parsedConfig, "device_config"); !config.empty()) {
                if (!config.isMember("tof")) {
                    // Absence of "tof" setting means tof is on
                    config["tof"] = true;
                }
                handleDeviceConfig(subscribedDeviceConfigs_, config);
            }

            const auto handleSystemConfig = [this](const auto& subscribedConfig, const auto& config) {
                for (const auto& configName : subscribedConfig) {
                    /* if entry disappear from system_config -> send nullValue */
                    Json::Value entry = Json::nullValue;
                    if (config.isMember(configName)) {
                        entry = config[configName];
                    }
                    deviceContext_.fireSystemConfig(configName, jsonToString(entry, true));
                }
            };
            if (auto config = tryGetJson(*parsedConfig, "system_config")) {
                handleSystemConfig(subscribedSystemConfigs_, config);
            }

            const auto handleAccountConfig = [this](const auto& subscribedConfig, const auto& config) {
                for (const auto& configName : subscribedConfig) {
                    // If entry disappear from account_config -> send nullValue
                    Json::Value entry = Json::nullValue;
                    if (config.isMember(configName)) {
                        entry = config[configName];
                    }
                    deviceContext_.fireAccountConfig(configName, jsonToString(entry, true));
                }
            };
            if (auto config = tryGetJson(*parsedConfig, "account_config"); !config.empty()) {
                handleAccountConfig(subscribedAccountConfigs_, config);
            }

            // update own poll period
            const auto pollPeriod = tryGetSeconds((*parsedConfig)["system_config"]["syncd"], "pollPeriodSec", pollPeriodDefault_);
            delayTimingsPolicy_->updateCheckPeriod(pollPeriod, pollPeriod);
        } else {
            YIO_LOG_ERROR_EVENT("SyncEndpoint.BadJson.Config", "Cannot parse new config, incorrect JSON: " << currentConfig_.config());
        }
    }
}

// Must be called under mutex_
void SyncEndpoint::updateState(const State& newState) {
    YIO_LOG_DEBUG("Change state from " << stateToString(state_) << " to " << stateToString(newState));
    state_ = newState;
}

bool SyncEndpoint::wasSyncReceived() {
    std::lock_guard<std::mutex> guard(mutex_);
    return syncReceived_;
}

std::string SyncEndpoint::stateToString(State state) {
    switch (state) {
        case State::Updating:
            return "Updating";
        case State::Processing:
            return "Processing";
        case State::Stopped:
            return "Stopped";
        default:
            return "Unknown";
    }
}

const std::string SyncEndpoint::SERVICE_NAME = "syncd";
