#pragma once

#include <yandex_io/interfaces/auth/i_auth_provider.h>
#include <yandex_io/interfaces/device_state/i_device_state_provider.h>
#include <yandex_io/libs/delay_timings_policy/delay_timings_policy.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/glagol_sdk/backend_api.h>
#include <yandex_io/libs/glagol_sdk/i_account_devices.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/ipc/i_connector.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/ipc/i_server.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>
#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto_forward.h>
#include <yandex_io/sdk/private/device_context.h>

#include <util/folder/path.h>

#include <thread>

namespace quasar {

    class SyncEndpoint {
    public:
        using AccountDevicesPtr = std::unique_ptr<glagol::IAccountDevices>;
        using AccountDevicesFactory = std::function<AccountDevicesPtr(std::shared_ptr<glagol::BackendApi>, std::string cacheFile)>;

        SyncEndpoint(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<IAuthProvider> authProvider,
            std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
            std::shared_ptr<IBackoffRetries> delayTimingsPolicy);

        SyncEndpoint(
            std::shared_ptr<YandexIO::IDevice> device,
            std::shared_ptr<ipc::IIpcFactory> ipcFactory,
            std::shared_ptr<IAuthProvider> authProvider,
            std::shared_ptr<IDeviceStateProvider> deviceStateProvider,
            std::shared_ptr<IBackoffRetries> delayTimingsPolicy,
            AccountDevicesFactory accountDevicesFactory,
            std::chrono::milliseconds maxUpdatePeriod,
            std::chrono::milliseconds minUpdatePeriod); // For testing purposes only!!!

        ~SyncEndpoint();

        static const std::string SERVICE_NAME;
        bool wasSyncReceived();

        // Methods
    private:
        struct SyncResult {
            bool configUpdated;
            bool subscriptionUpdated;

            enum class Status {
                OK,              // successfully got config
                HTTP_ERROR_CODE, // Got non-200 response code
                RESPONSE_ERROR,  // Broken Json/Backend Returned Response with error
                NETWORK_ERROR,   // Request Timeout/Error/Exception
            };
            Status status;
        };

        void updateLoop() noexcept;
        SyncResult sync(const std::string& authToken, const std::string& passportUid, long tag);

        void processQuasarMessage(const ipc::SharedMessage& message);

        static quasar::proto::ConfigStorage loadConfigStorage(const std::string& fileName);
        void saveConfigStorage(const std::string& fileName);

        void processNewConfig();

        // Data
    private:
        const std::shared_ptr<YandexIO::IDevice> device_;
        const std::shared_ptr<IAuthProvider> authProvider_;
        const std::shared_ptr<IDeviceStateProvider> deviceStateProvider_;
        std::set<std::string> subscribedDeviceConfigs_;
        std::set<std::string> subscribedSystemConfigs_;
        std::set<std::string> subscribedAccountConfigs_;

        std::mutex mutex_;
        quasar::SteadyConditionVariable wakeupVar_;
        bool syncReceived_ = false;
        bool backendUpdateReceived_ = false;
        long lastUpdateTimeSec_;
        quasar::proto::UserConfig currentConfig_;
        quasar::proto::UserConfig cachedConfig_;
        std::string currentSubscriptionInfo_;
        std::string currentUserId_;
        std::string authToken_;
        long tag_ = 0;
        bool authInvalid_ = false;
        std::thread updateThread_;
        const std::chrono::seconds pollPeriodDefault_;

        HttpClient backendClient_;

        const TFsPath storageFile_;

        std::shared_ptr<ipc::IConnector> pushdConnector_;
        std::shared_ptr<ipc::IServer> server_;

        const std::string backendUrl_;

        quasar::proto::ConfigStorage configStorage_;

        Lifetime lifetime_;
        const std::shared_ptr<NamedCallbackQueue> lifecycle_;
        std::atomic<bool> hasDevicesList_{false};
        AccountDevicesPtr accountDevices_;

        YandexIO::DeviceContext deviceContext_;
        const std::shared_ptr<IBackoffRetries> delayTimingsPolicy_;

        const bool authTokenNotNeeded_;

    private:
        enum State {
            Processing,
            Updating,
            Stopped
        };

        static std::string stateToString(State state);

        void updateState(const State& newState);

        State state_ = State::Processing;
    };

} // namespace quasar
