#include "telemetry_proxy_endpoint.h"

#include <yandex_io/protos/quasar_proto.pb.h>
#include <yandex_io/libs/json_utils/json_utils.h>

using namespace quasar;
using namespace YandexIO;

namespace {

    ITelemetry::IParams::ConnectionType fromProtoConnectionType(proto::ConnectionType type) {
        switch (type) {
            case proto::CONNECTION_TYPE_UNKNOWN:
                return ITelemetry::IParams::ConnectionType::UNKNOWN;
            case proto::CONNECTION_TYPE_WIFI:
                return ITelemetry::IParams::ConnectionType::WIFI;
            case proto::CONNECTION_TYPE_ETHERNET:
                return ITelemetry::IParams::ConnectionType::ETHERNET;
        }
    }

    ITelemetry::IParams::NetworkStatus fromProtoNetworkStatus(proto::NetworkStatus status) {
        auto fromProtoStatus = [](proto::NetworkStatus::Status status) {
            switch (status) {
                case proto::NetworkStatus::NOT_CONNECTED:
                    return ITelemetry::IParams::NetworkStatus::NOT_CONNECTED;
                case proto::NetworkStatus::CONNECTING:
                    return ITelemetry::IParams::NetworkStatus::CONNECTING;
                case proto::NetworkStatus::CONNECTED_NO_INTERNET:
                    return ITelemetry::IParams::NetworkStatus::CONNECTED_NO_INTERNET;
                case proto::NetworkStatus::CONNECTED:
                    return ITelemetry::IParams::NetworkStatus::CONNECTED;
                case proto::NetworkStatus::NOT_CHOSEN:
                    return ITelemetry::IParams::NetworkStatus::NOT_CHOSEN;
            }
        };

        return {
            fromProtoStatus(status.status()),
            fromProtoConnectionType(status.type()),
        };
    }

    std::vector<ITelemetry::IParams::WifiNetwork> fromProtoWifiList(proto::WifiList wifiList) {
        std::vector<ITelemetry::IParams::WifiNetwork> networks;
        networks.reserve(wifiList.hotspots_size());

        for (const auto& hotspot : wifiList.hotspots()) {
            networks.push_back(ITelemetry::IParams::WifiNetwork{
                hotspot.mac(),
                hotspot.rssi(),
                hotspot.ssid(),
                hotspot.is_connected(),
            });
        }

        return networks;
    }

} // namespace

const std::string TelemetryProxyEndpoint::SERVICE_NAME = "telemetry_proxy";

TelemetryProxyEndpoint::TelemetryProxyEndpoint(std::shared_ptr<IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory)
    : device_(std::move(device))
    , networkdConnector_(ipcFactory->createIpcConnector("networkd"))
    , wifidConnector_(ipcFactory->createIpcConnector("wifid"))
    , syncdConnector_(ipcFactory->createIpcConnector("syncd"))
    , deviceContext_(std::make_unique<YandexIO::DeviceContext>(ipcFactory, nullptr, false))
{
    monitordConnector_ = ipcFactory->createIpcConnector("monitord");
    monitordConnector_->setConnectHandler([this]() {
        sendConfigToMonitor();
        sendNetworkStatusToMonitor();
    });
    monitordConnector_->connectToService();

    networkdConnector_->setMessageHandler([this](const auto& message) {
        handleNetworkdMessage(message);
    });
    networkdConnector_->connectToService();

    wifidConnector_->setMessageHandler([this](const auto& message) {
        handleWifidMessage(message);
    });
    wifidConnector_->connectToService();

    syncdConnector_->setMessageHandler([this](const auto& message) {
        handleSyncdMessage(message);
    });
    syncdConnector_->connectToService();

    deviceContext_->onLocation = [this](const proto::Location& location) {
        handleLocation(location);
    };
    deviceContext_->onTimezone = [this](const proto::Timezone& timezone) {
        handleTimezone(timezone);
    };
    deviceContext_->connectToSDK();
}

TelemetryProxyEndpoint::~TelemetryProxyEndpoint()
{
    networkdConnector_->shutdown();
    wifidConnector_->shutdown();
    syncdConnector_->shutdown();
}

void TelemetryProxyEndpoint::handleNetworkdMessage(const ipc::SharedMessage& message) {
    if (!message->network_status().has_status()) {
        return;
    }
    {
        std::scoped_lock lock(networkStatusMutex_);
        networkStatus_ = message->network_status();
    }
    device_->telemetry()->params()->setNetworkStatus(fromProtoNetworkStatus(message->network_status()));
    sendNetworkStatusToMonitor();
}

void TelemetryProxyEndpoint::handleWifidMessage(const ipc::SharedMessage& message) {
    if (!message->has_wifi_list()) {
        return;
    }

    device_->telemetry()->params()->setWifiNetworks(fromProtoWifiList(message->wifi_list()));
}

void TelemetryProxyEndpoint::handleSyncdMessage(const ipc::SharedMessage& message) {
    if (!message->has_user_config_update()) {
        return;
    }
    {
        std::scoped_lock lock(configMutex_);
        configCache_ = message->user_config_update().config();
    }
    device_->telemetry()->params()->setConfig(
        message->user_config_update().config());

    sendConfigToMonitor();
}

void TelemetryProxyEndpoint::handleLocation(const proto::Location& location) {
    device_->telemetry()->params()->setLocation(
        location.latitude(),
        location.longitude());
}

void TelemetryProxyEndpoint::handleTimezone(const proto::Timezone& timezone) {
    device_->telemetry()->params()->setTimezone(
        timezone.timezone_name(),
        timezone.timezone_offset_sec());
}

void TelemetryProxyEndpoint::sendConfigToMonitor() {
    proto::QuasarMessage message;
    {
        std::scoped_lock lock(configMutex_);
        if (configCache_.empty()) {
            return;
        }
        message.mutable_metrica_message()->set_config(TString(configCache_));
    }
    monitordConnector_->sendMessage(std::move(message));
}

void TelemetryProxyEndpoint::sendNetworkStatusToMonitor() {
    proto::QuasarMessage message;
    {
        std::scoped_lock lock(networkStatusMutex_);
        message.mutable_metrica_message()->mutable_network_status()->CopyFrom(networkStatus_);
    }
    monitordConnector_->sendMessage(std::move(message));
}
