#include "crc32_check_policy.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <json/value.h>

#include <util/folder/path.h>

using namespace quasar;

namespace {
    std::string boolToString(bool a) {
        return a ? "true" : "false";
    }
} // namespace

Crc32CheckPolicy::Crc32CheckPolicy(const Json::Value& deviceUpdatesdConfig)
    : Crc32CheckPolicy(tryGetBool(deviceUpdatesdConfig, "checkCrc32AfterWrite", false),
                       tryGetString(deviceUpdatesdConfig, "attemptsCounterPath", ""),
                       tryGetInt(deviceUpdatesdConfig, "maxAttempts", 5))
{
}

Crc32CheckPolicy::Crc32CheckPolicy(bool enabled, const std::string& path, uint32_t maxAttempts)
    : enabledByDeviceConfig_(enabled)
    , attemptsCounterPath_(path)
    , maxAttempts_(maxAttempts)
{
    if (enabledByDeviceConfig_) {
        if (attemptsCounterPath_.IsDefined() && attemptsCounterPath_.Exists()) {
            Json::Value contents = readJsonFromFile(attemptsCounterPath_.GetPath());
            toVersionPersisted_ = tryGetString(contents, "toVersion", "");
            attemptsDone_ = tryGetInt(contents, "attemptsDone", 0);
        } else {
            toVersionPersisted_ = "";
            attemptsDone_ = 0;
            YIO_LOG_WARN("attemptsCounterPath is not specified in updatesd config: update attempts counter will be LOST on reboot");
        }
    }
}

void Crc32CheckPolicy::applySystemConfig(const Json::Value& systemUpdaterConfig) {
    std::scoped_lock lock(mutex_);
    enabledBySystemConfig_ = tryGetBool(systemUpdaterConfig, "checkCrc32AfterWrite", true);
}

bool Crc32CheckPolicy::checkAfterWriteEnabled() {
    std::scoped_lock lock(mutex_);
    return checkAfterWriteEnabledUnlocked();
}

bool Crc32CheckPolicy::checkAfterWriteEnabledUnlocked() const {
    bool result = enabledByDeviceConfig_ && enabledBySystemConfig_;
    YIO_LOG_INFO("Crc32 check enabled: by device config - " << ::boolToString(enabledByDeviceConfig_)
                                                            << ", by system config - " << ::boolToString(enabledBySystemConfig_)
                                                            << ", result - " << ::boolToString(result));
    return result;
}

bool Crc32CheckPolicy::updateIsAllowed(const std::string& toVersion) {
    std::scoped_lock lock(mutex_);
    if (!checkAfterWriteEnabledUnlocked()) {
        YIO_LOG_INFO("Update is allowed because crc32CheckAfterWrite is disabled");
        return true;
    }

    if (toVersionPersisted_ != toVersion) {
        YIO_LOG_INFO("Update is allowed because new version is available. Persisted version: "
                     << toVersionPersisted_ << ", new version:" << toVersion);
        return true;
    }

    if (attemptsDone_ < maxAttempts_) {
        YIO_LOG_INFO("Update is allowed because attempts left. Attemps done: "
                     << attemptsDone_ << ", max attempts: " << maxAttempts_);
        return true;
    }

    YIO_LOG_INFO("Update is disallowed because no attempts left. Attemps done: "
                 << attemptsDone_ << ", max attempts: " << maxAttempts_);
    return false;
}

void Crc32CheckPolicy::registerFail(const std::string& toVersion) {
    std::scoped_lock lock(mutex_);
    if (!checkAfterWriteEnabledUnlocked()) {
        return;
    }
    if (toVersion == toVersionPersisted_) {
        attemptsDone_++;
    } else {
        attemptsDone_ = 1;
    }
    persist(toVersion);
}

void Crc32CheckPolicy::registerSuccess(const std::string& toVersion) {
    std::scoped_lock lock(mutex_);
    if (!checkAfterWriteEnabledUnlocked()) {
        return;
    }
    attemptsDone_ = 0;
    persist(toVersion);
}

bool Crc32CheckPolicy::persist(const std::string& toVersion) {
    if (!attemptsCounterPath_.IsDefined()) {
        YIO_LOG_WARN("Can't persist update attempts counter because attemptsCounterPath is not defined")
        return false;
    }
    Json::Value result;
    toVersionPersisted_ = toVersion;
    result["toVersion"] = toVersion;
    result["attemptsDone"] = attemptsDone_;

    TFsPath(attemptsCounterPath_).Parent().MkDir();
    PersistentFile configFile(attemptsCounterPath_.c_str(), PersistentFile::Mode::TRUNCATE);
    return configFile.write(jsonToString(result));
}
