#include "ota_confirm_status.h"

#include <yandex_io/libs/logging/logging.h>

#include <util/generic/scope.h>

using namespace quasar;

OtaConfirmStatus::OtaConfirmStatus(std::chrono::seconds softTimeout, std::chrono::seconds hardTimeout)
    : softOtaConfirmTimeout_(softTimeout)
    , hardOtaConfirmTimeout_(hardTimeout)
{
    Y_VERIFY(hardTimeout >= softTimeout);
}

bool OtaConfirmStatus::isWaitingOtaConfirm() const {
    std::scoped_lock<std::mutex> guard(mutex_);
    switch (confirmStatus_) {
        case ConfirmStatus::NONE:
        case ConfirmStatus::CANCELED:
        case ConfirmStatus::CONFIRMED:
            return false;
        case ConfirmStatus::WAIT_CONFIRM:
        case ConfirmStatus::POSTPONED:
            return true;
    }
    return false;
}

void OtaConfirmStatus::postpone() {
    std::scoped_lock<std::mutex> guard(mutex_);
    if (confirmStatus_ == ConfirmStatus::NONE ||
        confirmStatus_ == ConfirmStatus::CANCELED) {
        return;
    }
    confirmStatus_ = ConfirmStatus::POSTPONED;
    condVar_.notify_one();
}

void OtaConfirmStatus::confirm() {
    std::scoped_lock<std::mutex> guard(mutex_);
    if (confirmStatus_ == ConfirmStatus::NONE ||
        confirmStatus_ == ConfirmStatus::CANCELED) {
        return;
    }
    confirmStatus_ = ConfirmStatus::CONFIRMED;
    condVar_.notify_one();
}

void OtaConfirmStatus::cancel() {
    std::scoped_lock<std::mutex> guard(mutex_);
    confirmStatus_ = ConfirmStatus::CANCELED;
    condVar_.notify_one();
}

bool OtaConfirmStatus::waitOtaConfirm() {
    YIO_LOG_INFO("Wait for OTA Confirm");
    std::unique_lock<std::mutex> lock(mutex_);

    Y_DEFER {
        /* Reset confirm status when exit function */
        confirmStatus_ = ConfirmStatus::NONE;
    };

    confirmStatus_ = ConfirmStatus::WAIT_CONFIRM;
    auto now = std::chrono::steady_clock::now();
    const auto hardDeadline = now + hardOtaConfirmTimeout_; /* Deadline that can't be postponed */
    while (confirmStatus_ != ConfirmStatus::CANCELED) {
        /* reset values */
        now = std::chrono::steady_clock::now();
        confirmStatus_ = ConfirmStatus::WAIT_CONFIRM;
        const auto softDeadline = now + softOtaConfirmTimeout_;
        const auto deadline = std::min(hardDeadline, softDeadline);
        condVar_.wait_until(lock, deadline, [this]() {
            /* Wake up before timeout only if ota was Confirmed or Canceled */
            return confirmStatus_ == ConfirmStatus::CONFIRMED || confirmStatus_ == ConfirmStatus::CANCELED;
        });
        const bool isHardDeadline = deadline == hardDeadline;
        switch (confirmStatus_) {
            case ConfirmStatus::CONFIRMED:
                YIO_LOG_INFO("OTA download Confirmed. Start to download it");
                return true;
            case ConfirmStatus::WAIT_CONFIRM:
                /* Status hasn't been changed during timeout. Need to download ota */
                YIO_LOG_INFO("OTA download was not confirmed during " << (isHardDeadline ? "Hard" : "Soft") << " deadline. Download OTA");
                return true;
            case ConfirmStatus::CANCELED:
                YIO_LOG_INFO("OTA confirm wait was canceled");
                return false;
            case ConfirmStatus::POSTPONED:
                /* hard deadline can't be postponed. Apply OTA anyway*/
                if (isHardDeadline) {
                    return true;
                }
                YIO_LOG_DEBUG("Ota download was postponed, move to next deadline")
                /* Soft deadline can be postponed -> so go to next iteration */
                continue;
            default:
                break;
        }
    }
    /* Ota Confirm wait was canceled */
    return false;
}
