#pragma once

#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <chrono>
#include <mutex>

namespace quasar {

    class OtaConfirmStatus final {
    public:
        /**
         * @param softTimeout - Soft Timeout in waitOtaConfirm. Soft timeout can be postponed by postpone method
         * @param hardTimeout - Hard timeout in waitOtaConfirm. Hard timeout can't be postponed. If OTA will not be
         *                      confirmed during Hard Timeout -> waitOtaConfirm will return true anyway
         * @note SoftTimeout should be <= HardTimeout
         */
        OtaConfirmStatus(std::chrono::seconds softTimeout, std::chrono::seconds hardTimeout);

        /**
         * @brief If waitOtaConfirmed is called from some thread -> will return true
         */
        bool isWaitingOtaConfirm() const;

        /**
         * @brief Postpone Ota Confirm. If waitOtaConfirm is waiting -> it will move soft deadline
         */
        void postpone();

        /**
         * @brief Confirm Ota. If waitOtaConfirm is waiting -> it will wake up immediately and return true
         */
        void confirm();

        /**
         * @brief Cancel waitOtaConfirm. If waitOtaConfirm is waiting -> it will wake up immediately and return false
         */
        void cancel();

        /**
         * @breif Wait for OTA Confirmation during soft/hard timeouts
         * @return true - ota should be applied
         *         false - ota should not be applied
         *         TRUE can be return in case:
         *              1) Ota was confirmed by "confirm" method
         *              2) Soft timeout exceeded (timeout was not postponed)
         *              3) Hard timeout exceeded (soft timeout was postponed many times, so hard deadline exceeded)
         *         FALSE - waitOtaConfirm was Canceled by "Cancel" method
         */
        bool waitOtaConfirm();

    private:
        enum class ConfirmStatus {
            NONE,
            WAIT_CONFIRM,
            POSTPONED,
            CONFIRMED,
            CANCELED
        };

        ConfirmStatus confirmStatus_{ConfirmStatus::NONE};
        const std::chrono::seconds softOtaConfirmTimeout_;
        const std::chrono::seconds hardOtaConfirmTimeout_;

        mutable std::mutex mutex_;
        SteadyConditionVariable condVar_;
    };

} /* namespace quasar */
