#include "update_apply_status.h"

#include <yandex_io/libs/base/persistent_file.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <chrono>

#include <unistd.h>

using namespace quasar;

UpdateApplyStatus::UpdateApplyStatus(std::shared_ptr<YandexIO::IDevice> device) {
    const auto& config = device->configuration()->getServiceConfig("updatesd");
    path_ = tryGetString(config, "updateInfoPath", DEFAULT_UPDATE_INFO_PATH);
}

void UpdateApplyStatus::load() {
    if (path_.Exists()) {
        wasAttempt_ = true;
        const Json::Value config = getConfigFromFile(path_.GetPath());
        startTime_ = config["startTime"].asInt64();
        fromVersion_ = config["fromVersion"].asString();
        toVersion_ = config["toVersion"].asString();
    } else {
        wasAttempt_ = false;
    }
    shouldSave_ = false;
}

void UpdateApplyStatus::setStarted(const std::string& fromVersion, const std::string& toVersion) {
    fromVersion_ = fromVersion;
    toVersion_ = toVersion;
    startTime_ = getNowTimestampMs();
    shouldSave_ = true;
}

void UpdateApplyStatus::setFinished(const std::string& curVersion) {
    success_ = toVersion_ == curVersion;
}

bool UpdateApplyStatus::success() const {
    return success_;
}

bool UpdateApplyStatus::wasAttempt() const {
    return wasAttempt_;
}

void UpdateApplyStatus::deleteFromDisk() {
    path_.ForceDelete();
    shouldSave_ = false;
}

std::string UpdateApplyStatus::getToVersion() const {
    return toVersion_;
}

std::string UpdateApplyStatus::getFromVersion() const {
    return fromVersion_;
}

void UpdateApplyStatus::saveToDisk() {
    Json::Value config;
    config["startTime"] = startTime_;
    config["fromVersion"] = fromVersion_;
    config["toVersion"] = toVersion_;

    path_.Parent().MkDirs();
    PersistentFile configFile(path_.GetPath(), PersistentFile::Mode::TRUNCATE);
    configFile.write(jsonToString(config));
    shouldSave_ = false;
}

UpdateApplyStatus::~UpdateApplyStatus() {
    if (shouldSave_) {
        saveToDisk();
    }
}
