#include "updater.h"

#include "ota_update_script.h"
#include "update_apply_status.h"

#include <yandex_io/interfaces/user_config/connector/user_config_provider.h>
#include <yandex_io/libs/base/crc32.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/threading/i_callback_queue.h>
#include <yandex_io/protos/quasar_proto.pb.h>

#include <build/scripts/c_templates/svnversion.h>

#include <util/generic/scope.h>

#include <chrono>
#include <cstdlib>
#include <memory>
#include <random>

#include <dirent.h>
#include <sys/stat.h>

YIO_DEFINE_LOG_MODULE("updates");

using namespace quasar;
using namespace quasar::proto;

Updater::Updater(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory, std::chrono::seconds softConfirmTimeout, std::chrono::seconds hardConfirmTimeout)
    : device_(std::move(device))
    , backendClient_("update-quasar-backend", device_)
    , downloadClient_("mds", device_)
    , otaConfirmStatus_(softConfirmTimeout, hardConfirmTimeout)
    , server_(ipcFactory->createIpcServer("updatesd"))
    , userConfigProvider_(std::make_unique<UserConfigProvider>(ipcFactory))
{
    auto commonConfig = device_->configuration()->getServiceConfig("common");
    currentVersion_ = getString(commonConfig, "softwareVersion");

    auto cryptographyConfig = getJson(commonConfig, "cryptography");
    deviceCryptography_ = device_->hal()->createDeviceCryptography(cryptographyConfig);

    UpdateApplyStatus applyStatus(device_);
    applyStatus.load();
    applyStatus.setFinished(currentVersion_);

    setBuildTimestamp(GetProgramBuildTimestamp());

    backendClient_.setTimeout(std::chrono::milliseconds{10000});
    backendClient_.setRetriesCount(0);
    const auto updatesdConfig = device_->configuration()->getServiceConfig(SERVICE_NAME);
    applyUpdateScript_ = getString(updatesdConfig, "applyUpdateScript");
    crc32CheckPolicy_ = std::make_unique<Crc32CheckPolicy>(updatesdConfig);

    constexpr long CONNECT_UPDATE_SERVER_TIMEOUT_MS = 30 /*sec*/ * 1000 /*ms*/;
    constexpr long DOWNLOAD_LOW_SPEED_LIMIT_BYTE_PER_SEC = 1;
    constexpr long DOWNLOAD_LOW_SPEED_TIMEOUT_THRESHOLD_SEC = 60;

    const auto connectToUpdatesServerTimeoutMs = tryGetInt64(updatesdConfig, "connectToUpdatesServerTimeoutMs", CONNECT_UPDATE_SERVER_TIMEOUT_MS);
    downloadLowSpeedLimitByteSec_ = tryGetInt64(updatesdConfig, "downloadLowSpeedLimitByteSec", DOWNLOAD_LOW_SPEED_LIMIT_BYTE_PER_SEC);
    downloadLowSpeedLimitTimeoutSec_ = tryGetInt64(updatesdConfig, "downloadLowSpeedLimitTimeoutSec", DOWNLOAD_LOW_SPEED_TIMEOUT_THRESHOLD_SEC);
    downloadClient_.setTimeout(std::chrono::milliseconds{3600000});
    downloadClient_.setCorruptedBlockSize(100 * 1024);
    downloadClient_.setConnectionTimeout(std::chrono::milliseconds(connectToUpdatesServerTimeoutMs));

    updateAllowedForAll_ = tryGetBool(updatesdConfig, "updateAllowedAtStartForAll", updateAllowedForAll_);
    updateAllowedForCritical_ = tryGetBool(updatesdConfig, "updateAllowedAtStartForCritical", updateAllowedForCritical_);

    YIO_LOG_INFO("Update permissions at start: updateAllowed = " << updateAllowedForAll_);

    backendUrl_ = getString(commonConfig, "backendUrl");

    /* Set up update Range using system time. Reset it when got Timezone Offset from geolocation module*/
    updateRange_ = HourRange(getInt(updatesdConfig, "minUpdateHour"), getInt(updatesdConfig, "maxUpdateHour"));
    updatesDir_ = getString(updatesdConfig, "updatesDir");
    updatesExt_ = getString(updatesdConfig, "updatesExt");

    /* Default update state is None (not downloading or applying) */
    updateState_.set_state(UpdateState_State_NONE);

    server_->setClientConnectedHandler([this](auto& connection) {
        std::lock_guard<std::mutex> lock(mutex_);
        if (criticalChecked_) {
            QuasarMessage message;
            if (!hasCriticalUpdate_) {
                message.mutable_no_critical_update_found();
            } else {
                message.mutable_start_critical_update();
            }
            connection.send(std::move(message));
        }
        /* Send current UpdateState */
        QuasarMessage updateStateMsg;
        updateStateMsg.mutable_update_state()->CopyFrom(updateState_);
        connection.send(std::move(updateStateMsg));

        /* If Updater is waiting for OTA confirm -> notify new listener about it */
        if (otaConfirmStatus_.isWaitingOtaConfirm()) {
            sendReadyToApplyUpdate(&connection);
        }
    });
    server_->setMessageHandler([this](const auto& message, auto& /*connection*/) {
        /* Any service can ask Updatesd to check if there is an update */
        std::lock_guard<std::mutex> guard(mutex_);
        if (message->has_check_updates()) {
            /* ping update loop to check updates */
            criticalChecked_ = false;
            wakeupVar_.notify_one();
        } else if (message->has_confirm_update_apply()) {
            otaConfirmStatus_.confirm();
        } else if (message->has_postpone_update_apply()) {
            otaConfirmStatus_.postpone();
        }
    });

    server_->listenService();

    Json::Value metricaMsgBody;
    metricaMsgBody["toVersion"] = applyStatus.getToVersion();
    metricaMsgBody["fromVersion"] = applyStatus.getFromVersion();
    const std::string metricaMsgJson = jsonToString(metricaMsgBody);

    if (applyStatus.wasAttempt()) {
        if (applyStatus.success()) {
            device_->telemetry()->reportEvent("updateApplySuccess", metricaMsgJson);
        } else {
            device_->telemetry()->reportEvent("updateApplyFailure", metricaMsgJson);
        }
        applyStatus.deleteFromDisk();
    }

    /* Check if reboot has occurred because of Update */
    const bool updateWasAttempted = applyStatus.wasAttempt();
    /* Check that Update was successfully applied (version changed) */
    const bool updateSucceeded = applyStatus.success();

    /* Notify YandexIOSDK was update attempted or not. This can be used to make sure that reboot has occurred because of
     * Update Apply
     */
    /* fixme: send UpdateState through IO SDK */
    deviceContext_ = std::make_unique<YandexIO::DeviceContext>(
        ipcFactory,
        makeSafeCallback(
            [this, updateWasAttempted, updateSucceeded]() {
                YIO_LOG_INFO("Send Last Update state: Was Attempted:" << updateWasAttempted << " Update Succeeded: " << updateSucceeded);
                deviceContext_->fireLastUpdateState(updateWasAttempted, updateSucceeded);
            }, lifetime_.tracker()), false /* auto connect */);

    deviceContext_->onAllowUpdate = [this](bool allowUpdateAll, bool allowUpdateCritical) {
        std::lock_guard<std::mutex> guard(mutex_);
        YIO_LOG_INFO("onAllowUpdate(" << allowUpdateAll << ", " << allowUpdateCritical << ")");
        if (updateAllowedForAll_ != allowUpdateAll || updateAllowedForCritical_ != allowUpdateCritical) {
            updateAllowedForAll_ = allowUpdateAll;
            updateAllowedForCritical_ = allowUpdateCritical;
            YIO_LOG_INFO("Update is now " << (updateAllowedForAll_ ? "allowed" : "prohibited") << " for non-crits and "
                                          << ((updateAllowedForCritical_ || updateAllowedForAll_) ? "allowed" : "prohibited") << " for crits");
            if (updateAllowedForAll_ || updateAllowedForCritical_) {
                wakeupVar_.notify_all();
            }
            testStateCV_.notify_all();
        }
    };

    deviceContext_->onTimezone = [this](const proto::Timezone& timezone) {
        onTimezone(timezone);
    };

    const auto nowSinceEpoch = std::chrono::system_clock::now().time_since_epoch();
    const uint64_t sinceEpochMs = std::chrono::duration_cast<std::chrono::milliseconds>(nowSinceEpoch).count();

    const auto checkPeriodMs = std::chrono::milliseconds(tryGetInt(updatesdConfig, "checkPeriodMs", 60 * 1000));
    const auto criticalCheckPeriodMs = std::chrono::milliseconds(tryGetInt(updatesdConfig, "criticalCheckPeriodMs", 5 * 1000));

    delayTimingsPolicy_ = BackoffRetriesWithRandomPolicy(getCrc32(device_->deviceId()) + sinceEpochMs);
    delayTimingsPolicy_.initCheckPeriod(checkPeriodMs, criticalCheckPeriodMs);

    // do not call delayTimingsPolicy_.getDelayMsBetweenCalls in constructor!
    backendClient_.setCalcRetryDelayFunction([this](int retryNum) {
        return delayTimingsPolicy_.calcHttpClientRetriesDelay(retryNum);
    });

    if (updatesdConfig.isMember("randomWaitLimitSec")) {
        const int randomWaitLimitSec = updatesdConfig["randomWaitLimitSec"].asInt();
        std::uniform_int_distribution<int> distribution{0, randomWaitLimitSec};
        randomWaitSeconds_ = distribution(delayTimingsPolicy_.getRandomGenerator());
    }

    userConfigProvider_->jsonChangedSignal(IUserConfigProvider::ConfigScope::SYSTEM, "updatesd").connect([this](auto updaterSystemConfig) {
        crc32CheckPolicy_->applySystemConfig(*updaterSystemConfig);
    }, Lifetime::immortal);

    /* Connect to SDK after callbacks set up to avoid possible race condition on Boot Up */
    deviceContext_->connectToSDK();
}

void Updater::start()
{
    updateThread_ = std::thread(&Updater::updateLoop, this);
}

Updater::~Updater()
{
    lifetime_.die();
    /* Stop all requests to backends */
    downloadClient_.cancelDownload();
    backendClient_.cancelRetries();
    downloadClient_.cancelRetries();
    {
        std::lock_guard<std::mutex> guard(mutex_);
        stopped_ = true;
        otaConfirmStatus_.cancel();
        wakeupVar_.notify_one();
    }
    if (updateThread_.joinable()) {
        updateThread_.join();
    }
    deviceContext_->shutdown();
    deviceContext_.reset();
    /* Stop server and connector with handlers */
    server_->shutdown();
}

int Updater::port() const {
    return server_->port();
}

void Updater::updateLoop() noexcept {
    std::unique_lock<std::mutex> lock(mutex_);

    while (!stopped_) {
        wakeupVar_.wait(lock, [this]() { return checkAllowUpdateUnlocked(); });
        if (stopped_) {
            break;
        }

        lock.unlock();

        checkUpdate();

        lock.lock();

        // stopped_ flag could be changed during unlocked checkUpdates() call.
        if (stopped_) {
            break;
        }

        if (hasCriticalUpdate_ || !criticalChecked_) {
            delayTimingsPolicy_.spedUpDelayBetweenCalls();
        } else {
            delayTimingsPolicy_.resetDelayBetweenCallsToDefault();
        }
        const auto nextCheck = std::chrono::steady_clock::now() + delayTimingsPolicy_.getDelayBetweenCalls();

        /* Device does not download or apply OTA, so send NONE state */
        updateState_.Clear();
        updateState_.set_state(UpdateState_State::UpdateState_State_NONE);

        QuasarMessage updateStateMsg;
        updateStateMsg.mutable_update_state()->CopyFrom(updateState_);
        server_->sendToAll(std::move(updateStateMsg));

        wakeupVar_.wait_until(lock, nextCheck);
    }

    YIO_LOG_DEBUG("Stopped");
}

bool Updater::applyOtaScript(const Json::Value& scriptJson) const {
    OtaUpdateScript::Type scriptType{OtaUpdateScript::Type::T_UNKNOWN};
    std::string script;

    if (scriptJson.isMember("updateScript")) {
        scriptType = OtaUpdateScript::Type::T_STORE_EXEC;
        script = scriptJson["updateScript"].asString();
    } else if (scriptJson.isMember("updateCommand")) {
        scriptType = OtaUpdateScript::Type::T_EXEC;
        script = scriptJson["updateCommand"].asString();
    } else {
        YIO_LOG_ERROR_EVENT("Updater.UnknownOtaScriptType", "updateScript or updateCommand field missing");
    }

    if (scriptType != OtaUpdateScript::Type::T_UNKNOWN) {
        OtaUpdateScript updateScript(
            device_,
            script,
            scriptJson["updateScriptSign"].asString(),
            scriptJson["updateScriptTimestamp"].asUInt64(),
            buildTimestamp_,
            scriptType);

        if (updateScript.verify()) {
            updateScript.execute();
            return true;
        } else {
            YIO_LOG_ERROR_EVENT("Updater.FailedOtaScriptVerify", "Update script verification failed!");
        }
    }
    return false;
}

void Updater::checkUpdate()
{
    try {
        std::stringstream urlParams;
        urlParams << "device_id=" << device_->deviceId()
                  << "&version=" << urlEncode(currentVersion_)
                  << "&platform=" << urlEncode(device_->platform());
        auto revision = device_->platformRevision();
        if (!revision.empty()) {
            urlParams << "&revision=" << urlEncode(revision);
        }

        auto urlParamsString = urlParams.str();
        HttpClient::Headers headers;

        try {
            const auto signature = deviceCryptography_->sign(urlParamsString);
            HttpClient::addSignatureHeaders(headers, signature, deviceCryptography_->getType());
        } catch (const std::exception& e) {
            YIO_LOG_WARN("Cannot sign /check_updates params:" << e.what());
        }

        const auto response = backendClient_.get("check-updates", backendUrl_ + "/check_updates?" + urlParamsString, headers);
        if (response.responseCode != 200) {
            YIO_LOG_ERROR_EVENT("Updater.FailedCheckUpdatesRequest", "Backend returned code: " << response.responseCode << ". Body: " << response.body);
            return;
        }
        const auto jsonResponse = parseJson(response.body);

        if (!jsonResponse.isMember("hasUpdate")) {
            YIO_LOG_ERROR_EVENT("Updater.BadCheckUpdatesResponse", "Bad check_update response: " << jsonResponse);
            return;
        }

        if (jsonResponse.isMember("updateScriptSign") &&
            jsonResponse.isMember("updateScriptTimestamp")) {
            applyOtaScript(jsonResponse);
        }

        std::unique_lock<std::mutex> lock(mutex_);
        const bool wasCriticalChecked = criticalChecked_;
        criticalChecked_ = true;
        if (jsonResponse["hasUpdate"].asBool()) {
            YIO_LOG_INFO("Has update: version: " << jsonResponse["version"].asString() << " URL: "
                                                 << jsonResponse["downloadUrl"].asString() << " critical: " << jsonResponse["critical"].asBool());
            uint32_t crc32 = 0;
            if (jsonResponse.isMember("crc32")) {
                crc32 = jsonResponse["crc32"].asUInt();
            }

            hasCriticalUpdate_ = jsonResponse["critical"].asBool();
            if (!hasCriticalUpdate_ && !wasCriticalChecked) {
                sendNoCriticalUpdates();
            }

            // If update is not critical and it's not allowed, we don't apply anything
            if (!hasCriticalUpdate_ && !updateAllowedForAll_) {
                return;
            }

            /* Check that update is necessary */
            if (hasCriticalUpdate_ || updateRange_.containsTimestamp(getCurrentTimestamp() - std::chrono::seconds(randomWaitSeconds_))) {
                /* Notify listeners that device is ready to be updated */
                lock.unlock();

                sendReadyToApplyUpdate();
                if (!otaConfirmStatus_.waitOtaConfirm()) {
                    YIO_LOG_INFO("OTA Confirm wait was canceled");
                    return;
                }
                YIO_LOG_INFO("Updating...");
                doUpdate(jsonResponse["version"].asString(), jsonResponse["downloadUrl"].asString(), crc32);
            }

        } else {
            YIO_LOG_DEBUG("Version is up to date");
            if (!wasCriticalChecked) {
                sendNoCriticalUpdates();
            }
        }
    } catch (const std::exception& e) {
        YIO_LOG_WARN("Cannot check updates: " << e.what());
    }
}

void Updater::removeOldUpdateFiles(const std::string& updatesDirPath, const std::string& fileNotToRemove)
{
    DIR* dp = opendir(updatesDirPath.c_str());
    if (dp == nullptr) {
        throw std::runtime_error("Cannot open directory " + updatesDirPath);
    }

    Y_DEFER {
        closedir(dp);
    };

    dirent* ep;
    while ((ep = readdir(dp)) != nullptr) {
        std::string_view file = ep->d_name;
        if (file.starts_with("update") && file != fileNotToRemove) {
            const std::string oldFileName = updatesDirPath + '/' + std::string(file);
            YIO_LOG_INFO("Removing old firmware update " << oldFileName);
            ::unlink(oldFileName.c_str());
        }
    }
}

void Updater::doUpdate(const std::string& newVersion, const std::string& downloadUrl, uint32_t expectedCrc32)
{
    if (!crc32CheckPolicy_->updateIsAllowed(newVersion)) {
        YIO_LOG_INFO("Update to version " << newVersion << "is not allowed by crc32 check policy");
        return;
    }

    const std::string fileName = "update_" + escapePath(newVersion) + updatesExt_;
    const std::string filePath = updatesDir_ + '/' + fileName;
    /* Remove all old ota images to save more space for new image (note fileName file won't be remove. it's current OTA) */
    removeOldUpdateFiles(updatesDir_, fileName);

    YIO_LOG_INFO("Downloading " << downloadUrl << " to " << filePath);

    {
        std::lock_guard<std::mutex> guard(mutex_);
        if (hasCriticalUpdate_) {
            /* Old fashion message that will notify services that device have critical update */
            sendStartCriticalUpdate();
            /* Send UpdateState with Download status */
            sendDownloadProgress(0);
        }
    }

    auto onProgress = [this, lastProgressCheck = std::chrono::steady_clock::now()](int64_t downloadedBytes, int64_t totalBytes) mutable {
        const auto now = std::chrono::steady_clock::now();
        const auto diff = std::chrono::duration_cast<std::chrono::milliseconds>(now - lastProgressCheck);
        const bool tenMsPassed = (diff > std::chrono::milliseconds(10));
        if (tenMsPassed || downloadedBytes == totalBytes) {
            std::lock_guard<std::mutex> guard(mutex_);

            auto calcProgress = [](int64_t downloaded, int64_t total) -> int {
                if (total != 0) {
                    return 100 * downloaded / total;
                } else {
                    return -1;
                }
            };

            int oldProgress = calcProgress(updateDownloadedBytes_, updateTotalBytes_);

            updateTotalBytes_ = totalBytes;
            updateDownloadedBytes_ = downloadedBytes;
            int progress = calcProgress(updateDownloadedBytes_, updateTotalBytes_);

            if (progress >= 0 && progress - oldProgress >= 1) {
                sendDownloadProgress(progress);
                if (hasCriticalUpdate_) {
                    sendCriticalProgress(progress);
                }
            }
            lastProgressCheck = now;
        }
    };

    Json::Value metricaMsgBody;
    metricaMsgBody["toVersion"] = newVersion;
    metricaMsgBody["fromVersion"] = currentVersion_;
    std::string metricaMsgJson = jsonToString(metricaMsgBody);

    if (fileExists(filePath)) {
        device_->telemetry()->reportEvent("updateDownloadResume", metricaMsgJson);
    } else {
        device_->telemetry()->reportEvent("updateDownloadStart", metricaMsgJson);
    }

    uint32_t downloadedCRC32 = 0;
    const std::chrono::steady_clock::time_point start = std::chrono::steady_clock::now();
    try {
        downloadedCRC32 = downloadClient_.download("ota-update", downloadUrl, filePath,
                                                   {},
                                                   onProgress,
                                                   downloadLowSpeedLimitByteSec_,
                                                   downloadLowSpeedLimitTimeoutSec_);
        status_.addPeriod(newVersion, start, std::chrono::steady_clock::now());
    } catch (const std::exception& e) {
        status_.addPeriod(newVersion, start, std::chrono::steady_clock::now());
        device_->telemetry()->reportEvent("updateDownloadError", metricaMsgJson);
        throw;
    }

    if (expectedCrc32 != 0) {
        if (downloadedCRC32 != expectedCrc32) {
            YIO_LOG_ERROR_EVENT("Updater.BadChecksum", "CRC32 differs: " << downloadedCRC32 << " of downloaded file instead of " << expectedCrc32 << " expected.");
            ::unlink(filePath.c_str());
            crc32CheckPolicy_->registerFail(newVersion);
            return;
        }

        if (crc32CheckPolicy_->checkAfterWriteEnabled()) {
            quasar::Crc32 crc32;
            crc32.processFile(filePath);
            uint32_t persistedFileCrc32 = crc32.checksum();
            if (persistedFileCrc32 != expectedCrc32) {
                YIO_LOG_ERROR_EVENT("Updater.BadChecksumAfterWrite",
                                    "CRC32 differs: " << persistedFileCrc32 << " of persisted file instead of "
                                                      << expectedCrc32 << " expected.");
                ::unlink(filePath.c_str());
                crc32CheckPolicy_->registerFail(newVersion);
                return;
            }
        }
    }

    const uint64_t downloadDuration = status_.getDurationInMs();
    metricaMsgBody["durationMs"] = downloadDuration;
    metricaMsgJson = jsonToString(metricaMsgBody);
    device_->telemetry()->reportEvent("updateDownloadComplete", metricaMsgJson);
    crc32CheckPolicy_->registerSuccess(newVersion);

    YIO_LOG_INFO("Download of '" << filePath + "' is complete. "
                                 << "CRC32: " << downloadedCRC32);

    {
        std::lock_guard<std::mutex> guard(mutex_);
        if (!checkAllowUpdateUnlocked() || !(hasCriticalUpdate_ ||
                                             updateRange_.containsTimestamp(getCurrentTimestamp() - std::chrono::seconds(randomWaitSeconds_)))) {
            // If we received update restriction (e.g. our device discharged too much_), we return to checkUpdates() loop.
            // When update will be allowed again, we'll return here (with previously downloaded file).
            YIO_LOG_INFO("Update allowance changed or time of update passed, return to the main loop");
            return;
        }
    }

    YIO_LOG_INFO("Executing OTA script ...");

    /* Save information that device tries to update */
    UpdateApplyStatus applyStatus(device_);
    applyStatus.setStarted(currentVersion_, newVersion);
    applyStatus.saveToDisk();

    /* Notify users that starting applying OTA */
    {
        std::lock_guard<std::mutex> guard(mutex_);
        sendApplyingOta();
    }

    std::ignore = system((applyUpdateScript_ + " " + filePath).c_str());
    YIO_LOG_DEBUG("Rebooting...");
}

/* should be called under mutex */
void Updater::sendApplyingOta()
{
    QuasarMessage updateStateMsg;
    updateState_.clear_download_progress();
    updateState_.set_state(UpdateState_State::UpdateState_State_APPLYING);
    updateState_.set_is_critical(hasCriticalUpdate_);
    updateStateMsg.mutable_update_state()->CopyFrom(updateState_);
    server_->sendToAll(std::move(updateStateMsg));
}

void Updater::sendNoCriticalUpdates()
{
    QuasarMessage message;
    message.mutable_no_critical_update_found();
    server_->sendToAll(std::move(message));
}

/* should be called under mutex_ */
void Updater::sendDownloadProgress(double progress)
{
    /* Save download progress */
    updateState_.set_state(UpdateState_State::UpdateState_State_DOWNLOADING);
    updateState_.set_download_progress(progress);
    updateState_.set_is_critical(hasCriticalUpdate_);

    QuasarMessage updateStateMsg;
    updateStateMsg.mutable_update_state()->CopyFrom(updateState_);
    server_->sendToAll(std::move(updateStateMsg));
}

void Updater::sendCriticalProgress(double progress)
{
    QuasarMessage message;
    message.mutable_critical_update_progress()->set_downloaded_percent(progress);

    server_->sendToAll(std::move(message));
}

void Updater::sendStartCriticalUpdate()
{
    QuasarMessage message;
    message.mutable_start_critical_update();
    server_->sendToAll(std::move(message));
}

void Updater::sendReadyToApplyUpdate(ipc::IServer::IClientConnection* connection)
{
    QuasarMessage message;
    message.mutable_ready_to_apply_update();
    if (connection) {
        connection->send(std::move(message));
    } else {
        server_->sendToAll(std::move(message));
    }
}

std::chrono::system_clock::time_point Updater::getCurrentTimestamp() const {
    return std::chrono::system_clock::now();
}

void Updater::waitUntilTimezoneReceived(int shiftHours)
{
    std::unique_lock<std::mutex> lock(mutex_);
    testStateCV_.wait(lock, [this, shiftHours]() {
        return updateRange_.getTimezoneShiftHours() == shiftHours;
    });
}

void Updater::waitForAllowState(bool updateStateForAll, bool updateStateForCrits) {
    std::unique_lock<std::mutex> lock(mutex_);
    testStateCV_.wait(lock, [&]() {
        return (updateAllowedForAll_ == updateStateForAll && updateAllowedForCritical_ == updateStateForCrits) || stopped_;
    });
}

int Updater::getRandomWaitSeconds() const {
    return randomWaitSeconds_;
}

/**
 * Check if we can stop waiting in update cycle.
 * Must be called under mutex !!!
 * After it's true, we should check stopped_ and if it's true.
 * @return can we apply update or destructor is called.
 */
bool Updater::checkAllowUpdateUnlocked() const {
    return updateAllowedForAll_ || updateAllowedForCritical_ || stopped_;
}

void Updater::setBuildTimestamp(size_t timestamp)
{
    buildTimestamp_ = timestamp;
}

std::string Updater::escapePath(const std::string& path) {
    std::string result;
    result.reserve(path.size());
    for (auto c : path) {
        if (c == '/') {
            result += '@';
        } else {
            result += c;
        }
    }
    return result;
}

void Updater::onTimezone(const proto::Timezone& timezone) {
    if (!timezone.has_timezone_offset_sec()) {
        return;
    }

    // Timezone changed. Set up new timezone shift hours.
    // From this moment updateRange_ uses timezone shift
    // instead of system timezone (which is not working)
    const int timezoneOffsetHour = timezone.timezone_offset_sec() / (60 /*min*/ * 60 /*sec*/);
    std::lock_guard<std::mutex> guard(mutex_);
    updateRange_.setTimezoneShiftHours(timezoneOffsetHour);
    testStateCV_.notify_one();
}

const std::string Updater::SERVICE_NAME = "updatesd";
