#pragma once

#include "crc32_check_policy.h"
#include "ota_confirm_status.h"
#include "update_download_status.h"

#include <yandex_io/interfaces/user_config/i_user_config_provider.h>
#include <yandex_io/libs/base/hour_range.h>
#include <yandex_io/libs/delay_timings_policy/delay_timings_policy.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/device_cryptography/device_cryptography.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/ipc/i_ipc_factory.h>
#include <yandex_io/libs/threading/lifetime.h>
#include <yandex_io/libs/threading/steady_condition_variable.h>

#include <yandex_io/protos/model_objects.pb.h>
#include <yandex_io/protos/quasar_proto_forward.h>
#include <yandex_io/sdk/private/device_context.h>

#include <future>
#include <memory>
#include <mutex>
#include <random>
#include <thread>

namespace quasar {
    class Updater {
    public:
        Updater(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<ipc::IIpcFactory> ipcFactory, std::chrono::seconds softConfirmTimeout, std::chrono::seconds hardConfirmTimeout);
        virtual ~Updater();

        /**
         * @brief Run update loop, that will check updates
         */
        void start();

        int port() const;

        void checkUpdate();

        void setBuildTimestamp(size_t timestamp);
        virtual std::chrono::system_clock::time_point getCurrentTimestamp() const; // For testing purposes
        /**
         * @brief Wait until receive expected timezone shift in hours
         * @note for testing purposes
         */
        void waitUntilTimezoneReceived(int shiftHours);
        void waitForAllowState(bool updateStateForAll, bool updateStateForCrits); // For testing purposes. Waits until state will be equal to required one
        int getRandomWaitSeconds() const;

        static std::string escapePath(const std::string& path);

        static const std::string SERVICE_NAME;

    private:
        void updateLoop() noexcept;

        /**
         * @brief Use Ota script if it's correct (check updateScript, updateCommand, updateScriptSign and updateScriptTimestamp fields)
         * @param scriptJson Input Json that store information about update script. If Json is correct -> script from
         *        updateScript or updateCommand fields will be used
         * @return true - if used OtaScript, false - json is incorrect, skipped script
         */
        bool applyOtaScript(const Json::Value& scriptJson) const;
        void doUpdate(const std::string& newVersion, const std::string& downloadUrl, uint32_t expectedCrc32);

        void sendNoCriticalUpdates();
        /**
         * @brief Send download progress via UpdateState
         * @param progress - percent value [0..100]
         * @todo: Send all UpdateStates to IO SDK through DeviceContext
         */
        void sendDownloadProgress(double progress);
        /**
         * @brief Send Progress via old quasar message (mutable_critical_update_progress)
         * @param progress - percent value [0..100]
         * @fixme: Deprecate and remove in future. Use UpdateState only
         */
        void sendCriticalProgress(double progress);
        /**
         * @brief Send Update State with state APPLYING (so notify user that start applying OTA)
         */
        void sendApplyingOta();
        /**
         * @brief Notify listeners that started critical update
         * @fixme: Deprecate and remove in future. Use UpdateState only
         */
        void sendStartCriticalUpdate();

        void sendReadyToApplyUpdate(ipc::IServer::IClientConnection* connection = nullptr);

        /**
         * @brief Remove all files from dirPath that starts with "update" and not equal fileNotToRemove
         * @param dirPath - directory to remove files from
         * @param fileNotToRemove - files that should not be removed (It's necessary not remove file if trying to redownload ota).
         */
        static void removeOldUpdateFiles(const std::string& dirPath, const std::string& fileNotToRemove);

        void onTimezone(const quasar::proto::Timezone& timezone);

    private:
        Lifetime lifetime_;
        std::shared_ptr<YandexIO::IDevice> device_;

        std::mutex mutex_;
        quasar::SteadyConditionVariable wakeupVar_;
        quasar::SteadyConditionVariable testStateCV_;
        bool stopped_ = false;
        std::thread updateThread_;

        HttpClient backendClient_;
        HttpClient downloadClient_;

        OtaConfirmStatus otaConfirmStatus_;

        bool criticalChecked_ = false;
        bool hasCriticalUpdate_ = false;
        int64_t updateTotalBytes_ = 0;
        int64_t updateDownloadedBytes_ = 0;

        std::string backendUrl_;
        std::string currentVersion_;
        std::string applyUpdateScript_;

        HourRange updateRange_;
        int randomWaitSeconds_ = 0; // Random offset from updateRange_ in seconds to prevent all devices to download update in first minute of the range.

        std::string updatesDir_;
        std::string updatesExt_;

        size_t buildTimestamp_;
        long downloadLowSpeedLimitByteSec_ = 0;
        long downloadLowSpeedLimitTimeoutSec_ = 0;

        BackoffRetriesWithRandomPolicy delayTimingsPolicy_;

        UpdateDownloadStatus status_;

        bool updateAllowedForAll_{true};
        bool updateAllowedForCritical_{true};

        bool checkAllowUpdateUnlocked() const;

        quasar::proto::UpdateState updateState_;

        std::shared_ptr<ipc::IServer> server_;

        std::unique_ptr<YandexIO::DeviceContext> deviceContext_;
        std::unique_ptr<YandexIO::DeviceCryptography> deviceCryptography_;
        std::unique_ptr<IUserConfigProvider> userConfigProvider_;
        std::unique_ptr<Crc32CheckPolicy> crc32CheckPolicy_;
    };

} // namespace quasar
