#include <yandex_io/libs/appmetrica/app_metrica.h>
#include <yandex_io/libs/appmetrica/app_metrica_sender.h>
#include <yandex_io/libs/appmetrica/startup_client.h>
#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/hal/null/null_hal.h>
#include <yandex_io/libs/logging/logging.h>
#include <yandex_io/libs/logging/setup/setup.h>
#include <yandex_io/libs/telemetry/null/null_metrica.h>
#include <yandex_io/metrica/metricad/app_metrica_endpoint.h>

#include <boost/program_options.hpp>

#include <stdio.h>

using namespace YandexIO;
using namespace quasar;

std::shared_ptr<YandexIO::IDevice> makeDevice(const std::string& configFile) {
    auto configuration = YandexIO::makeConfiguration(configFile, std::map<std::string, std::string>());
    std::string deviceId = "appmetrica_test";
    auto telemetry = std::make_unique<NullMetrica>();
    auto hal = std::make_unique<NullHAL>();
    return std::make_shared<YandexIO::Device>(
        std::move(deviceId),
        std::move(configuration),
        std::move(telemetry),
        std::move(hal));
}

struct WifiInfo {
    std::string ssid;
    std::string mac;
    int32_t level;
    bool isConnected;
};

ReportMessage_Session_Event_NetworkInfo convertWifiList(const std::vector<WifiInfo>& hotspots)
{
    ReportMessage_Session_Event_NetworkInfo networkInfo;

    for (const auto& hotspot : hotspots) {
        auto network = networkInfo.add_wifi_networks();
        network->set_mac(TString(hotspot.mac));
        network->set_signal_strength(hotspot.level);
        network->set_ssid(TString(hotspot.ssid));
        network->set_is_connected(hotspot.isConnected);
    }
    return networkInfo;
}

WifiInfo wifiInfoFromString(std::string_view stringInfo) {
    WifiInfo info;
    size_t pos = stringInfo.find(',');
    info.ssid = stringInfo.substr(0, pos);
    stringInfo.remove_prefix(pos + 1);
    pos = stringInfo.find(',');
    info.mac = stringInfo.substr(0, pos);
    stringInfo.remove_prefix(pos + 1);
    pos = stringInfo.find(',');
    info.level = atoi(stringInfo.substr(0, pos).data());
    stringInfo.remove_prefix(pos + 1);
    info.isConnected = atoi(stringInfo.data());
    return info;
}

std::shared_ptr<ReportConfiguration> makeReportConfiguration(std::shared_ptr<IDevice> device, const std::string& UUID) {
    const auto& metricaConfig = device->configuration()->getServiceConfig("metricad");
    const std::string appMetricaStartupHost = metricaConfig["appMetricaStartupHost"].asString();
    const std::string appMetricaApiKey = metricaConfig["apiKey"].asString();

    const Json::Value commonConfig = device->configuration()->getCommonConfig();
    const std::string deviceType = commonConfig["deviceType"].asString();
    const std::string metricaAppId = commonConfig["metricaAppId"].asString();

    MetricaMetadata metricaMetadata;

    metricaMetadata.UUID = UUID;
    metricaMetadata.deviceID = device->deviceId();

    const auto appVersionName = device->softwareVersion();
    auto appVersion = appVersionName;
    appVersion.erase(std::remove_if(appVersion.begin(), appVersion.end(),
                                    [](char c) { return !std::isdigit(c); }), appVersion.end());

    auto startupConfig = StartupConfiguration{
        .startTime = std::time(nullptr),
        .startupHost = appMetricaStartupHost,
        .apiKey = appMetricaApiKey,
        .deviceID = metricaMetadata.deviceID,
        .UUID = metricaMetadata.UUID,
        .model = deviceType,
        .appVersion = appVersion,
        .appVersionName = appVersionName,
        .metricaAppID = metricaAppId,
        .revision = device->platformRevision(),
        .osVersion = StartupConfiguration::systemOsVersion(),
    };

    StartupClient startupClient(startupConfig, device);

    std::atomic<bool> stub{false};

    return startupClient.getReportConfig(stub);
}

int main(int argc, char** argv) {
    quasar::Logging::initLoggingToStdout("debug");

    boost::program_options::options_description desc("Allowed options");
    desc.add_options()("help", "help message")("config-path", boost::program_options::value<std::string>()->default_value("config.json"), "config path")("metrica-uuid", boost::program_options::value<std::string>(), "metrica-uuid")("hotspots", boost::program_options::value<std::vector<std::string>>(), "wifi info in format: ssid,mac,level,is_connected (number 1 or 0). Can be used several times")("event-name", boost::program_options::value<std::string>()->default_value("test"), "event name")("event-value", boost::program_options::value<std::string>()->default_value("test"), "event value")("session-id", boost::program_options::value<uint64_t>()->default_value(1), "session id");

    boost::program_options::variables_map opts;
    boost::program_options::store(boost::program_options::parse_command_line(argc, argv, desc), opts);

    if (opts.count("help")) {
        std::cout << desc << std::endl;
        return 0;
    }

    if (!opts.count("metrica-uuid")) {
        std::cout << "You must set metrica-uuid paran\n";
        return 1;
    }

    auto device = makeDevice(opts["config-path"].as<std::string>());
    auto report = makeReportConfiguration(device, opts["metrica-uuid"].as<std::string>());

    auto db = std::make_shared<EventsDatabase>("appmetrica.db", 2048);
    auto sendQueue = std::make_shared<blockingQueue>(1);
    AppMetricaSender appMetricaSender(report, db, sendQueue, device);

    auto event = ReportMessage_Session_Event();
    event.set_type(ReportMessage_Session_Event_EventType::ReportMessage_Session_Event_EventType_EVENT_START);
    auto startTime = std::time(nullptr);
    event.set_time(startTime);
    event.set_number_in_session(1);

    sendQueue->tryPush(ReportOneEvent(event, std::map<std::string, std::string>(), MetricaSessionProvider::Session{opts["session-id"].as<uint64_t>(), static_cast<uint64_t>(startTime), 1}));

    std::vector<WifiInfo> hotspots{WifiInfo{.ssid = "TESTSSID", .mac = "2C:54:91:88:C9:E3", .level = 1, .isConnected = true}};

    if (opts.count("hotspots")) {
        auto unparsedHotspots = opts["hotspots"].as<std::vector<std::string>>();
        hotspots.clear();
        hotspots.reserve(unparsedHotspots.size());
        for (const auto& info : unparsedHotspots) {
            hotspots.push_back(wifiInfoFromString(info));
        }
    }

    YIO_LOG_INFO("Setting wifi hotspots\n");
    for (const auto& hotspot : hotspots) {
        YIO_LOG_INFO("SSID: " << hotspot.ssid << '\n'
                              << "MAC: " << hotspot.mac << '\n'
                              << "level: " << hotspot.level << '\n'
                              << "is_connected: " << hotspot.isConnected << '\n');
    }

    appMetricaSender.setNetworkInfo(convertWifiList(hotspots));

    proto::DatabaseMetricaEvent metricaEvent;
    auto newEvent = metricaEvent.mutable_new_event();
    newEvent->set_type(proto::DatabaseMetricaEvent::NewEvent::CLIENT);
    newEvent->set_timestamp(std::time(nullptr));
    newEvent->set_name(TString(opts["event-name"].as<std::string>()));
    newEvent->set_value(TString(opts["event-value"].as<std::string>()));
    newEvent->set_serial_number(2);
    newEvent->set_session_id(opts["session-id"].as<uint64_t>());
    newEvent->set_connection_type(proto::CONNECTION_TYPE_UNKNOWN);
    newEvent->set_session_start_time(startTime);

    db->addEvent(metricaEvent);

    YIO_LOG_INFO(appMetricaSender.sendReports() << " reports was sent\n");
    return 0;
}
