#include <yandex_io/libs/audio/alsa/alsa_audio_reader.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>

#include <boost/program_options.hpp>

#include <util/generic/scope.h>

#include <atomic>
#include <exception>
#include <iostream>
#include <thread>

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

using namespace quasar;

int main(int argc, char** argv)
{
    TerminateWaiter waiter;
    boost::program_options::options_description desc("Allowed options");
    desc.add_options()("help", "produce help message")("card-number,C", boost::program_options::value<uint32_t>(), "card number")("device-number,D", boost::program_options::value<uint32_t>(), "device number")("device-name,d", boost::program_options::value<std::string>(), "device name (ignoring device & card numbers)")("channels,c", boost::program_options::value<uint32_t>()->required(), "channels")("sample-rate,r", boost::program_options::value<uint32_t>()->required(), "sample rate")("sample-size-bits,b", boost::program_options::value<uint32_t>()->default_value(16), "sample size bits (16, 32)")("period-size,p", boost::program_options::value<uint32_t>()->default_value(256), "period size (samples)")("dump-path", boost::program_options::value<std::string>()->required(), "dump path");

    boost::program_options::variables_map opts;
    boost::program_options::store(boost::program_options::parse_command_line(argc, argv, desc), opts);

    if (opts.count("help")) {
        std::cout << desc << std::endl;
        return 0;
    }

    if (opts.count("device-name") == 0 && (opts.count("card-number") == 0 || opts.count("device-number") == 0)) {
        std::cout << "specify card-number & device-number or device-name" << std::endl;
        return 0;
    }

    try {
        boost::program_options::notify(opts);
    } catch (const boost::program_options::required_option& e) {
        std::cerr << "Error: " << e.what() << '\n';
        std::cerr << desc << std::endl;
        return -EXIT_FAILURE;
    }

    std::string deviceName;
    if (opts.count("device-name")) {
        deviceName = opts["device-name"].as<std::string>();
    } else {
        const auto card = opts["card-number"].as<uint32_t>();
        const auto device = opts["device-number"].as<uint32_t>();
        deviceName = AlsaAudioReader::buildHwDeviceName(card, device);
    }

    const auto channels = opts["channels"].as<uint32_t>();
    const auto rate = opts["sample-rate"].as<uint32_t>();
    const auto dumpPath = opts["dump-path"].as<std::string>();
    const auto periodSize = opts["period-size"].as<uint32_t>();

    const auto sampleSizeBits = opts["sample-size-bits"].as<uint32_t>();
    snd_pcm_format_t pcmFormat = SND_PCM_FORMAT_S16_LE;
    switch (sampleSizeBits) {
        case 16: {
            pcmFormat = SND_PCM_FORMAT_S16_LE;
            break;
        }
        case 32: {
            pcmFormat = SND_PCM_FORMAT_S32_LE;
            break;
        }
        default: {
            std::cerr << "Unexpected sample size bits: " << sampleSizeBits << ".Should be 16 or 32\n";
            return -EXIT_FAILURE;
        }
    }

    AlsaAudioReader reader;
    reader.open(deviceName, channels, rate, pcmFormat, periodSize, 4);

    auto dumpFile = open(dumpPath.c_str(), O_TRUNC | O_SYNC | O_WRONLY | O_CREAT);
    if (dumpFile < 0) {
        throw ErrnoException(errno, "Can't open file \"" + dumpPath + "\" for dump");
    }

    Y_DEFER {
        close(dumpFile);
    };

    std::atomic_bool isRunning{true};

    auto captureThread = std::thread([&]() {
        std::vector<uint8_t> inBufMic;
        while (isRunning.load()) {
            if (!reader.read(inBufMic, 1024)) {
                try {
                    std::cerr << "Mic Audio read returned false; Error: " << reader.getError() << ". Trying to recover\n";
                    reader.tryRecover();
                    continue;
                } catch (const std::runtime_error& e) {
                    std::cerr << "Mic AudioReader error with unsuccessful recover: " << e.what() << '\n';
                    exit(-EXIT_FAILURE);
                }
            }
            const int ret = write(dumpFile, inBufMic.data(), inBufMic.size());
            if (ret < 0) {
                std::cerr << "Can't write to dump file: " << strError(errno);
                exit(-EXIT_FAILURE);
            }
        }
    });

    waiter.wait();
    isRunning = false;
    captureThread.join();
    return 0;
}
