#include <yandex_io/libs/audio_player/gstreamer/gstreamer_audio_player.h>
#include <yandex_io/libs/audio_player/base/utils.h>
#include <yandex_io/libs/json_utils/json_utils.h>

#include <boost/program_options.hpp>
#include <contrib/libs/fmt/include/fmt/format.h>
#include <util/stream/file.h>

#include <string>
#include <memory>
#include <optional>

using namespace quasar;
using namespace quasar::gstreamer;

class PlayWaiter: public AudioPlayer::SimpleListener {
public:
    void onEnd() override {
        std::scoped_lock guard(mutex_);
        end_ = true;
        cv_.notify_all();
    }

    void waitPlayEnd() {
        std::unique_lock guard(mutex_);
        cv_.wait(guard, [this]() {
            return end_;
        });
    }

private:
    bool end_ = false;
    std::mutex mutex_;
    std::condition_variable cv_;
};

int main(int argc, char** argv) {
    boost::program_options::options_description desc("Allowed options");
    desc.add_options()("help", "produce help message")("filepath,F", boost::program_options::value<std::string>()->required(), "inputfile")("output,O", boost::program_options::value<std::string>()->default_value("output.wav"), "outputFile")("norm,N", boost::program_options::bool_switch()->default_value(false), "use normalization")("true_peak", boost::program_options::value<double>(), "true peak")("integrated_loudness", boost::program_options::value<double>(), "integrated loudness")("lufs", boost::program_options::value<double>()->default_value(-14.0), "lufs")("equal_config", boost::program_options::value<std::string>(), "equalizer config");

    boost::program_options::variables_map opts;
    boost::program_options::store(boost::program_options::parse_command_line(argc, argv, desc), opts);

    std::optional<AudioPlayer::Params::Normalization> normalization;
    if (opts.count("norm") && opts["norm"].as<bool>()) {
        normalization.emplace(AudioPlayer::Params::Normalization{});
        if (!opts.count("true_peak")) {
            throw std::runtime_error("set true peak");
        }
        normalization->truePeak = opts["true_peak"].as<double>();
        if (!opts.count("integrated_loudness")) {
            throw std::runtime_error("set integrated loudness");
        }
        normalization->integratedLoudness = opts["integrated_loudness"].as<double>();
        normalization->targetLufs = opts["lufs"].as<double>();
    }

    std::string ouputPlugin = " ! filesink location={output}";
    ouputPlugin = fmt::format(ouputPlugin, fmt::arg("output", "\"" + opts["output"].as<std::string>() + "\""));
    std::string pipeline = "filesrc location={file_path} "
                           "! decodebin ! volume name=volume0 ! {opt_normalization} ! audioconvert ! "
                           "audioresample ! audio/x-raw,rate=48000,channels=2,format=S16LE ! volume name=equalizer-preamp ! equalizer-nbands name=equalizer" +
                           ouputPlugin;

    AudioPlayer::Params params;

    params.setGstreamerPipeline(pipeline);
    params.setFilePath(opts["filepath"].as<std::string>());
    std::cout << "Using pipeline: " << params.gstPipelineProcessed() << std::endl;

    if (normalization.has_value()) {
        std::cout << "Apply normalization values: truePeak: " << normalization->truePeak
                  << ", integratedLoudness: " << normalization->integratedLoudness
                  << ", targetLufs: " << normalization->targetLufs << std::endl;
        params.setNormalization(*normalization);
    }
    GstreamerAudioPlayerFactory factory(ensureGstreamerInitialized());
    const auto player = factory.createPlayer(params);

    if (opts.count("equal_config")) {
        player->setEqualizerConfig(YandexIO::EqualizerConfig::fromJson(quasar::parseJson(TFileInput(TString(opts["equal_config"].as<std::string>())).ReadAll())));
    }

    auto listener = std::make_shared<PlayWaiter>();
    player->addListener(listener);
    player->playAsync();
    listener->waitPlayEnd();
    return 0;
}
