#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/hal/null/null_hal.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/logging/setup/setup.h>
#include <yandex_io/libs/telemetry/null/null_metrica.h>

#include <boost/program_options.hpp>

#include <cstdio>
#include <iostream>
#include <memory>

std::shared_ptr<YandexIO::IDevice> makeDevice(const std::string& certificateFile) {
    Json::Value config;
    if (!certificateFile.empty()) {
        config["common"]["caCertsFile"] = certificateFile;
    }
    return std::make_shared<YandexIO::Device>(
        "stub_device_id",
        std::make_shared<YandexIO::Configuration>(config),
        std::make_unique<NullMetrica>(),
        std::make_unique<quasar::NullHAL>());
}

int main(int argc, char** argv) {
    quasar::Logging::initLoggingToStdout("debug");

    boost::program_options::options_description desc("Allowed options");
    desc.add_options()("help", "produce help message")("url,U", boost::program_options::value<std::string>()->required(), "download url")("directory,D", boost::program_options::value<std::string>()->default_value("."), "download directory")("certificate,C", boost::program_options::value<std::string>()->default_value(""), "CA certificate file");

    boost::program_options::variables_map opts;
    boost::program_options::store(boost::program_options::parse_command_line(argc, argv, desc), opts);

    if (opts.count("help")) {
        std::cout << desc << std::endl;
        return 0;
    }

    try {
        boost::program_options::notify(opts);
    } catch (const boost::program_options::required_option& e) {
        std::cerr << "Error: " << e.what() << '\n';
        std::cerr << desc << std::endl;
        return -EXIT_FAILURE;
    }

    const auto device = makeDevice(opts["certificate"].as<std::string>());

    auto filePath = opts["directory"].as<std::string>() + "/downloaded_file";
    if (quasar::fileExists(filePath)) {
        std::remove(filePath.c_str());
    }
    std::cout << "File will be downloaded to [" << filePath << "]" << std::endl;

    quasar::HttpClient httpClient("downloader-tool", device);
    httpClient.setTimeout(std::chrono::milliseconds{60 * 60 * 1000});
    httpClient.setCorruptedBlockSize(100 * 1024);
    httpClient.setConnectionTimeout(std::chrono::milliseconds(30000));

    auto onProgress = [](int64_t downloaded, int64_t total) {
        std::cout << "Downloaded: " << std::to_string(downloaded) << " Total: " << std::to_string(total) << std::endl;
    };
    httpClient.download("target", opts["url"].as<std::string>(), filePath, {}, onProgress, 0, 0);

    httpClient.cancelDownload();
    httpClient.cancelRetries();

    return 0;
}
