#include <yandex_io/libs/base/utils.h>
#include <yandex_io/libs/cryptography/cryptography.h>
#include <yandex_io/libs/device/defines.h>
#include <yandex_io/libs/device/device.h>
#include <yandex_io/libs/http_client/http_client.h>
#include <yandex_io/libs/json_utils/json_utils.h>
#include <yandex_io/services/authd/auth_endpoint.h>

#include <iostream>
#include <sstream>

using namespace quasar;
using std::chrono::milliseconds;

const int RETRY_TIMEOUT_MS = 1000;
const int RETRY_COUNT = 10;

// key from https://yav.yandex-team.ru/secret/sec-01dweya1p9n9wn4m8gnf803vvp/explore/version/ver-01dweya1q7fqwhgbtn2tnmy449
// private part is stored there
const char* PUBKEY = R"(
-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAq5q3qOZm26tYBPtYvirQ
sL/I4b2c7gRsN6ArCUhDeq2C+7e9nBrLXBspddJEyjp2VLBSgsIR+DXl5UA+fBpj
vETYjwVnxL2ntI36QrXDByXA5JzFsITnhsQzcyJoby42LhvEj3EHsAxB+kdjUsmO
xVBTldve7GfF0H1ulm6REFKTFhFbYVX/FS74r51V2jyBm9aGcnQGFfTxaUccfrLh
wgi6Q3NwKkTxbvzN96/6L03t2oImP8vHV1Eu2JzTIg9UOpYVTmbQ4nP1PkF6KGk7
vOqZgvl9gN/4dWft6spj8c2p+oglkTirDDbEzSIzhATByLP3R2WREXcKcP0LE0T+
QwIDAQAB
-----END PUBLIC KEY-----
)";

int main(int argc, char* argv[])
{
    if (argc < 2) {
        std::cerr << "Usage: " << argv[0] << " <url>";
        return 1;
    }

    auto configuration = YandexIO::makeConfiguration();
    auto device = std::make_shared<YandexIO::Device>(
        "fake_device_id",
        std::move(configuration),
        nullptr,
        nullptr);

    auto config = device->configuration()->getServiceConfig(AuthEndpoint::SERVICE_NAME);
    // FIXME: /data/quasar/ is added unconditionally in run.sh, we have no way to find that out
    auto fileName = "/data/quasar/" + getString(config, "accountStorageFile");

    if (!fileExists(fileName))
    {
        std::cerr << "File missing: " << fileName;
        return 1;
    }

    const TString serialized = getFileContent(fileName);
    quasar::proto::AccountStorage storage;
    if (!storage.ParseFromString(serialized))
    {
        std::cerr << "Cannot parse account storage data from file: " << fileName;
        return 1;
    }

    HttpClient client("lazarus", device);
    // these 10 retries are for curl'y errors -- connrefused, DNS failure, etc
    client.setRetriesCount(RETRY_COUNT);
    client.setMinRetriesTime(std::chrono::milliseconds{RETRY_TIMEOUT_MS});

    Cryptography crypto;
    crypto.setPublicKey(PUBKEY);

    for (const auto& accountInfo : storage.accounts())
    {
        std::cout << "Read accountInfo for account id { " << accountInfo.id() << " }";

        Json::Value body;

        body["x_token"] = accountInfo.xtoken();
        body["oauth_token"] = accountInfo.auth_token();

        std::string payload = crypto.encrypt(jsonToString(body));

        std::stringstream url;
        url << argv[1] << "?blob=" << urlEncode(base64Encode(payload.c_str(), payload.size()));

        auto sendRequest = [&]() {
            try {
                return client.get("refresh", url.str()).responseCode == 200;
            } catch (const std::runtime_error&) {
                return false;
            }
        };

        // ensure at least three successful requests land on server
        for (int i = 0; i < 3; i++)
        {
            // up to 10 retries with up to 16 sec sleep, exp rising
            if (!tryUntilSuccess(sendRequest, milliseconds(RETRY_TIMEOUT_MS), milliseconds(RETRY_TIMEOUT_MS * 16), RETRY_COUNT, 2)) {
                // something is way off, let's abort altogether
                std::cerr << "Failed, after all retries, giving up";
                return 1;
            }
        }

        std::cout << "Success, next.." << std::endl;
    }
    std::cout << "Done" << std::endl;

    return 0;
}
