#include "metrics_collector_tool.h"

#include <yandex_io/libs/hal/null/null_hal.h>
#include <yandex_io/libs/ipc/mixed/mixed_ipc_factory.h>
#include <yandex_io/libs/telemetry/null/null_metrica.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>
#include <yandex_io/libs/threading/periodic_executor.h>
#include <yandex_io/metrica/monitor/metrics_collector/metrics_collector.h>
#include <yandex_io/metrica/monitor/metrics_collector/ping_manager/ping_manager.h>

#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/getopt/small/last_getopt_opt.h>
#include <library/cpp/getopt/small/last_getopt_parse_result.h>

#include <ctime>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <memory>
#include <sstream>
#include <string>
#include <string_view>

namespace {

    std::string listServices(const std::set<std::string> services, const std::string_view del) {
        std::stringstream ss;
        for (auto it = services.cbegin(); it != services.cend();) {
            ss << *it;
            ++it;
            if (it != services.cend()) {
                ss << del;
            }
        }
        return ss.str();
    }

    std::shared_ptr<YandexIO::Configuration> makeConfiguration(const std::set<std::string> services) {
        Json::Value config;
        config["iohub_services"]["port"] = 11111; // some stub to make DeviceContext happy
        {
            config["monitord"]["servicesList"] = listServices(services, ",");
        }
        return std::make_unique<YandexIO::Configuration>(config);
    }

    std::shared_ptr<YandexIO::IDevice> makeDevice(const std::set<std::string>& services) {
        return std::make_shared<YandexIO::Device>(
            "stub_device_id",
            makeConfiguration(services),
            std::make_unique<NullMetrica>(),
            std::make_unique<quasar::NullHAL>());
    }

    struct Params {
        std::set<std::string> services;
        std::chrono::seconds period;
        std::string dumpFileName;
        bool logToStdout{false};
    };

    class MetricsCollectorWrapper: public quasar::MetricsCollectorBase {
    public:
        MetricsCollectorWrapper(std::shared_ptr<YandexIO::IDevice> device, std::shared_ptr<quasar::ipc::IIpcFactory> ipcFactory, const Params& params)
            : MetricsCollectorBase(std::move(device), ipcFactory, std::make_shared<quasar::PingManager>())
        {
            executor_ = std::make_unique<quasar::PeriodicExecutor>([this, params]() {
                collectSystemMetrics();
                const auto metrics = getMetrics();
                auto t = std::time(nullptr);
                struct tm buf;
                ::localtime_r(&t, &buf);
                std::stringstream ss;
                ss << "Collect time: " << std::put_time(&buf, "%d-%m-%Y %H:%M:%S") << '\n';
                ss << quasar::jsonToString(metrics) << '\n';

                if (!params.dumpFileName.empty()) {
                    std::ofstream fs(params.dumpFileName, std::ios::app);
                    fs << ss.str();
                }
                if (params.logToStdout) {
                    std::cout << ss.str();
                }
            }, params.period);
        }

    private:
        std::unique_ptr<quasar::PeriodicExecutor> executor_;
    };

    int RunMetricsCollector(const Params& params) {
        quasar::TerminateWaiter waiter;
        std::cout << "Run metrics collector.\nParams:";
        std::cout << "\n\tServices: " << (params.services.empty() ? "NONE" : listServices(params.services, ", "));
        std::cout << "\n\tPeriod: " << params.period.count() << " sec";
        std::cout << "\n\tDump FileName: " << (params.dumpFileName.empty() ? "NONE" : params.dumpFileName);
        std::cout << "\n\tLogs to stdout: " << (params.logToStdout ? "enabled" : "disabled");
        std::cout << "\n\n";
        // todo: update metrics collector, so it won't depend on YandexIO::Device
        auto device = makeDevice(params.services);
        auto ipcFactory = std::make_shared<quasar::ipc::MixedIpcFactory>(quasar::ipc::MixedIpcFactory::createDefaultContext(device->sharedConfiguration()));
        ipcFactory->setProcessName("metrics_collector");
        MetricsCollectorWrapper collector(device, ipcFactory, params);
        waiter.wait();
        return 0;
    }

} // namespace

int RunMetricsCollector(int argc, char** argv) {
    Params params;
    {
        // handle argc/argv
        NLastGetopt::TOpts opts;
        opts.AddLongOption('t', "time-period-sec")
            .Help("Time Period seconds for MetricsCollector")
            .RequiredArgument("PERIOD_SEC")
            .Required()
            .Handler1T<int>([&](const int sec) {
                params.period = std::chrono::seconds(sec);
            });
        opts.AddLongOption('p', "process")
            .Help("Process name to monitor. Option can be used multiple times")
            .RequiredArgument("PROCESS")
            .Handler1T<std::string>([&](const std::string& val) {
                params.services.insert(val);
            });
        opts.AddLongOption('f', "dump-file")
            .Help("Path to file used to dump metrics collector data to. If not used -> logs to stdout will be enabled")
            .RequiredArgument("PATH")
            .Handler1T<std::string>([&](const std::string& file) {
                params.dumpFileName = file;
            });
        opts.AddLongOption("logs-to-stdout")
            .Help("Enable logging to stdout")
            .StoreTrue(&params.logToStdout);
        NLastGetopt::TOptsParseResult r(&opts, argc, argv);

        // enable logs to stdout if dump file name is not set up
        params.logToStdout = params.logToStdout || params.dumpFileName.empty();
    }

    return RunMetricsCollector(params);
}
