import os
import argparse
import json
import logging
import shutil
from ppadb.client import Client as AdbClient


ENABLE_VALUES = ('on', 'enable', 'enabled')
DISABLE_VALUES = ('off', 'disable', 'disabled')

logging.basicConfig()
logger = logging.getLogger()


class QuasarDevice:
    def __init__(self, ppadb_device):
        self.d = ppadb_device
        try:
            self.d.root()
        except RuntimeError as e:
            logging.warn(f"Root error: {e}")
        try:
            self.d.remount()
        except RuntimeError as e:
            logging.warn(f"Remount error: {e}")

    @classmethod
    def default(cls):
        client = AdbClient()
        devices = client.devices()

        for device in devices:
            logging.debug('Got device %s' % device.serial)

        if devices:
            logging.debug('Using device %s' % devices[0].serial)
        else:
            raise ValueError('No devices!')

        return cls(devices[0])

    def shell(self, command: str):
        logging.debug('Executing command <%s>' % command)
        result = self.d.shell(command)
        logging.debug('Result is <%s>' % result)

        return result

    def pull(self, file: str, dest_file: str):
        self.d.pull(file, dest_file)
        logging.debug(f'Pulled file: {file} to: {dest_file}')

    def push(self, file: str, dest_file: str):
        logging.debug(f'Try to push file: {file} to: {dest_file}')
        self.d.push(file, dest_file)
        logging.debug(f'Pushed file: {file} to: {dest_file}')

    def reboot(self):
        logging.debug('Rebooting device')
        self.d.reboot()

    def restart_quasar(self):
        logging.debug('Restart quasar via /system/vendor/quasar/q restart')
        self.shell('/system/vendor/quasar/q restart')

    def switch_backed(self, use_dev_backend):
        logging.debug('Switching to quasar backed: ' + "dev" if use_dev_backend else "prod")
        if use_dev_backend:
            self.shell("/system/vendor/quasar/q backend-dev")
        else:
            self.shell("/system/vendor/quasar/q backend-prod")

    def forward_port(self, port: int):
        self.d.forward(f"tcp:{port}", f"tcp:{port}")


def groom_config(config_file: str):

    with open(config_file, 'r') as f:
        config = json.load(f)

    # groom config and dump it
    modified_config, already_presmoked = enable_presmoke_mode(config)

    # do not save config if it has bind field in service config already because we want to save "real" device config to guarantee proper work of --restore option
    if not already_presmoked:
        logging.debug("Saving pulled device config")
        shutil.copyfile(config_file, config_file + '.orig')

    with open(config_file, 'w') as f:
        json.dump(modified_config, f, sort_keys=True, indent=2)


def print_info(config_file: str):
    with open(config_file, 'r') as f:
        config = json.load(f)
    # print info - tcp mic port, current listening interfaces
    if config['audiod'].get('audioDevice') is None or config['audiod']['audioDevice'].get('device') != 'tcp':
        logger.info('Device is not in tcp-listening mode')
    else:
        logger.info(
            'Device is listening for audio chunks on {}:{}'.format(
                config['audiod']['audioDevice'].get('bind', 'localhost'), config['audiod']['audioDevice'].get('port')
            )
        )
    for service_name, service_config in config.items():
        if 'port' in service_config:
            logger.info(
                'Service "{}" listens on {}:{}'.format(
                    service_name, service_config.get("bind", "localhost"), service_config.get("port")
                )
            )


def ports_parser(ports=None):
    ports = ports if ports is not None else list()

    def pairs_hook(pairs):
        port_names = ['port', 'beaconPort', 'externalPort', 'httpPort']
        asd = list(x[1] for x in filter(lambda a: a[0] in port_names, pairs))
        ports.extend(asd)

    return pairs_hook


def prepare_adb_device_for_presmoke(use_dev_backend):
    d = QuasarDevice.default()
    cwd = os.getcwd()
    device_config_path = '/system/vendor/quasar/quasar.cfg'
    config_path = os.path.join(cwd, 'quasar.cfg')

    d.pull(device_config_path, config_path)

    groom_config(config_path)

    d.push(config_path, device_config_path)

    d.switch_backed(use_dev_backend)

    adb_forward_ports(config_path, d)


def restore_adb_device_config():
    logger.info("Restore device config (push quasar.cfg.orig)")
    d = QuasarDevice.default()
    device_config_path = '/system/vendor/quasar/quasar.cfg'
    orig_config = os.path.join(os.getcwd(), 'quasar.cfg.orig')
    d.push(orig_config, device_config_path)
    d.restart_quasar()


def enable_presmoke_mode(config):
    already_presmoke = False
    for service_name, service_config in config.items():
        if service_name == "presmoke" and service_config == "true":
            already_presmoke = True
        if 'port' in service_config:
            service_config['bind'] = '0.0.0.0'
        if 'Logging' in service_config and 'file' in service_config["Logging"]:
            service_config['Logging']['file']['maxFileSize'] = "10MB"
    config['audiod']['audioDevice'] = {'device': 'tcp', 'bind': '0.0.0.0', 'port': 10000}
    config['testpoint'] = {}
    config['testpoint']["use_testpoint_preprocessor"] = True
    config['testpoint']['bind'] = '0.0.0.0'
    config['testpoint']['port'] = 9999
    config['presmoke'] = "true"
    config["common"]["backendUrl"] = "https://quasar.yandex.ru"

    return config, already_presmoke


def adb_forward_ports(config_path, adb_device=None):
    if not os.path.isfile(config_path):
        raise RuntimeError(f"There is no groomed quasar.cfg in: {config_path}")

    with open(config_path, 'r') as f:
        ports = list()
        json.load(f, object_pairs_hook=ports_parser(ports))
        logger.debug(f'Ports to forward: {ports}')
        d = adb_device
        if d is None:
            d = QuasarDevice.default()
        for port in ports:
            try:
                d.forward_port(port)
            except Exception as e:
                logger.error(f"Failed to forward port: {port}, {e}")


def print_success():
    logger.info('Finished successfully')


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--config_file', type=str, help='Config file to work with')
    parser.add_argument('--print_info', action="store_true")
    parser.add_argument('--presmoke_mode', action='store_true')
    parser.add_argument("--verbose", action="store_true")
    parser.add_argument("--silent", action="store_true")
    parser.add_argument("--dev_backend", action="store_true")
    parser.add_argument(
        "--forward_ports",
        action="store_true",
        help="Use adb forward tcp:PORT tcp:PORT on each port in groomed quasar.cfg",
    )
    parser.add_argument("--restore", action="store_true", help="Push quasar.cfg.orig to device and reboot it")

    args = parser.parse_args()

    if args.verbose:
        logger.warn("--verbose option is depricated. Debug level is default now. Use --silent to set info log level")

    if args.silent:
        logger.setLevel(logging.INFO)
    else:
        logger.setLevel(logging.DEBUG)

    default_config_path = os.path.join(os.getcwd(), "quasar.cfg")
    config_file = args.config_file if args.config_file is not None else default_config_path

    if args.forward_ports:
        adb_forward_ports(config_file)
        print_success()
        return

    if args.restore:
        restore_adb_device_config()
        print_success()
        return

    if args.print_info:
        print_info(config_file)
        print_success()
        return

    if args.presmoke_mode:
        if args.config_file is None:
            # if config_file was not specified -> fetch it via adb and push it back via adb
            prepare_adb_device_for_presmoke(args.dev_backend)
            print_success()
            return
        else:
            # only update config_file
            groom_config(args.config_file)
            print_success()
            return
