#include <yandex_io/libs/audio/tinyalsa/tiny_alsa_audio_reader.h>
#include <yandex_io/libs/errno/errno_exception.h>
#include <yandex_io/libs/terminate_waiter/terminate_waiter.h>

#include <boost/program_options.hpp>

#include <util/generic/scope.h>

#include <atomic>
#include <exception>
#include <iostream>
#include <thread>

#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

using namespace quasar;

int main(int argc, char** argv)
{
    TerminateWaiter waiter;

    boost::program_options::options_description desc("Allowed options");
    desc.add_options()("help", "produce help message")("card,D", boost::program_options::value<uint32_t>()->required(), "card")("device,d", boost::program_options::value<uint32_t>()->required(), "device")("channels,c", boost::program_options::value<uint32_t>()->required(), "channels")("sample-rate,r", boost::program_options::value<uint32_t>()->required(), "sample rate")("sample-size-bits,b", boost::program_options::value<uint32_t>()->default_value(16), "sample size bits (16, 32)")("dump-path", boost::program_options::value<std::string>()->required(), "dump path");

    boost::program_options::variables_map opts;
    boost::program_options::store(boost::program_options::parse_command_line(argc, argv, desc), opts);

    if (opts.count("help")) {
        std::cout << desc << std::endl;
        return 0;
    }

    try {
        boost::program_options::notify(opts);
    } catch (const boost::program_options::required_option& e) {
        std::cerr << "Error: " << e.what() << '\n';
        std::cerr << desc << std::endl;
        return -EXIT_FAILURE;
    }

    const auto card = opts["card"].as<uint32_t>();
    const auto device = opts["device"].as<uint32_t>();
    const auto channels = opts["channels"].as<uint32_t>();
    const auto rate = opts["sample-rate"].as<uint32_t>();
    const auto dumpPath = opts["dump-path"].as<std::string>();
    const auto sampleSizeBits = opts["sample-size-bits"].as<uint32_t>();
    pcm_format pcmFormat = PCM_FORMAT_S16_LE;
    switch (sampleSizeBits) {
        case 16: {
            pcmFormat = PCM_FORMAT_S16_LE;
            break;
        }
        case 32: {
            pcmFormat = PCM_FORMAT_S32_LE;
            break;
        }
        default: {
            std::cerr << "Unexpected sample size bits: " << sampleSizeBits << ".Should be 16 or 32\n";
            return -EXIT_FAILURE;
        }
    }

    TinyAlsaAudioReader reader;
    reader.open(card, device, channels, rate, pcmFormat, 1024, 4);

    auto dumpFile = open(dumpPath.c_str(), O_TRUNC | O_SYNC | O_WRONLY | O_CREAT);
    if (dumpFile < 0) {
        throw ErrnoException(errno, "Can't open file \"" + dumpPath + "\" for dump");
    }

    Y_DEFER {
        close(dumpFile);
    };

    std::atomic_bool isRunning{true};

    auto captureThread = std::thread([&]() {
        std::vector<uint8_t> inBufMic;
        while (isRunning.load()) {
            if (!reader.read(inBufMic, 1024)) {
                try {
                    std::cerr << "Mic Audio read returned false; Error: " << reader.getError() << ". Trying to recover\n";
                    reader.tryRecover();
                    continue;
                } catch (const std::runtime_error& e) {
                    std::cerr << "Mic AudioReader error with unsuccessful recover: " << e.what() << '\n';
                    exit(-EXIT_FAILURE);
                }
            }
            const int ret = write(dumpFile, inBufMic.data(), inBufMic.size());
            if (ret < 0) {
                std::cerr << "Can't write to dump file: " << strError(errno);
                exit(-EXIT_FAILURE);
            }
        }
    });

    waiter.wait();
    isRunning = false;
    captureThread.join();
    return 0;
}
