#include <yandex_io/libs/ntp/ntp_client.h>
#include <yandex_io/libs/ntp/time_over_http.h>

#include <chrono>
#include <ctime>
#include <iomanip>
#include <iostream>
#include <sstream>
#include <thread>

#include <sys/time.h>

quasar::NtpClient::Addr parseAddress(const std::string& text)
{
    quasar::NtpClient::Addr addr;
    std::size_t found = text.rfind(':');
    if (found != std::string::npos) {
        auto it = text.begin() + found;
        addr.host = std::string(text.begin(), it);
        ++it;
        try {
            addr.port = std::stoi(std::string(it, text.end()));
        } catch (...) {
            addr.port = 123;
        }

    } else {
        addr.host = text;
    }
    return addr;
}

void printUsageAnsExit()
{
    std::cerr << "Usage: yntpclient {show|sync} [server1:port1 [server2:port2 [...]]]" << std::endl;
    std::cerr << "Usage: yntpclient http <URL>" << std::endl;
    exit(1);
}

std::string microsecondsToText(int64_t microseconds)
{
    std::stringstream ss;
    ss << microseconds / 1000000 << "." << std::setfill('0') << std::setw(6) << std::abs(microseconds % 1000000);
    return ss.str();
}

template <class T>
std::string timePointToHumanTime(T tp)
{
    std::time_t t = static_cast<std::time_t>(std::chrono::duration_cast<std::chrono::seconds>(tp.time_since_epoch()).count());
    std::stringstream ss;
    ss << std::put_time(std::gmtime(&t), "%c %Z");
    return ss.str();
}

template <class T>
std::string timePointToText(T tp)
{
    return microsecondsToText(std::chrono::duration_cast<std::chrono::microseconds>(tp.time_since_epoch()).count());
}

template <class T>
std::string durationToText(T d)
{
    return microsecondsToText(std::chrono::duration_cast<std::chrono::microseconds>(d).count());
}

template <class T1, class T2>
int64_t microsecondsDiff(T1 tp1, T2 tp2)
{
    return std::chrono::duration_cast<std::chrono::microseconds>(tp1.time_since_epoch()).count() - std::chrono::duration_cast<std::chrono::microseconds>(tp2.time_since_epoch()).count();
}

int main(int argc, char** argv)
{
    if (argc < 2)
    {
        printUsageAnsExit();
    }

    std::string cmd = argv[1];
    if (cmd == "http") {
        if (argc < 3) {
            printUsageAnsExit();
        } else {
            try {
                auto timestamp = quasar::getUtcTimeOverHttp(argv[2], std::chrono::seconds{5});
                std::time_t t = std::chrono::system_clock::to_time_t(timestamp);
                std::cout << std::ctime(&t) << std::endl;
            } catch (const std::exception& ex) {
                std::cerr << ex.what() << std::endl;
                exit(1);
            }
            return 0;
        }
    } else if (cmd != "show" && cmd != "sync" && cmd != "http")
    {
        printUsageAnsExit();
    }

    quasar::NtpClient::Params params;
    for (int i = 2; i < argc; ++i)
    {
        auto addr = parseAddress(argv[i]);
        if (!addr.host.empty())
        {
            params.ntpServers.emplace_back(std::move(addr));
        }
    }

    try {
        bool sync = cmd == "sync";
        if (cmd == "show" || sync)
        {
            params.timeout = std::chrono::milliseconds{200};
            params.sufficientMeasuringCount = (sync ? 20 : 10);
            quasar::NtpClient ntpClient(params);

            std::cout << "Request timeout : " << durationToText(params.timeout) << " seconds\n";
            auto tsync0 = std::chrono::steady_clock::now();
            auto ntpResult = ntpClient.sync(quasar::NtpClient::SyncMode::DISCOVERY);
            auto tsync1 = std::chrono::steady_clock::now();

            std::this_thread::yield();
            auto serverTime = ntpResult.syncTime();
            auto systemTime = std::chrono::system_clock::now();
            auto systemTimeSync = systemTime;

            int syncErrNo = 0;
            if (sync) {
                for (size_t i = 0; i < 5; ++i) {
                    std::this_thread::yield();
                    serverTime = ntpResult.syncTime();
                    auto timestamp = std::chrono::duration_cast<std::chrono::microseconds>(serverTime.time_since_epoch()).count();
                    struct timeval tv {
                        static_cast<time_t>(timestamp / 1000000), static_cast<suseconds_t>(timestamp % 1000000)
                    };
                    auto t0 = std::chrono::steady_clock::now();
                    auto errorCode = settimeofday(&tv, nullptr);
                    auto t1 = std::chrono::steady_clock::now();
                    if (errorCode == -1) {
                        syncErrNo = errno;
                    } else {
                        std::this_thread::yield();
                        serverTime = ntpResult.syncTime();
                        systemTimeSync = std::chrono::system_clock::now();
                        if (std::abs(microsecondsDiff(serverTime, systemTimeSync)) > 1000) {
                            std::cout << "Try sync again... (call time of \"settimeofday(...)\" " << durationToText(t1 - t0) << "\n";
                            continue;
                        }
                    }
                    break;
                }
            }

            std::cout << "Sync time       : " << durationToText(tsync1 - tsync0) << " ms\n";
            std::cout << "Resolved servers: \n";
            for (const auto& a : ntpResult.resolvedIps) {
                std::cout << "    " << a.host << "\n";
            }
            std::cout << "Server  hostname: " << ntpResult.addr.host << ":" << ntpResult.addr.port << " (" << ntpResult.ipAddress << ")\n";
            std::cout << "Ntp reference id: " << ntpResult.referenceId << "\n";
            std::cout << "Server timestamp: " << timePointToText(serverTime) << " (" << timePointToHumanTime(serverTime) << ")\n";
            if (sync && syncErrNo == 0) {
                std::cout << "System timestamp: \n";
                std::cout << "             was: " << timePointToText(systemTime) << " (" << timePointToHumanTime(systemTime) << ")\n";
                std::cout << "            sync: " << microsecondsToText(microsecondsDiff(serverTime, systemTime)) << " seconds\n";
                std::cout << "             now: " << timePointToText(systemTimeSync) << " (" << timePointToHumanTime(systemTimeSync) << ")\n";
                std::cout << "Sync delta      : " << microsecondsToText(microsecondsDiff(serverTime, systemTimeSync)) << " seconds\n";
            } else {
                std::cout << "System timestamp: " << timePointToText(systemTime) << " (" << timePointToHumanTime(systemTime) << ")\n";
                std::cout << "Sync delta      : " << microsecondsToText(microsecondsDiff(serverTime, systemTime)) << " seconds\n";
            }
            if (sync && syncErrNo) {
                std::cerr << "Fail to synchronize system clock, errno=" << syncErrNo << "\n";
                switch (syncErrNo) {
                    case EINVAL:
                        std::cerr << "Error: An attempt was made to set the time to a value less than the current value of the CLOCK_MONOTONIC clock\n";
                        break;
                    case EPERM:
                        std::cerr << "Error: Process has insufficient privilege to sync time\n";
                        break;
                    default:
                        break;
                }
                exit(1);
            }
        } else {
            exit(1);
        }
    } catch (const std::exception& ex) {
        std::cerr << ex.what() << "\n";
        exit(1);
    } catch (...) {
        std::cerr << "Unexpected exception\n";
        exit(1);
    }

    return 0;
}
