from rest_framework.throttling import ScopedRateThrottle

from yaphone.advisor.advisor.models.client import USER_AGENT_REGEXP
from yaphone.advisor.common.exceptions import BadRequestAPIError


class ScopedRateUUIDThrottle(ScopedRateThrottle):
    """
    Limits the rate of API calls by different amounts for various parts of
    the API.  Any view that has the `throttle_scope` property set will be
    throttled.  The unique cache key will be generated by concatenating the
    user UUID and the scope of the view being accessed.
    """

    @staticmethod
    def parse_user_agent(request):
        """
        Extract launcher version from User Agent
        """
        user_agent = USER_AGENT_REGEXP.match(request.META.get('HTTP_USER_AGENT', ''))
        if user_agent:
            return user_agent.groupdict()

    def allow_request(self, request, view):
        """
        Do not throttle clients with 'qa' in app version
        """
        application = self.parse_user_agent(request)
        if application and any([
            'qa' in application['app_version_string'].lower(),
            'dev' in application['app_version_string'].lower(),
            application['app_name'].lower() == 'monrun.checker'
        ]):
            return True
        return super(ScopedRateUUIDThrottle, self).allow_request(request, view)

    def get_cache_key(self, request, view):
        """
        Create cache key from uuid string and scope
        """
        uuid_string = request.META.get('HTTP_X_YAUUID')

        if not uuid_string:
            raise BadRequestAPIError('X-YAUUID header is missing')
        return self.cache_format % {
            'scope': self.scope,
            'ident': uuid_string
        }
