from datetime import datetime

from mongoengine import Document, EmbeddedDocument, UUIDField
from mongoengine.fields import (
    EmbeddedDocumentField, ReferenceField,
    IntField, FloatField,
    StringField, ListField,
    BooleanField, DateTimeField,
)

CATEGORY_ID_TEMPLATE = 'backend_wallpapers_col_{category_name}'
COVER_ID_TEMPLATE = 'backend_wallpapers_cover_{text}'
COLOR_REGEX = r'#[0-9A-Fa-f]{8}|#[0-9A-Fa-f]{6}'
COLOR_BLACK = '#000000'
LANGUAGE_CHOICES = ('en', 'es', 'pt', 'ru', 'be', 'kk', 'tr', 'uk')


class S3Image(EmbeddedDocument):
    key = StringField(required=False, default=None)
    width = IntField(required=True)
    height = IntField(required=True)
    hash = StringField(required=False, default=None)

    meta = {
        'strict': False,
    }


class WallpaperCovers(EmbeddedDocument):
    language = StringField(required=True)
    image_full = EmbeddedDocumentField(S3Image, required=True)

    meta = {
        'strict': False,
    }


class WallpaperColors(Document):
    group = StringField(required=True)
    id_ = StringField(required=True)
    value = StringField(required=True)
    order = IntField(required=True)

    meta = {
        'indexes': ['group', 'id_'],
        'strict': False,
    }

    def __unicode__(self):
        return u'{}/{}'.format(self.group, self.id_)


class WallpaperBadges(Document):
    id_ = StringField(required=True, primary_key=True)
    text_color = StringField(required=True)
    background_color = StringField(required=True)

    meta = {
        'strict': False,
    }


class WallpaperImages(EmbeddedDocument):
    image = EmbeddedDocumentField(S3Image, required=False, default=S3Image)
    url_short = StringField(required=False)
    preview_base64 = StringField(required=False)
    colors = ListField(ReferenceField(WallpaperColors), required=False)
    badges = ListField(ReferenceField(WallpaperBadges), required=False)
    collection_id = StringField(required=False)

    meta = {
        'strict': False,
    }

    @property
    def id_(self):
        return self.image.key


class ColoredText(EmbeddedDocument):
    text = StringField(default='', required=False)
    color = StringField(required=False, regex=COLOR_REGEX, default=COLOR_BLACK)

    meta = {
        'strict': False,
    }

    def __unicode__(self):
        return u'%s colored %s' % (self.text, self.color)

    @property
    def id_(self):
        key = self.text.lower().replace(' ', '_')
        return COVER_ID_TEMPLATE.format(text=key)


class CompositeCoverBackground(EmbeddedDocument):
    color = StringField(required=False, regex=COLOR_REGEX, default=COLOR_BLACK)
    image = EmbeddedDocumentField(S3Image, required=False)
    has_gradient = BooleanField(required=False)

    meta = {
        'strict': False,
    }

    def __unicode__(self):
        return u'Background: %s %s gradient' % (self.color, 'with' if self.has_gradient else 'without')


class CompositeCover(EmbeddedDocument):
    title = EmbeddedDocumentField(ColoredText, required=False)
    subtitle = EmbeddedDocumentField(ColoredText, required=False)
    background = EmbeddedDocumentField(CompositeCoverBackground, required=False)
    # full_size = BooleanField(default=False, required=False)
    generated_covers = ListField(EmbeddedDocumentField(WallpaperCovers), required=False)

    meta = {
        'strict': False,
    }

    def __unicode__(self):
        return u'Title: %s, Subtitle: %s, Background: %s' % (self.title, self.subtitle, self.background)


class WallpaperCategories(Document):
    name = StringField(required=True, unique=True)
    covers = ListField(EmbeddedDocumentField(WallpaperCovers), required=False)
    composite_cover = EmbeddedDocumentField(CompositeCover, required=False)
    images = ListField(EmbeddedDocumentField(WallpaperImages), required=False)
    updated_at = DateTimeField(required=False)
    use_in_autochange = BooleanField(required=False, default=True)
    use_in_feed = BooleanField(required=False, default=True)

    meta = {
        'strict': False,
    }

    def __unicode__(self):
        return u'Wallpapers %s, %d images, %d covers, %s composite cover' % (
            self.name, len(self.images), len(self.covers), self.composite_cover
        )

    @property
    def id_(self):
        return CATEGORY_ID_TEMPLATE.format(category_name=self.name)


class WallpaperFeedback(Document):
    uuid = UUIDField(required=True)
    wallpaper = StringField(required=True)
    action = StringField(required=True)
    created_at = DateTimeField(default=datetime.utcnow)

    meta = {
        'indexes': [
            ['uuid', '-created_at'],
            {
                'fields': ['created_at'],
                'expireAfterSeconds': 60 * 60 * 24 * 30
            }
        ],
        'strict': False
    }


class WallpaperStatus(Document):
    wallpaper = StringField(required=True)
    collection_pick_count = IntField(required=False, default=0)
    force_next_count = IntField(required=False, default=0)
    auto_next_count = IntField(required=False, default=0)
    auto_next_rate = FloatField(required=False, default=0)

    def save(self, *args, **kwargs):
        if self.auto_next_count > 0:
            self.auto_next_rate = 1. * self.auto_next_count / (self.force_next_count + self.auto_next_count)
        else:
            self.auto_next_rate = 0
        return super(WallpaperStatus, self).save(*args, **kwargs)

    meta = {
        'indexes': [
            ['wallpaper']
        ],
        'strict': False
    }


class WallpapersFeed(Document):
    uuid = UUIDField(required=True)
    feed_id = UUIDField(required=True)
    color = ReferenceField(WallpaperColors, required=False)
    wallpapers = ListField(EmbeddedDocumentField(WallpaperImages))
    created_at = DateTimeField(default=datetime.utcnow)

    meta = {
        'indexes': [
            ['feed_id', '-created_at'],
            {
                'fields': ['created_at'],
                'expireAfterSeconds': 60 * 60 * 24 * 2
            }
        ],
        'strict': False
    }
