import urllib

import logging
from django.db import models
from rest_framework import serializers

from yaphone.advisor.advisor.serializers.base import BaseSerializer
from yaphone.advisor.common.localization_helpers import translate
from yaphone.advisor.common.resizer import make_scaled_image_url
from yaphone.advisor.common.screen_properties import dp_to_px
from yaphone.advisor.common.tools import make_s3_mds_url
from yaphone.advisor.launcher.wallpapers import WALLPAPER_COVER_HEIGHT_DP, PREVIEW_IMAGE_SIZES_DP

DEFAULT_LANGUAGE = 'en'

logger = logging.getLogger(__name__)


def make_image_url(query, host, height):
    url = make_s3_mds_url(query)
    scale_url = make_scaled_image_url(url=url, height=height, host=host, path_prefix='/resizer')
    return scale_url


def make_wallpaper_url(width, height, s3_image, host):
    mds_url = make_s3_mds_url(s3_image.key)
    device_aspect_ratio = float(width) / height
    wallpaper_aspect_ratio = float(s3_image.width) / s3_image.height
    resizer_kwargs = dict(
        url=mds_url,
        host=host,
        path_prefix='/resizer',
        crop=False,
    )
    if device_aspect_ratio > wallpaper_aspect_ratio:
        # device is wider than wallpaper, resize to make equal width of device and wallpaper
        return make_scaled_image_url(width=width, **resizer_kwargs)
    else:
        return make_scaled_image_url(height=height, **resizer_kwargs)


class NoTranslationsError(KeyError):
    pass


class WallpaperImageSerializer(serializers.Serializer):
    id = serializers.CharField(required=True, source='id_')
    urls = serializers.SerializerMethodField()

    def get_urls(self, obj):
        previews = {}

        full_url = make_wallpaper_url(
            width=self.context['width'],
            height=self.context['height'],
            s3_image=obj.image,
            host=self.context['host']
        )
        mds_url = make_s3_mds_url(obj.image.key)
        preview_image_sizes = self.context['preview_image_sizes']
        for name in preview_image_sizes:
            size_px = dp_to_px(PREVIEW_IMAGE_SIZES_DP[name], self.context['dpi'])
            previews[name] = make_scaled_image_url(url=mds_url, width=size_px, height=size_px, crop='yes',
                                                   host=self.context['host'], path_prefix='/resizer')
        urls = {
            'preview': previews,
            'shareUrl': obj.url_short,
            'full': full_url,
        }
        if self.context['base64_previews']:
            urls['preview_base64'] = obj.preview_base64
        return urls


class TranslateField(serializers.CharField):
    def to_representation(self, value):
        translated = translate(self.context['client'], value)
        if not self.allow_null and not translated:
            raise NoTranslationsError('No translations for "%s"' % value)
        return translated


class S3ImageField(serializers.CharField):
    def __init__(self, **kwargs):
        self.resize = kwargs.pop('resize', False)
        self.wallpaper = kwargs.pop('wallpaper', False)
        self.crop = kwargs.pop('crop', True)

        super(S3ImageField, self).__init__(**kwargs)

    def to_representation(self, value):
        if self.wallpaper:
            return make_wallpaper_url(
                width=self.context['width'],
                height=self.context['height'],
                s3_image=value,
                host=self.context.get('host'),
            )
        if self.resize:
            return make_scaled_image_url(
                url=make_s3_mds_url(value.key),
                height=self.context.get('height', 0),
                width=self.context.get('width', 0),
                host=self.context.get('host'),
                crop=self.crop,
            )
        else:
            url = make_s3_mds_url(value.key, host=self.context.get('host'))
            return urllib.quote(url, ':/')


class WallpaperCategorySerializer(serializers.Serializer):
    id = serializers.CharField(required=True, source='id_')
    title = TranslateField(source='id_', allow_null=True)
    cover = serializers.SerializerMethodField()
    wallpapers = WallpaperImageSerializer(many=True, source='images', required=True)


class PreparedCategorySerializer(WallpaperCategorySerializer):

    def get_cover(self, obj):
        cover_collection = obj.covers if len(obj.covers) > 0 else obj.composite_cover.generated_covers
        covers = {cover.language: cover for cover in cover_collection}
        cover = covers.get(self.context['language']) or covers.get(DEFAULT_LANGUAGE)
        if cover is None:
            raise Exception('No language for cover in category %s' % obj.name)

        height = dp_to_px(WALLPAPER_COVER_HEIGHT_DP, self.context['dpi'])
        return make_image_url(cover.image_full.key, self.context['host'], height)


class ColoredTextSerializer(serializers.Serializer):
    text = TranslateField(allow_null=True)
    color = serializers.CharField(required=True)


class BackgroundSerializer(serializers.Serializer):
    color = serializers.CharField()
    image = serializers.SerializerMethodField()
    has_gradient = serializers.BooleanField(default=True)

    def get_image(self, obj):
        height = dp_to_px(WALLPAPER_COVER_HEIGHT_DP, self.context['dpi'])
        return make_image_url(obj.image.key, self.context['host'], height)


class CompositeCoverSerializer(serializers.Serializer):
    title = ColoredTextSerializer()
    subtitle = ColoredTextSerializer()
    background = BackgroundSerializer()
    # full_size = serializers.BooleanField(default=False)


class CompositeCategorySerializer(WallpaperCategorySerializer):
    cover = CompositeCoverSerializer(source='composite_cover')


class ButtonSerializer(BaseSerializer):
    OMIT_FIELDS_IF_EMPTY = ('action',)

    id = serializers.CharField(required=True)
    caption = TranslateField(required=True)
    action = serializers.CharField(required=False)
    style = serializers.CharField(required=True)


class SettingsPromoBlockSerializer(BaseSerializer):
    OMIT_FIELDS_IF_EMPTY = ('image',)

    id = serializers.CharField(required=True)
    badge = serializers.BooleanField(required=True)
    title = TranslateField(required=True)
    description = TranslateField(required=True)
    buttons = ButtonSerializer(many=True, required=True)
    image = S3ImageField(required=True, resize=True)
    show_conditions = serializers.DictField(required=True)
    priority = serializers.IntegerField(required=True)


class SettingsPromoBlockListSerializer(serializers.ListSerializer):
    child = SettingsPromoBlockSerializer()

    def to_representation(self, data):
        iterable = data.all() if isinstance(data, models.Manager) else data

        result = []
        for item in iterable:
            try:
                result.append(self.child.to_representation(item))
            except NoTranslationsError as e:
                logger.error(e)

        return result


class AutochangeImageSerializer(serializers.Serializer):
    id = serializers.SerializerMethodField()
    url = S3ImageField(required=True, source='image.image', wallpaper=True)
    category_id = serializers.CharField(required=True, source='category.id_')
    category_title = TranslateField(source='category.id_', allow_null=True)

    def get_id(self, obj):
        return obj['image'].id_.split('/')[-1]


class WallpapersAutochangeSerializer(serializers.Serializer):
    wallpapers = AutochangeImageSerializer(many=True, required=True, source='*')


class WallpapersBadgeSerializer(serializers.Serializer):
    text = TranslateField(source='id_', required=True)
    text_color = serializers.CharField(required=True)
    background_color = serializers.CharField(required=True)


class WallpapersFeedImageSerializer(WallpaperImageSerializer):
    badges = WallpapersBadgeSerializer(many=True)
    colors = serializers.SerializerMethodField()
    collection_id = serializers.CharField(required=True)

    def get_colors(self, obj):
        return ['{}/{}'.format(color.group, color.id_)
                for color in obj.colors]


class ColorFilterSerializer(serializers.Serializer):
    color_value = serializers.CharField(required=True, source='value')
    color_group_and_id = serializers.SerializerMethodField()

    def get_color_group_and_id(self, obj):
        return '{}/{}'.format(obj.group, obj.id_)


class WallpapersFeedSerializer(serializers.Serializer):
    lifetime_seconds = serializers.IntegerField(required=True)
    feed_id = serializers.UUIDField(required=True)
    wallpapers = WallpapersFeedImageSerializer(many=True, required=False)
    color_filters = ColorFilterSerializer(many=True, required=True)
    next_url = serializers.URLField(required=True)


class ThemeSerializer(BaseSerializer):
    OMIT_FIELDS_IF_EMPTY = ('preview', 'tracking_url')

    title = TranslateField(allow_null=True, required=True)
    id = serializers.CharField(source='theme_id', required=True)
    device_type = serializers.ListField(serializers.CharField(required=False))
    screens = serializers.SerializerMethodField(required=False)
    tracking_url = serializers.CharField(required=False)
    min_version = serializers.CharField(required=False)
    preview = serializers.SerializerMethodField(required=False)
    preview_text_color = serializers.CharField(required=False)
    preview_background_color = serializers.CharField(required=False)

    def get_screens(self, obj):
        urls = []
        for screen in obj.screens:
            url = make_image_url(screen.key, self.context['host'], self.context['height'])
            urls.append(url)
        return urls

    def get_preview(self, obj):
        try:
            image = obj.preview_image
            return make_image_url(image.key, self.context['host'], self.context['height'])
        except AttributeError:
            logger.debug('Theme has no preview_image field')


class IconSerializer(serializers.Serializer):
    icon_url = S3ImageField(required=True, source='*')
    hash = serializers.CharField(required=True)
