from pytz import UnknownTimeZoneError
from rest_framework.exceptions import NotFound, ValidationError
from rest_framework.fields import CharField, IntegerField, BooleanField, ChoiceField, UUIDField
from rest_framework.serializers import Serializer

from yaphone.advisor.advisor.models.categories import ya_category_cache
from yaphone.advisor.advisor.serializers.base import MultipleParametersField, CommaSeparatedListField
from yaphone.advisor.advisor.serializers.device import LocationSerializer, GsmSerializer, WifiSerializer
from yaphone.advisor.common.android_tz import AndroidTimezone
from yaphone.advisor.common.exceptions import KnownParseError
from yaphone.advisor.common.screen_properties import DPI_NAME_MAP
from yaphone.advisor.launcher.models.app_icons import Icons


class PlaceField(CharField):
    def __init__(self, **kwargs):
        super(PlaceField, self).__init__(**kwargs)
        self.validators.append(self._validate_place)

    @staticmethod
    def _validate_place(value):
        try:
            return value.encode("ascii")
        except UnicodeEncodeError:
            raise ValidationError("contains non-ascii symbols")


# noinspection PyAbstractClass
class DummyValidator(Serializer):
    pass


# noinspection PyAbstractClass
class SearchQueryValidator(Serializer):
    query = CharField(required=False, min_length=2, max_length=100, trim_whitespace=True)
    package_name = CharField(required=False, min_length=1)
    count = IntegerField(required=False, default=4)

    def validate(self, attrs):
        query = attrs.get('query')
        package_name = attrs.get('package_name')

        if not any((query, package_name)):
            raise ValidationError('any (query, package_name) is required or both')

        if package_name:
            attrs.pop('query', None)
            attrs['count'] = 1

        if query and len(query.strip()) < 3:
            raise KnownParseError(
                "validation error: {'query': [u'Ensure this field has at least 3 non-space characters.']}"
            )

        return attrs


# noinspection PyAbstractClass
class ThemesQueryValidator(Serializer):
    screen_height = IntegerField(min_value=240, required=True)
    screen_width = IntegerField(min_value=240, required=True)
    dpi = IntegerField(min_value=120, required=True)


# noinspection PyAbstractClass
class WallpaperQueryValidator(Serializer):
    screen_height = IntegerField(min_value=240, required=True)
    screen_width = IntegerField(min_value=240, required=True)
    dpi = IntegerField(min_value=120, required=True)
    base64_previews = BooleanField(required=False, default=False)


# noinspection PyAbstractClass
class GetWallpaperQueryValidator(Serializer):
    screen_height = IntegerField(min_value=240, required=True)
    id = CharField(required=True)


# noinspection PyAbstractClass
class WallpaperAutochangeQueryValidator(Serializer):
    mode = ChoiceField(required=True, choices=('hourly', 'daily', 'off'))
    screen_height = IntegerField(min_value=240, required=True)
    screen_width = IntegerField(min_value=240, required=True)
    dpi = IntegerField(required=True)


# noinspection PyAbstractClass
class WallpaperActionQueryValidator(Serializer):
    id = CharField(required=False)
    action = ChoiceField(required=False, choices=('collection_pick', 'force_next', 'auto_next'))


# noinspection PyAbstractClass
class WallpapersFeedQueryValidator(WallpaperQueryValidator):
    feed_id = UUIDField(required=False)
    offset = IntegerField(required=False, default=0)
    limit = IntegerField(required=False, default=20)
    color = CharField(required=False)

    @staticmethod
    def validate_color(value):
        try:
            group, color = value.split('/')
        except ValueError:
            raise ValidationError('Color should contain group and color name separated by slash')
        return value

    def validate(self, attrs):
        if ('offset' in attrs) ^ ('limit' in attrs):
            raise ValidationError('Offset and limit should both be either present or absent')
        if 'color' in attrs and 'feed_id' not in attrs:
            raise ValidationError('Request with color should also contain feed_id')
        return attrs


# noinspection PyAbstractClass
class SettingsPromoBlockQueryValidator(Serializer):
    image_width = CharField(required=True)
    image_height = CharField(required=True)


# noinspection PyAbstractClass
class VangaModelQueryValidator(Serializer):
    current_model_name = CharField(required=False)


class TimezoneField(CharField):
    default_error_messages = {
        'invalid': '"{input}" is not a valid timezone.',
    }

    def to_internal_value(self, data):
        try:
            AndroidTimezone(data)
            return super(TimezoneField, self).to_internal_value(data)
        except UnknownTimeZoneError:
            self.fail('invalid', input=data)


# noinspection PyAbstractClass
class LbsInfoValidator(Serializer):
    location = LocationSerializer(required=False, default=None)
    cells = GsmSerializer(many=True, required=False, default=None)
    wifi_networks = WifiSerializer(many=True, required=False, default=None)

    def validate(self, attrs):
        if any((attrs['location'], attrs['cells'], attrs['wifi_networks'])):
            return attrs
        raise ValidationError('Please provide location or cells or wi-fi network info')


# noinspection PyAbstractClass
class ContentBlocksQueryValidator(LbsInfoValidator):
    timezone = TimezoneField(required=False)

    def validate(self, attrs):
        return attrs


# noinspection PyAbstractClass
class LocaleTimezoneValidator(Serializer):
    locale = CharField(required=False)
    timezone = TimezoneField(required=False)


class IconsForAppsValidator(Serializer):
    package_names = CommaSeparatedListField(required=True)
    size = ChoiceField(choices=DPI_NAME_MAP.values(), required=True)
    pack = ChoiceField(choices=Icons.PACKS, default=Icons.DEFAULT_PACK, required=False)
