from mongoengine import Q
from rest_framework.renderers import JSONRenderer
from rest_framework.response import Response

from yaphone.advisor.advisor.views.base import MobileApiView, StatelessView
from yaphone.advisor.launcher.models.app_icons import Icons
from yaphone.advisor.launcher.serializers import IconSerializer
from yaphone.advisor.launcher.validators import IconsForAppsValidator

FIELD_PRIORITY = (
    'package_name',
    'clid1',
    'activity',
    'language',
)


def get_priority(icon):
    return tuple(icon[field] is not None for field in FIELD_PRIORITY)


def split_package_name(package_name_with_activity):
    package_name, _, activity = package_name_with_activity.partition('@')
    return package_name, activity


def load_icons(package_names, clid1, language, pack):
    """
    Forms db query and loads icons for apps.

    :param package_names: package name with activity pairs [com.android.dialer@com.android.dialer.DialerAcivity, ... ]
    :param clid1: client's clid1 value string
    :param language: client's language
    :param pack: icon pack ID
    :return: map of (package_name, activity): Icon if icon for specific activity exists
             or (package_name, None): Icon otherwise
    """
    query = Q()
    for package_name_with_activity in package_names:
        package_name, activity = split_package_name(package_name_with_activity)
        query |= Q(package_name=package_name, activity=None)
        if activity:
            query |= Q(package_name=package_name, activity=activity)
    query &= Q(language=None) | Q(language=language)
    query &= Q(clid1=None) | Q(clid1=clid1)
    query &= Q(pack=pack)
    icons = sorted(Icons.objects(query), key=get_priority)
    return {(icon.package_name, icon.activity): icon for icon in icons}


def select_icon(icons, package_name_with_activity):
    """
    Selects best icon with or without activity for specified package_name, based on get_priority function
    :param icons: load_icons result
    :param package_name_with_activity: package name with activity pairs
                                      [com.android.dialer@com.android.dialer.DialerAcivity, ... ]
    :return: selected Icon model or None if no matching icon found
    """
    package_name, activity = split_package_name(package_name_with_activity)
    icon_with_activity = icons.get((package_name, activity))
    icon_without_activity = icons.get((package_name, None))
    if icon_with_activity and icon_without_activity:
        return max(icon_with_activity, icon_without_activity, key=get_priority)
    else:
        return icon_with_activity or icon_without_activity


# noinspection PyUnresolvedReferences
class IconsForAppsViewMixin(object):
    validator_class = IconsForAppsValidator

    def get_serializer_context(self):
        return {
            'client': self.client,
            'host': self.host,
        }

    # noinspection PyUnusedLocal
    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)

        package_names = validated_data['package_names']
        pack = validated_data['pack']
        size = validated_data['size']
        language = self.client.language
        clid1 = self.client.clids.get('clid1')

        icons = load_icons(package_names, clid1, language, pack)
        package_name_icons_map = self.serialize_icons(icons, package_names, size)
        return Response(package_name_icons_map)

    def serialize_icons(self, icons, package_names, size):
        result = {}
        serializer_context = self.get_serializer_context()
        for package_name_with_activity in package_names:
            icon = select_icon(icons, package_name_with_activity)
            if icon:
                serializer = IconSerializer(icon[size], context=serializer_context)
                result[package_name_with_activity] = serializer.data
        return result


class IconsForAppsView(IconsForAppsViewMixin, MobileApiView):
    # use default renderer class because ujson one doesn't support OrderedDicts
    # OrderedDicts are required because of old version client's bug: hash field should be always after icon_url
    renderer_classes = (JSONRenderer,)


class IconsForAppsViewV2(IconsForAppsViewMixin, StatelessView):
    pass
