import logging
from rest_framework.exceptions import NotFound
from rest_framework.response import Response

from yaphone.advisor.advisor.app_info_loader import AppInfoLoaderException
from yaphone.advisor.advisor.jafar import RecommendationContext, AppsBlock, JafarSearch
from yaphone.advisor.advisor.models.app import App
from yaphone.advisor.advisor.modifiers.base import AppsInfoFiller, ImpressionIDFiller
from yaphone.advisor.advisor.serializers.apps import AppSerializer
from yaphone.advisor.advisor.views.base import MobileApiView
from yaphone.advisor.launcher.validators import SearchQueryValidator

logger = logging.getLogger(__name__)


class SearchAppsView(MobileApiView):
    # noinspection PyAbstractClass
    validator_class = SearchQueryValidator
    pipeline = [AppsInfoFiller, ImpressionIDFiller]

    def init_modifiers(self):
        self.modifiers = [modifier_class(context=self.context, client=self.client) for modifier_class in self.pipeline]

    def apply_modifiers(self, recommendations):
        for modifier in self.modifiers:
            modifier.apply_multi(recommendations)

    def get_recommender(self, cache_key=None):
        return JafarSearch(
            client=self.client,
            context=self.context,
            cache_key=cache_key,
        )

    # noinspection PyUnusedLocal
    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)

        count = validated_data['count']
        package_name = validated_data.get('package_name')
        query = validated_data.get('query')

        self.context = RecommendationContext(host=self.host,
                                             group_size=count,
                                             experiment='search')
        self.init_modifiers()
        if query:
            apps_blocks = [AppsBlock(apps=[])]
        elif package_name:
            apps_blocks = [AppsBlock(apps=[App(package_name)])]
            try:
                self.apply_modifiers(apps_blocks)
            except AppInfoLoaderException:
                raise NotFound(detail='No app with this package_name')
        else:
            raise Exception('query or package_name should be defined')
        apps = apps_blocks[0].apps
        if apps:
            serializer = AppSerializer(apps, many=True)
            return Response(serializer.data)
        raise NotFound(detail='No applications found')
