from collections import namedtuple

import logging
import requests
import requests.exceptions
from PIL import Image
from cStringIO import StringIO
from lxml import etree
from rest_framework.response import Response
from rest_framework.serializers import Serializer, CharField, ListField
from urlparse import urljoin

from yaphone.advisor.advisor.views.base import BaseAPIView, StatelessView
from yaphone.advisor.common.cache import shared_memoize
from yaphone.advisor.common.tools import get_http_header, remove_bom

logger = logging.getLogger(__name__)

logging.getLogger("requests").setLevel(logging.WARNING)
logging.getLogger("PIL.PngImagePlugin").setLevel(logging.WARNING)
logging.getLogger("PIL.ImageFile").setLevel(logging.FATAL)

TIMEOUTS_TUPLE = namedtuple('Timeouts', ['connect', 'read'])

timeout_values = TIMEOUTS_TUPLE(connect=0.5, read=1.0)


# for readability only
def get_icons_xpath():
    xpath_conditions = [
        '@rel="icon"',
        '@rel="icon short"',
        '@rel="shortcut icon"',
        'contains(@rel, "apple-touch-icon")'
    ]

    xpath_exp_template = '//link[{condition}]/@href'
    return ' | '.join([xpath_exp_template.format(condition=condition) for condition in xpath_conditions])


extract_icons_xpath_expression = get_icons_xpath()

pretty_icons_request_headers = {
    'user-agent': 'Mozilla/5.0 (iPhone; CPU iPhone OS 9_3 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko)'
                  ' Version/9.0 Mobile/13E188a Safari/601.1'
}

extra_relative_icon_urls = [
    '/apple-touch-icon.png',
    '/apple-touch-icon-precomposed.png',
    '/favicon.ico',
    '/favicon.png'
]


def get_extra_icon_urls(session, base_url):
    for extra_url in extra_relative_icon_urls:
        icon_url = urljoin(base_url, extra_url)
        response = session.head(icon_url, allow_redirects=True, timeout=timeout_values)
        if response.ok and 'image' in response.headers.get('Content-Type', ''):
            yield icon_url


def image_by_url(session, url):
    response = session.get(url, timeout=timeout_values)
    response.raise_for_status()
    image = Image.open(StringIO(response.content))
    return image


def get_icon_info(session, url):
    # noinspection PyShadowingNames
    @shared_memoize(timeout=10 * 24 * 3600)
    def get_image_size(url):
        return image_by_url(session, url).size

    w, h = get_image_size(url)
    return {'url': url, 'width': w, 'height': h}


def process_icon_urls(session, icon_urls):
    icons = []
    for icon_url in icon_urls:
        try:
            icons.append(get_icon_info(session, icon_url))
        except requests.HTTPError:
            logger.warning('server return not 200;ok for image url=%s', icon_url, exc_info=1)
        except IOError:
            logger.warning('on this url=%s is no picture', icon_url, exc_info=1)
    return icons


@shared_memoize(timeout=60 * 60 * 24)
def process_domain_url(url):
    session = requests.Session()
    session.max_redirects = 3
    response = session.get(url, headers=pretty_icons_request_headers, timeout=timeout_values)
    response.raise_for_status()
    data = remove_bom(response.content)
    root = etree.HTML(data)
    if not root:
        logging.warning('cannot parse html page: data=%r len=%d, url=%s',
                        response.content[:20], len(response.content), url)
        return []

    rel_icon_urls = root.xpath(extract_icons_xpath_expression)
    icon_urls = [urljoin(response.url, icon_url) for icon_url in rel_icon_urls]
    icon_urls.extend(get_extra_icon_urls(session, response.url))
    return process_icon_urls(session, frozenset(icon_urls))


def get_icons_info_from_uri(url):
    icons = []
    # noinspection PyBroadException
    try:
        icons = process_domain_url(url)
    except requests.HTTPError:
        logger.warning('server return not 200;ok for requested url=%s', url, exc_info=1)
    except etree.LxmlError:
        logger.warning('some problem with lxml', exc_info=1)
    except Exception:
        logger.warning('some problem with connect to %s', url, exc_info=1)
    return {'url': url, 'icons': icons}


# noinspection PyAbstractClass
class IconsForUrlsQueryParamsValidator(Serializer):
    url = ListField(child=CharField(), required=True, allow_empty=False)


def get_icons(urls):
    result = []
    for url in urls:
        info = get_icons_info_from_uri(url)
        if info:
            result.append(info)
    return Response(result)


class IconsForUrlsView(BaseAPIView):
    validator_class = IconsForUrlsQueryParamsValidator

    # noinspection PyUnusedLocal,PyUnusedLocal
    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)
        return get_icons(validated_data['url'])

    @staticmethod
    def get_uuid(request):
        return get_http_header(request, 'X_YAUUID')

    def load_database_client(self, request):
        pass


class IconsForUrlsViewV2(StatelessView):
    validator_class = IconsForUrlsQueryParamsValidator

    # noinspection PyUnusedLocal,PyUnusedLocal
    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)
        return get_icons(validated_data['url'])
