# -*- coding: utf-8 -*-
import hashlib
import json
import logging
from rest_framework.exceptions import NotFound
from rest_framework.fields import CharField, IntegerField, BooleanField, SerializerMethodField
from rest_framework.response import Response
from rest_framework.serializers import Serializer
from rest_framework.views import APIView
from ylog.context import log_context

from yaphone.advisor.advisor.views.base import MobileApiView, StatelessView
from yaphone.advisor.common.localization_helpers import translate, LocalizationItem
from yaphone.advisor.launcher.configs import get_config
from yaphone.advisor.launcher.models import localization_db

logger = logging.getLogger(__name__)


class ConfigView(MobileApiView):
    # noinspection PyUnusedLocal
    def get(self, request, *args, **kwargs):
        return Response(
            get_config(self.client)
        )


# noinspection PyAbstractClass
class LocalizationItemSerializer(Serializer):
    name = CharField(required=True)
    value = CharField(required=True, source='serialized_value')
    report_to_appmetrika = BooleanField(required=True)
    report_to_appmetrika_environment = BooleanField(required=True)
    expire_time = IntegerField(required=False, allow_null=False)


def preprocess(client, items):
    all_items = {}
    for item in items:
        if item.value is not None:
            # Client does not support "null" values, ignore it
            if isinstance(item.value, dict) and 'translate_key' in item.value:
                # translate items that have 'translate_key'
                # see https://st.yandex-team.ru/ADVISOR-1090 to know why
                key = item.value['translate_key']
                item._value = key
                translated_value = translate(client, key)
                if translated_value:
                    yield {
                        'name': '%s.translated' % item.name,
                        'serialized_value': translated_value,
                        'report_to_appmetrika': False,
                        'report_to_appmetrika_environment': False,
                    }
                else:
                    logger.error('Could not find translation for key "%s"' % key)
            else:
                # collect all items to be hashed later
                all_items[item.name] = item.value
            # Do not send any experiment to appmetrika_environment
            # https://st.yandex-team.ru/ADVISOR-2140
            item.report_to_appmetrika_environment = False
            yield item
    # Calculate hash from all experiments to store only it in appmetrika_environment
    # https://st.yandex-team.ru/ADVISOR-2140
    sorted_items = ((key, all_items[key]) for key in sorted(all_items.keys()))
    hashed_string = ','.join('%s:%s' % item for item in sorted_items)
    experiments_md5 = hashlib.md5(hashed_string).hexdigest()
    yield {
        'name': 'hash',
        'serialized_value': experiments_md5,
        'report_to_appmetrika': False,
        'report_to_appmetrika_environment': True,
    }


def get_experiments_response(client):
    experiments = sorted(client.user_specific_config.get_all_enabled_items(), key=lambda local_obj: local_obj.name)
    items = preprocess(client, experiments)
    serializer = LocalizationItemSerializer(items, many=True)
    with log_context(uuid=client.uuid.hex,
                     device_id=client.profile.device_id.hex if client.profile else '',
                     method='experiments',
                     device_type=client.user_agent.device_type):
        logger.info(
            json.dumps(serializer.data, ensure_ascii=False).encode('utf8')
        )

    return Response(serializer.data)


class ExperimentsView(MobileApiView):
    # noinspection PyUnusedLocal
    def get(self, request, *args, **kwargs):
        response = get_experiments_response(self.client)
        return response


class ExperimentsViewV2(StatelessView):
    save_client = True

    # noinspection PyUnusedLocal
    def get(self, request, *args, **kwargs):
        response = get_experiments_response(self.client)
        return response


class DummyLocalizationItem(object):
    def __init__(self, value):
        self.value = value


class Experiment(Serializer):
    key = CharField(source='_id')
    values = SerializerMethodField()

    @staticmethod
    def get_values(doc):
        return [
            {'value': LocalizationItem(DummyLocalizationItem(item['value'])).value}
            for item in doc.get('values', [])
            if 'value' in item
        ]


class AllExperimentsView(APIView):
    """ Shows all experiments and their possible values """
    PROJECT_CHOICES = ('launcher', 'updater')

    # noinspection PyUnusedLocal
    def get(self, request, project_name, *args, **kwargs):
        if project_name not in self.PROJECT_CHOICES:
            raise NotFound(detail='%s not found' % project_name)
        collection_name = 'localization.%s' % project_name
        serializer = Experiment(localization_db[collection_name].find(), many=True)
        return Response(serializer.data)
