import logging
import requests
from django.conf import settings
from django.template import Context, Template
from rest_framework.exceptions import NotFound
from rest_framework.response import Response

from yaphone.advisor.advisor.modifiers.explanation import RegionExplanation, ExplanationError
from yaphone.advisor.advisor.views.base import StatelessView
from yaphone.advisor.common.localization_helpers import translate
from yaphone.advisor.common.template import InvalidTemplateVarException
from yaphone.advisor.common.tools import make_requests_session
from yaphone.advisor.launcher.validators import ContentBlocksQueryValidator
from yaphone.advisor.weather.views import get_location_from_lbs_info, get_weather
from yaphone.utils import geo

logger = logging.getLogger(__name__)


class NewsException(NotFound):
    pass


class ContentBlocksView(StatelessView):
    validator_class = ContentBlocksQueryValidator
    http = make_requests_session(pool_maxsize=100, max_retries=2)

    def get_news(self):
        try:
            params = {'lang': self.locale}
            if self.location:
                params['lat'] = self.location['lat']
                params['lon'] = self.location['lon']

            news_response = self.http.get(settings.PORTAL_NEWS_URL, params=params,
                                          headers={'X-Forwarded-For': self.ip})
            news_response.raise_for_status()
        except (requests.RequestException, requests.exceptions.BaseHTTPError) as e:
            logger.warn('Bad news request, %s', e.message)
            raise NewsException(e)

        try:
            result = news_response.json()
            if 'block' in result:
                return result
            raise ValueError
        except ValueError:
            logger.warn('Bad news response, %s %s' % (news_response.status_code, news_response.content))
            raise NewsException('Bad news response')

    def get_weather(self):
        if not self.location:
            return {}
        try:
            weather = get_weather(locale=self.locale,
                                  tz_dst=self.timezone,
                                  **self.location)
            return weather
        except NotFound:
            pass  # get_weather logs errors itself
        return {}

    def get_header(self, region_id=None):
        if not region_id:
            region_id = geo.geobase_lookuper.get_region_id_by_location(lat=self.location['lat'],
                                                                                          lon=self.location['lon'])

        template = translate(self.client, settings.CONTENT_BLOCK_HEADER_TEMPLATE_KEY)
        if template is None:
            raise ExplanationError('Translation for key "%s" not found' % settings.CONTENT_BLOCK_HEADER_TEMPLATE_KEY)

        context = Context(autoescape=False)
        context['region'] = RegionExplanation(
            language=template.language,
            region_id=region_id
        )

        try:
            template = Template(template)
            return template.render(context)
        except InvalidTemplateVarException:
            raise ExplanationError('Explanation template error for template "%s" ' % template)

    def compile_response(self, news, weather, header):
        news['block'].append({'ttv': settings.DEFAULT_WEATHER_TTV, 'ttl': settings.DEFAULT_WEATHER_TTL,
                              'id': 'weather', 'data': weather})
        news['title'] = header
        return news

    def _prepare_request_params(self, request):
        # validate POST parameters
        lbs_info = self.get_validated_data(request.data)
        self.timezone = lbs_info.get('timezone')
        self.locale = str(self.client.locale)

        self.location = get_location_from_lbs_info(lbs_info, uuid=self.uuid, ip=self.ip)

    def post(self, request, *args, **kwargs):
        logger.info('-1 screen content blocks request: uuid=%s\trequest=%s', self.uuid.hex, request.body)

        self._prepare_request_params(request)

        news = self.get_news()
        weather = self.get_weather()
        header = self.get_header(region_id=news.get('geo'))

        return Response(self.compile_response(news, weather, header))
