import logging
from rest_framework.exceptions import NotFound
from rest_framework.response import Response
from yaphone.advisor.common import localization_keys

from yaphone.advisor.advisor.views.base import MobileApiView, StatelessView
from yaphone.advisor.common.localization_helpers import get_config_value
from yaphone.advisor.launcher.models.themes import Theme
from yaphone.advisor.launcher.serializers import ThemeSerializer
from yaphone.advisor.launcher.validators import ThemesQueryValidator

logger = logging.getLogger(__name__)


def get_theme_ids(client):
    theme_config = get_config_value(client, localization_keys.THEMES)
    if theme_config is None:
        raise NotFound("Themes localization data not found")
    return theme_config['themes']


class ThemesMixin(object):

    def get_serializer_context(self, request):
        raise NotImplementedError("Please Implement this method")

    def get(self, request, *args, **kwargs):
        theme_ids = get_theme_ids(self.client)
        context = self.get_serializer_context(request)
        themes = Theme.objects(theme_id__in=theme_ids, device_type=context['device_type'])
        serializer = ThemeSerializer(themes, many=True, context=context)

        return Response(serializer.data)


class ThemesView(ThemesMixin, MobileApiView):

    def get_serializer_context(self, request):
        display_metrics = self.client.profile.android_info.display_metrics

        return {
            'client': self.client,
            'host': self.host,
            'language': self.client.language,
            'height': display_metrics.height_pixels,
            'dpi': display_metrics.density_dpi,
            'device_type': self.client.user_agent.device_type
        }


class ThemesViewV2(ThemesMixin, StatelessView):
    validator_class = ThemesQueryValidator

    def get_serializer_context(self, request):
        validated_data = self.get_validated_data(request.query_params)

        return {
            'client': self.client,
            'host': self.host,
            'language': self.client.language,
            'height': validated_data['screen_height'],
            'dpi': validated_data['dpi'],
            'device_type': self.client.user_agent.device_type
        }
