import logging
import random
from django.conf import settings
from django.http import HttpResponseRedirect
from rest_framework.exceptions import NotFound
from rest_framework.renderers import JSONRenderer
from rest_framework.response import Response
from rest_framework.views import APIView
from yaphone.advisor.common import localization_keys

from yaphone.advisor.advisor.views.base import MobileApiView, BaseAPIView, StatelessView, RequestValidatorMixin
from yaphone.advisor.common.localization_helpers import get_config_value
from yaphone.advisor.common.resizer import make_scaled_image_url
from yaphone.advisor.common.tools import make_mds_url, make_s3_mds_url
from yaphone.advisor.launcher.models.wallpapers import WallpaperCategories, WallpaperFeedback, WallpaperStatus
from yaphone.advisor.launcher.serializers import (PreparedCategorySerializer, CompositeCategorySerializer,
                                                  WallpapersAutochangeSerializer, WallpapersFeedSerializer)
from yaphone.advisor.launcher.validators import (WallpaperQueryValidator, GetWallpaperQueryValidator,
                                                 WallpaperAutochangeQueryValidator, WallpaperActionQueryValidator,
                                                 WallpapersFeedQueryValidator, DummyValidator)
from yaphone.advisor.launcher.wallpapers import DEFAULT_PREVIEW_IMAGE_SIZES, PREVIEW_LARGE, PREVIEW_XLARGE
from yaphone.advisor.launcher.wallpapers import wallpapers_issuers, WallpapersFeedBuilder, normalize_image_key

logger = logging.getLogger(__name__)


class WallpapersMixin(object):
    renderer_classes = (JSONRenderer,)
    serializer_class = PreparedCategorySerializer
    validator_class = DummyValidator
    preview_image_sizes = DEFAULT_PREVIEW_IMAGE_SIZES

    def get_serializer_context(self, request, validated_data):
        return {
            'client': self.client,
            'host': self.host,
            'language': self.client.language,
            'height': validated_data['screen_height'],
            'width': validated_data['screen_width'],
            'dpi': validated_data['dpi'],
            'base64_previews': False,
            'preview_image_sizes': self.preview_image_sizes,
        }

    def filter_category(self, category):
        """Checks whether a provided category is needed or should not be used"""
        return category is not None

    def get_categories(self):
        wallpapers_config = get_config_value(self.client, localization_keys.WALLPAPERS)
        if not wallpapers_config:
            detail = 'Wallpapers localization data not found'
            logger.error(detail)
            raise NotFound(detail=detail)
        category_names = wallpapers_config['categories']
        categories = WallpaperCategories.objects(name__in=category_names)
        categories = {cat.name: cat for cat in categories if self.filter_category(cat)}
        wallpapers_shuffle = get_config_value(self.client, localization_keys.WALLPAPERS_SHUFFLE_EXPERIMENT)
        if wallpapers_shuffle == 'shuffle':
            local_random = random.Random(self.client.id)
            local_random.shuffle(category_names)
            for name in categories:
                local_random.shuffle(categories[name].images)
        for name in category_names:
            if name in categories:
                yield categories[name]
            else:
                logger.warning('Missing wallpaper category "%s"', name)

    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)
        context = self.get_serializer_context(request, validated_data)
        categories = self.get_categories()
        serializer = self.serializer_class(categories, many=True, context=context)
        return Response(serializer.data)


class WallpapersView(WallpapersMixin, MobileApiView):
    # noinspection PyUnusedLocal
    def get_serializer_context(self, request, validated_data):
        display_metrics = self.client.profile.android_info.display_metrics

        return {
            'client': self.client,
            'host': self.host,
            'language': self.client.profile.android_info.user_settings.locale.language,
            'height': display_metrics.height_pixels,
            'width': display_metrics.width_pixels,
            'dpi': display_metrics.density_dpi,
            'base64_previews': False,
            'preview_image_sizes': DEFAULT_PREVIEW_IMAGE_SIZES,
        }


class WallpapersViewV2(WallpapersMixin, StatelessView):
    validator_class = WallpaperQueryValidator

    def filter_category(self, category):
        return bool(category['covers']) or bool(category['composite_cover']['generated_covers'])

    def get_serializer_context(self, request, validated_data):
        context = super(WallpapersViewV2, self).get_serializer_context(request, validated_data)
        context.update({
            'base64_previews': validated_data['base64_previews'],
        })
        return context


class WallpapersViewV3(WallpapersViewV2):
    serializer_class = CompositeCategorySerializer

    def filter_category(self, category):
        return bool(category['composite_cover'])


class GetWallpaperView(APIView, RequestValidatorMixin):
    validator_class = GetWallpaperQueryValidator

    @staticmethod
    def get_wallpaper_redirect(source_url, height, host):
        if source_url.startswith('wallpapers/images'):
            storage_url = make_s3_mds_url(source_url)
        else:
            storage_url = make_mds_url(source_url)
        scaled_url = make_scaled_image_url(url=storage_url, height=height, host=host, path_prefix='/resizer')
        return HttpResponseRedirect(scaled_url)

    # noinspection PyUnusedLocal
    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)

        source_url = validated_data['id']
        height = validated_data['screen_height']
        host = self.request.get_host()

        return self.get_wallpaper_redirect(source_url, height, host)


class WallpapersFeedView(WallpapersMixin, StatelessView):
    renderer_classes = (JSONRenderer,)
    validator_class = WallpapersFeedQueryValidator
    serializer_class = WallpapersFeedSerializer
    preview_image_sizes = (PREVIEW_LARGE, PREVIEW_XLARGE)

    def filter_category(self, category):
        return category['use_in_feed']

    def get_serializer_context(self, request, validated_data):
        context = super(WallpapersFeedView, self).get_serializer_context(request, validated_data)
        context.update({
            'color': validated_data.get('color'),
            'offset': validated_data.get('offset'),
            'limit': validated_data.get('limit'),
            'feed_id': validated_data.get('feed_id'),
            'base64_previews': validated_data['base64_previews'],
        })
        return context

    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)
        context = self.get_serializer_context(request, validated_data)
        color_group = get_config_value(self.client,
                                       localization_keys.WALLPAPERS_FEED_COLOR_GROUP,
                                       log_missing=False)
        feed_builder = WallpapersFeedBuilder(request=request, uuid_=self.client.uuid,
                                             categories=self.get_categories(), color_group=color_group)
        feed = feed_builder.build_feed(feed_id=context['feed_id'], color=context['color'],
                                       offset=context['offset'], limit=context['limit'])
        serializer = self.serializer_class(feed, context=context)
        return Response(serializer.data)


class WallpapersAutochangeView(WallpapersMixin, StatelessView):
    validator_class = WallpaperAutochangeQueryValidator
    serializer_class = WallpapersAutochangeSerializer

    def filter_category(self, category):
        return category['use_in_autochange']

    def get_blacklist(self):
        query = WallpaperFeedback.objects(uuid=self.uuid).order_by('-created_at').limit(
            settings.AUTOCHANGE_BLACKLIST_SIZE)
        return {action.wallpaper for action in query}

    def get(self, request, *args, **kwargs):
        validated_data = self.get_validated_data(request.query_params)
        context = self.get_serializer_context(request, validated_data)
        strategy = get_config_value(self.client,
                                    localization_keys.WALLPAPERS_AUTOCHANGE_STRATEGY,
                                    log_missing=False)
        try:
            issuer = wallpapers_issuers[strategy]()
        except KeyError:
            logger.error('Bad wallpapers autochange strategy: %s. Falling back to random', strategy)
            issuer = wallpapers_issuers['random']()
        wallpapers = issuer.get_wallpapers(
            self.get_categories(), self.get_blacklist()
        )[:settings.AUTOCHANGE_RESPONSE_SIZE]
        serializer = self.serializer_class(wallpapers, context=context)
        return Response(serializer.data)


class WallpapersActionView(APIView, RequestValidatorMixin):
    validator_class = WallpaperActionQueryValidator

    def post(self, request, *args, **kwargs):
        data = self.get_validated_data(request.data)
        if data.get('id') is None or data.get('action') is None:
            # Just answer 200 and do nothing, see ADVISOR-2534 for details
            return Response('OK')
        wallpaper = normalize_image_key(data['id'])
        action = data['action']
        WallpaperFeedback(uuid=BaseAPIView.get_uuid(request),
                          action=action,
                          wallpaper=wallpaper).save()
        WallpaperStatus.objects(
            wallpaper=wallpaper
        ).update_one(
            upsert=True,
            **{'inc__{}_count'.format(action): 1}
        )
        WallpaperStatus.objects.get(wallpaper=wallpaper).save()
        return Response('OK')
